/**********************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.provisional.control.provider;

import java.util.Iterator;

import org.eclipse.hyades.trace.internal.ui.PDProjectExplorer;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.actions.DeleteAction;
import org.eclipse.hyades.trace.ui.internal.actions.OpenWizardAction;
import org.eclipse.hyades.trace.ui.internal.actions.SaveAction;
import org.eclipse.hyades.trace.ui.internal.core.TraceUIImages;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.trace.ui.internal.wizard.OpenProcessWizard;
import org.eclipse.hyades.trace.ui.internal.wizard.OpenProfAgentWizard;
import org.eclipse.hyades.ui.extension.INavigatorContribution;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.trace.ui.internal.control.provider.ControlProviderManager;
import org.eclipse.tptp.trace.ui.internal.control.provider.TraceControlItemManager;
import org.eclipse.tptp.trace.ui.internal.control.provider.application.ControlMessages;



/**
 * The control provider for resource items that appear in the profiling
 * monitor view.  Examples of resource items are processes and agent.  Contributors can 
 * extends this class to augment the control items available for a specific resource
 * item.
 * 
 * @see org.eclipse.tptp.trace.ui.provisional.control.provider.IControlProvider
 * @see org.eclipse.tptp.trace.ui.provisional.control.provider.LogicalItemControlProvider
 * 
 * @author Ali Mehregani
 */
public class ResourceItemControlProvider extends LogicalItemControlProvider 
{
	/** The id of the contributed items */
	private static final String REFRESH_ITEM = "org.eclipse.hyades.trace.ui.internal.monitorsview.popupMenu.refreshViews";
	private static final String REFRESH_OPTION = "org.eclipse.hyades.trace.ui.internal.monitorsview.popupMenu.refreshOption";
	
	/** The delete item */
	private IControlItem delete;
	
	/** The save item */
	private IControlItem save;
	
	/** The refresh view item */
	private IControlItem refresh;
	
	/** The refresh option item */
	private IControlItem refreshOption;

	/** The properties item */
	private IControlItem properties;

	/** The entity that this provider is affiliated to */
	private String entity;
	
	public void initialize (String entity)
	{
		super.initialize(entity);
		this.entity = entity;
	}
	
	/**
	 * @see org.eclipse.tptp.trace.ui.provisional.control.provider.AbstractLogicalItem.initializeControlItems()
	 */
    protected void initializeControlItems()
	{
    	super.initializeControlItems();
    	addControlItem (createDeleteControlItem());
    	addControlItem (createSaveControlItem());
    	addControlItem (createRefreshViewsControlItem());
    	addControlItem (createRefreshOptionControlItem());
    	addControlItem (createPropertiesControlItem());  
	}
    
    protected IControlItem createDeleteControlItem()
    {
    	if (delete != null && !isContextDifferent)
    		return delete;
    	
    	    	
    	delete = new ControlItemAdapter(INavigatorContribution.GROUP_DELETE, new DeleteAction(TraceControlItemManager.getNavigator()));
    	return delete;
    }
    
    
    protected IControlItem createSaveControlItem()
    {
    	if (save != null && !isContextDifferent)
    		return save;
    	
    	    	
    	save = new ControlItemAdapter(INavigatorContribution.GROUP_SAVE, new SaveAction(TraceControlItemManager.getNavigator()));
    	return save;
    }
    
    
    protected IControlItem createRefreshViewsControlItem()
    {
    	if (refresh != null)
    		return refresh;
    	
    	class RefreshView extends ControlItem
    	{    		
    		public RefreshView()
    		{
    			super (REFRESH_ITEM, CommonUITraceConstants.PROFILE_REFRESH_GROUP, ControlMessages.CONTROL_ITEM_REFRESH_VIEWS, TraceUIImages.INSTANCE.getImageDescriptor("c", TraceUIImages.IMG_REFRESH_VIEWS));
    		}
    		
    		public void run()
    		{
    			if (input == null)
    				return;
    			
    			for (Iterator sources = input.iterator(); sources.hasNext();)
    			{					
    				ProfileEvent event = UIPlugin.getDefault().getRefreshViewEvent(sources.next());
    				UIPlugin.getDefault().notifyProfileEventListener(event);
    			}
    		}
    	}
    	refresh = new RefreshView();
    	return refresh;
    }
    
    /**
     * Navid Mehregani - bugzilla 153453
     * Used to create a toolbar button to allow the user to easily change
     * the refresh options
     * @return A control item used to change the refresh options
     */
    protected IControlItem createRefreshOptionControlItem()
    {
    	if (refreshOption != null)
    		return refreshOption;
    	
    	class RefreshOption extends ControlItem
    	{    		
    		public RefreshOption()
    		{
    			super (REFRESH_OPTION, CommonUITraceConstants.PROFILE_REFRESH_GROUP, ControlMessages.CONTROL_ITEM_REFRESH_OPTION, TraceUIImages.INSTANCE.getImageDescriptor("c", TraceUIImages.IMG_REFRESH_OPTIONS));
    		}
    		
    		/* This method is invoked when ever user clicks the 'Change Refresh Options' button.
    		 * It simply invokes the 'handleProfileEvent' method of PDProjectExplorer */
    		public void run()
    		{
    			if (input == null)
    				return;
    			
    			for (Iterator sources = input.iterator(); sources.hasNext();)
    			{					
    				ProfileEvent event = UIPlugin.getDefault().getRefreshOptionEvent(sources.next());
    				UIPlugin.getDefault().notifyProfileEventListener(event);
    			}
    		}
    	}
    	refreshOption = new RefreshOption();
    	return refreshOption;
    }
    
    
    protected IControlItem createPropertiesControlItem()
    {
    	if (properties != null && !isContextDifferent)
    		return properties;
    		
    	if (entity.equals(ControlProviderManager.PROCESS_ENTITY))
    		properties = new ControlItemAdapter(INavigatorContribution.GROUP_PROPERTIES, 
    				new OpenWizardAction((PDProjectExplorer)TraceControlItemManager.getNavigator(),
    					OpenProcessWizard.class,
    					UIPlugin.getResourceString(TraceMessages.PROP)));
    	else if (entity.equals(ControlProviderManager.AGENT_ENTITY))
    		properties = new ControlItemAdapter(INavigatorContribution.GROUP_PROPERTIES, 
    				new OpenWizardAction((PDProjectExplorer)TraceControlItemManager.getNavigator(),
						OpenProfAgentWizard.class,
						UIPlugin.getResourceString(TraceMessages.PROP)));
    		    	
    	return properties;
    }

	public String getEntity() 
	{
		return entity;
	}
	
}
