/**********************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.tptp.trace.ui.provisional.control.provider;

import java.util.Iterator;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.actions.AttachAction;
import org.eclipse.hyades.trace.ui.internal.actions.DetachAction;
import org.eclipse.hyades.trace.ui.internal.actions.StartTraceAction;
import org.eclipse.hyades.trace.ui.internal.actions.StopTraceAction;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IWorkbenchPart;


/**
 * A standard implementation of <code> IAgentStateModifier </code> that
 * looks up the execution agent to change its state.
 * 
 * @author Ali Mehregani
 */
public class StandardAgentStateModifier implements IAgentStateModifier
{
	private static final byte ATTACH_ACTION = 0x00;
	private static final byte DETACH_ACTION = 0x01;
	private static final byte PAUSE_ACTION = 0x02;
	private static final byte RESUME_ACTION = 0x03;
	
	/** The detach action */
	private ExtendedDetachAction detachAction;	
	
	/** The attach action */
	private ExtendedAttachAction attachAction;
	
	/** The start monitoring action */
	private StartTraceAction startAction;
	
	/** The pause action */
	private StopTraceAction stopAction;

	/** The input */
	private StructuredSelection input;
	
	
	public StandardAgentStateModifier()
	{
		detachAction = new ExtendedDetachAction();
		attachAction = new ExtendedAttachAction();
		startAction = new StartTraceAction();
		stopAction = new StopTraceAction();
	}
	
	
	/**
	 * 
	 */
	public void detach() throws CoreException
	{
		performProfileAction (DETACH_ACTION, true);
	}

	public void attach() throws CoreException
	{
		performProfileAction (ATTACH_ACTION, true);	
	}

	public void startMonitoring() throws CoreException
	{
		performProfileAction (RESUME_ACTION, true);		
	}

	public void pauseMonitoring() throws CoreException
	{
		performProfileAction (PAUSE_ACTION, true);		
	}

	public boolean canAttach()
	{
		try
		{
			return performProfileAction (ATTACH_ACTION, false);
		} catch (CoreException e)
		{
			return false;
		}
	}

	public boolean canDetach()
	{
		try
		{
			return performProfileAction (DETACH_ACTION, false);
		} catch (CoreException e)
		{
			return false;
		}
	}

	public boolean canPause()
	{
		try
		{
			return performProfileAction (PAUSE_ACTION, false);
		} catch (CoreException e)
		{
			return false;
		}	
	}

	public boolean canResume()
	{
		try
		{
			return performProfileAction (RESUME_ACTION, false);
		} catch (CoreException e)
		{
			return false;
		}	
	}

	public void setInput(StructuredSelection input)
	{
		this.input = input;
	}

	
	private boolean performProfileAction (byte actionIndicator, boolean run) throws CoreException
	{		
		boolean isEnabled = input.size() > 0;
		try
		{			
			for (Iterator agents = input.iterator(); isEnabled && agents.hasNext();)
			{
				Object currentAgent = agents.next();
				switch (actionIndicator)
				{
					case ATTACH_ACTION:
						if (run)
						{
							if (attachAction.getView() == null)
							{
								IWorkbenchPart workbenchPart = UIPlugin.getActiveWorkbenchWindow().getActivePage().getActivePart();
								if (workbenchPart instanceof IViewPart)
									attachAction.setView((IViewPart)workbenchPart);
							}
							attachAction.doAction(currentAgent);
						}							
						else
						{
							isEnabled = isEnabled && attachAction.isEnabledFor(currentAgent);
						}
						
						break;
					case DETACH_ACTION:
						if (run)
						{
							if (detachAction.getView() == null)
							{
								IWorkbenchPart workbenchPart = UIPlugin.getActiveWorkbenchWindow().getActivePage().getActivePart();
								if (workbenchPart instanceof IViewPart)
									detachAction.setView((IViewPart)workbenchPart);
							}
							detachAction.doAction(currentAgent);
						}
						else
						{
							isEnabled = isEnabled && detachAction.isEnabledFor(currentAgent);
						}
						break;
					case PAUSE_ACTION:
						if (run)
							stopAction.doAction(currentAgent);
						else
							isEnabled = isEnabled && stopAction.isEnabledFor(currentAgent);
						break;
					case RESUME_ACTION:
						if (run)
							startAction.doAction(currentAgent);
						else
							isEnabled = isEnabled && startAction.isEnabledFor(currentAgent);
						break;
				}
			}
		
		} 
		catch (Exception e)
		{
			throw LauncherUtility.createCoreException(IStatus.ERROR, e.getMessage(), e);
		}
			
		return isEnabled;
	}
	
	
	/**
	 * The detach action with a few extended operation over {@link DetachAction}
	 * 
	 * @author Ali Mehregani
	 */
	private static class ExtendedDetachAction extends DetachAction
	{
		/**
		 * Expose the view part associated with this action
		 * 
		 * @return The view part
		 */
		public IViewPart getView()
		{
			return super.getView();
		}
		
		/**
		 * Expost the setter method for the view part associated with 
		 * this action
		 * 
		 * param viewPart The view part
		 */
		public void setView(IViewPart viewPart)
		{
			super.setView(viewPart);
		}
	}
	
	
	
	/**
	 * The attach action with a few extended operation over {@link AttachAction}
	 * 
	 * @author Ali Mehregani
	 */
	private static class ExtendedAttachAction extends AttachAction
	{
		/**
		 * Expose the view part associated with this action
		 * 
		 * @return The view part
		 */
		public IViewPart getView()
		{
			return super.getView();
		}
		
		/**
		 * Expost the setter method for the view part associated with 
		 * this action
		 * 
		 * param viewPart The view part
		 */
		public void setView(IViewPart viewPart)
		{
			super.setView(viewPart);
		}
	}
}
