/**********************************************************************
 * Copyright (c) 2006, 2010 IBM Corporation, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.provisional.launcher;

import java.io.File;
import java.util.ArrayList;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.ILaunchConfigurationWorkingCopy;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.launcher.HyadesTraceUIExtensionSupportUtil;
import org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants;
import org.eclipse.hyades.trace.ui.internal.util.FilterTableElement;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.hyades.trace.ui.internal.util.WorkspaceFiltersExtractor;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.launching.JavaRuntime;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.core.CommonDelegateHelper;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;
import org.eclipse.ui.IWorkingSet;

/**
 * An abstract delegator class that provides a set of common functionality between 
 * the delegates.
 * 
 * @author Ali Mehregani
 */
public abstract class AbstractProcessLauncher
implements IDataCollectorProcessLauncher
{	
	/* The execution node (may or may not be null depending on whether the primary
	 * launcher has called createNode()) */
	protected Node node;
	
	/* The associated configuration */
	protected ILaunchConfiguration conf;
	
	/* The host name associated with the launch */
	private String hostName; 

	/* The port that the launch is associated with */
	private int port;

	/* The location that the launch should use */
	private String location;
	
	/* The process parameters */
	private ProcessParameters processParameter;
	
	/* The class path */
	private String classpath;
		
	/* Profile file */
	private String profileFile;

	/* The agent name */
	private String agentType;

	/* The agent type */
	private String agentName;
	
	/* The monitor that keeps track of the progress */
	private IProgressMonitor monitor;
	
	public AbstractProcessLauncher (String agentName, String agentType)
	{
		this.agentName = agentName;
		this.agentType = agentType;
	}

	/**
	 * BUG 312125 - Use the file encoding specified by the launch if set. This allows
	 * the user to specify an alternate console encoding in the case their application
	 * needs to use an encoding other than the system's default (in particular, on English 
	 * Windows systems the default is Cp1252 which is single-byte and thus is unable to encode
	 * many Unicode characters).
	 *  
	 * @param launch 
	 * @param vmArgs
	 * @return vmArgs + the file encoding argument added if it was set in the launch
	 */
	private String ensureFileEncoding(final ILaunch launch, final String vmArgs) {

		if( vmArgs.indexOf("-Dfile.encoding") < 0 ) {
			String encoding = launch.getAttribute(DebugPlugin.ATTR_CONSOLE_ENCODING);
			if( encoding != null )
				return vmArgs + " -Dfile.encoding=" + encoding;
		}

		return vmArgs;
	}
	
	public void preLaunch(ILaunchConfiguration configuration, String mode, ILaunch launch, IProgressMonitor monitor) throws CoreException
	{		
		this.conf = configuration;
		this.monitor = monitor;
		monitor.beginTask("", 7);
		
		hostName = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, CommonUITraceConstants.LOCAL_HOST);
		
		String workingDir = getWorkingDirectory(conf);
		if (workingDir != null) {
			location = workingDir;
		}
		else
		{
			location = System.getProperty("user.dir");
		}

		// BUG 312125 - ensure we use the specified console encoding in the launched JVM
		processParameter = new ProcessParameters (ensureFileEncoding(launch, getVMArguments(conf)), getMainTypeName(conf), getProgramArguments(conf));
		classpath = getClasspathString(conf);
		port = LauncherUtility.getPort(conf);
		
		
		if (conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILE_TO_FILE, false))
			profileFile = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_FILE, (String) null);
		
		monitor.worked(1);

		final IWorkingSet[] workingSets = LauncherUtility.getActiveWorkingSets();

		ILaunchConfigurationWorkingCopy wc = conf.getWorkingCopy();
		// Provide launch objects to extensions to allow for filter processing, as needed (bug 323330)
		HyadesTraceUIExtensionSupportUtil.applyExtensionFilterProcessing(wc, workingSets);
		wc.doSave();
		
		// Alexander Simbirtsev: when "Automatically determine filtering criteria" option
		// is enabled we determine filter set analysing launch configuration and workspace 
		if (LauncherUtility.isAutoFilteringCriteria(conf)) {
			LauncherUtility.storeAutogeneratedFilterSet(getLaunchTypeWorkspaceFilters(conf));
		}
	}
	
	
	/**
	 * Returns the working directory based on the configuration passed in
	 *  
	 * @param conf The configuration
	 * @return The working directory path
	 */
	abstract protected String getWorkingDirectory(ILaunchConfiguration conf);


	/**
	 * Returns the program argument based on the configuration passed in
	 *  
	 * @param conf The configuration
	 * @return The program argument
	 */
	abstract protected String getProgramArguments(ILaunchConfiguration conf);
	
	
	/**
	 * Returns the main type name based on the configuration passed in
	 *  
	 * @param conf The configuration
	 * @return The main type name
	 */
	abstract protected String getMainTypeName(ILaunchConfiguration conf);
	
	
	/**
	 * Returns the VM argument based on the configuration passed in
	 *  
	 * @param conf The configuration
	 * @return The VM argument
	 */
	abstract protected String getVMArguments(ILaunchConfiguration conf);	


	public Process createProcess() throws CoreException
	{
		return CommonDelegateHelper.createProcess(node);
	}

	public ProcessParameters getProcessParameters(ILaunchConfiguration launchConfiguration) 
	{
		monitor.worked(1);
		return processParameter;
	}
		
	public void configureProcess(Process process, TRCProcessProxy trcProcessProxy) throws CoreException
	{	
		String applicationName = processParameter.getApplicationName();
		
		/* Format the name better */
		if (applicationName != null && applicationName.length() > 0)
			trcProcessProxy.setName(applicationName);
		else
		{
			try
			{
				trcProcessProxy.setName(LauncherUtility.resolveProcessAttributes(process)[0]);
			} 
			catch (InactiveProcessException e)
			{
				/* Doesn't need to be handled */
			}
		}
		
		String nodeName;
		
		try
		{
			/* If process or its node is null, then we didn't make a successful connection with the Agent Controller */
			if (process == null || process.getNode() == null)
			{
				throw new CoreException(new Status(IStatus.ERROR, UIPlugin.getPluginId(), IStatus.ERROR, null, null));
			}
				
			nodeName = process.getNode().getName();
		} 
		catch (InactiveProcessException e)
		{
			nodeName = getHostName();
		}
		
		CommonDelegateHelper.configureProcess(process, trcProcessProxy, nodeName, getLocation(), getClassPath());
		
		monitor.worked(1);
	}

	
	/**
	 * By default, a single agent is associated with the process passed in.
	 * Contributors should overwrite this method if they wish to associate multiple
	 * agents to the process.
	 */
	public Agent createAgent(Process process, TRCProcessProxy trcProcessProxy) throws CoreException
	{
		Agent agent = CommonDelegateHelper.createAgent (conf, process, trcProcessProxy, getAgentName(), getAgentType(), getProfileFile());
		monitor.worked(1);
		return agent;
	} 

	
	protected String getClasspathString(ILaunchConfiguration conf) throws CoreException 
	{
		String classPath = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_CLASSPATH, (String) null);

		if (classPath == null) {
			StringBuffer buf = new StringBuffer();
			String[] entries = getClasspath(conf);
			if (entries == null) {
				return buf.toString();
			}
			for (int i = 0; i < entries.length - 1; ++i) {
				buf.append(entries[i]);
				buf.append(File.pathSeparatorChar);
			}
			buf.append(entries[entries.length - 1]);
			return buf.toString();
		}
		else {
			return classPath;
		}
	}


	/**
	 * Returns the class path entry of the configuration passed in
	 * @param conf The configuration
	 * @return The class path entry
	 */
	abstract protected String[] getClasspath(ILaunchConfiguration conf);


	/**
	 * @see IDataCollectorMutualLauncher#createNode()
	 */	
	public Node createNode()
	{
		node = PDCoreUtil.profileConnect(getHostName(), String.valueOf(getPortNumber()));
		monitor.worked(1);
		return node;
	}
	
	/**
	 * @see org.eclipse.tptp.trace.ui.provisional.launcher.AbstractPrimitiveProcessLauncher#postLaunch(org.eclipse.core.runtime.IStatus)
	 */
	public void postLaunch(IStatus launchStatus)
	{
		monitor.worked(1);
		monitor.done();

		// Alexander Simbirtsev: Auto-Generated filter set should be removed to prevent it from
		// exploring by the user
		LauncherUtility.removeAutogeneratedFilterSet();
	}

	/**
	 * By default, this is a process launcher
	 */
	public boolean isProcessLauncher()
	{
		return true;
	}
	
	/**
	 * By default, this is a mutual launcher
	 */
	public boolean isMutualLauncher()
	{
		return true;
	}


	/**
	 * Contributors can provide their own host name value by over writting this
	 * method.
	 * 
	 * @return The host name that the connection should use
	 */
	protected String getHostName() 
	{	
		return hostName;
	}

	
	/**
	 * Contributors can provide their own port no. by over writting this
	 * method.
	 * 
	 * @return The port no that the connection should use
	 */
	protected int getPortNumber() 
	{
		return port;
	}

	/**
	 * Contributors can provide their own location by over writting this
	 * method.
	 * 
	 * @return The location that the process should use
	 */
	protected String getLocation()
	{
		return location;
	}
	
	
	/**
	 * Contributors can provide their own classpath by over writting this
	 * method.
	 * 
	 * @return The classpath that the process should use
	 */
	protected String getClassPath()
	{
		return classpath;
	}

	
	/**
	 * Contributors can provide their own profile file by over writting this
	 * method.
	 * 
	 * @return The profile file that the data should be directed to
	 */
	protected String getProfileFile() 
	{	
		return profileFile;
	}
	
	
	
	/**
	 * Contributors are obligated to overwrite this method if they
	 * choose to use createAgent.
	 * 
	 * @return The name of the agent that is associated with the process
	 */
	protected String getAgentName()
	{
		return agentName;
	}
	
	
	/**
	 * Contributors are obligated to overwrite this method if they
	 * choose to use createAgent.
	 * 
	 * @return The type of the agent that is associated with the process
	 */
	protected String getAgentType()
	{
		return agentType;
	}

	/**
	 * This methods returns auto-generated set of filters depending on the type
	 * of launch configuration. The filter set includes all classes from the
	 * project associated with the launch configuration and it's dependencies
	 * and excludes all the rest classes. Any active working set in the
	 * workbench is automatically taken into account (class not included in any
	 * active working set is excluded from the scope of profiling). This method
	 * should be overridden in subclasses if any additional criteria for the
	 * filter set is added.
	 * 
	 * @param configuration
	 *            launch configuration to determine filter set for
	 * @return the array of <code>FilterTableElement</code> elements. Each
	 *         element includes single class or package except for the last
	 *         element which excludes all the remaining classes.
	 * @throws CoreException
	 *             if an exception occurs while accessing corresponding
	 *             resources
	 */
	protected ArrayList<FilterTableElement> getLaunchTypeWorkspaceFilters(final ILaunchConfiguration configuration) throws CoreException {
		IJavaProject javaProject = JavaRuntime.getJavaProject(configuration);
		if (javaProject == null || javaProject.getProject() == null) {
			return new ArrayList();
		}
		
		IWorkingSet[] workingSets = LauncherUtility.getActiveWorkingSets();
		
		ArrayList<FilterTableElement> result;
		result = WorkspaceFiltersExtractor.extractFilters(javaProject.getProject(), workingSets);
		return result;
	}
}
