/**********************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: PrimaryLaunchDelegate.java,v 1.15 2007/12/21 11:59:06 ialelekov Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.trace.ui.provisional.launcher;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Status;
import org.eclipse.debug.core.DebugEvent;
import org.eclipse.debug.core.DebugPlugin;
import org.eclipse.debug.core.ILaunch;
import org.eclipse.debug.core.ILaunchConfiguration;
import org.eclipse.debug.core.model.ILaunchConfigurationDelegate;
import org.eclipse.debug.ui.DebugUITools;
import org.eclipse.debug.ui.IDebugUIConstants;
import org.eclipse.hyades.internal.execution.local.control.Agent;
import org.eclipse.hyades.internal.execution.local.control.InactiveProcessException;
import org.eclipse.hyades.internal.execution.local.control.Node;
import org.eclipse.hyades.internal.execution.local.control.Process;
import org.eclipse.hyades.internal.execution.local.control.ProcessFactory;
import org.eclipse.hyades.internal.execution.local.control.ProcessListener;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.trace.internal.ui.PDProjectExplorer;
import org.eclipse.hyades.trace.ui.UIPlugin;
import org.eclipse.hyades.trace.ui.internal.launcher.IProfileLaunchConfigurationConstants;
import org.eclipse.hyades.trace.ui.internal.launcher.ProfileLaunchUtil;
import org.eclipse.hyades.trace.ui.internal.launcher.RunLaunchProfileStatusHandler;
import org.eclipse.hyades.trace.ui.internal.util.DeleteUtil;
import org.eclipse.hyades.trace.ui.internal.util.PDCoreUtil;
import org.eclipse.hyades.trace.ui.internal.util.ProcessAdapter;
import org.eclipse.hyades.trace.ui.internal.util.ProcessMap;
import org.eclipse.hyades.trace.ui.internal.util.TraceMessages;
import org.eclipse.hyades.trace.ui.launcher.IProfilingSetType;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceConstants;
import org.eclipse.tptp.trace.ui.internal.launcher.core.AnalysisType;
import org.eclipse.tptp.trace.ui.internal.launcher.core.DataCollectorManager;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherMessages;
import org.eclipse.tptp.trace.ui.internal.launcher.core.LauncherUtility;


/**
 * This general delegator will look up the selected data collector(s) and delegate the launch to the
 * respective data collector(s).  The delegate attribute of the launch configuration type should be
 * set to this class if data collection mechanisms are intended to be used. 
 * 
 * This class is not meant to be subclassed.
 * @author Ali Mehregani
 */
public class PrimaryLaunchDelegate implements ILaunchConfigurationDelegate, Runnable
{
	/* The model entity representing a process that is launched as a 
	 * result of the profile/monitor launch */	
	private TRCProcessProxy trcProcessProxy;
	
	/* The execution node */
	private Node node;
	
	/* The resolved node name */
	private String nodeName;

	/* The launch configuration */
	private ILaunchConfiguration configuration;
	
	/* The mode of the launch */
	private String mode;
	
	/* The result of a launch */
	private ILaunch launch;
	
	/* The monitor */
	private IProgressMonitor monitor;
	
	/* Set if any errors occurred while launching */
	private CoreException error;
	
	/* Indicates the completion of the launcher */
	private boolean finished;
	
	
	public PrimaryLaunchDelegate()
	{		
	}
	
	public PrimaryLaunchDelegate(ILaunchConfiguration configuration, String mode, ILaunch launch, IProgressMonitor monitor)
	{
		this.configuration = configuration;
		this.mode = mode;
		this.launch = launch;
		this.monitor = monitor;
		this.finished = false;
	}
	
	/**
	 * Delegate the launch to the data collector(s) used.
	 * 
	 * @throws CoreException If an error occurs while attempting to delegate the launch
	 * to the selected data collector(s). 
	 */
	public void launch(ILaunchConfiguration configuration, String mode, ILaunch launch, IProgressMonitor monitor) throws CoreException 
	{	
		/* Create a new instance of the primary launch */
		PrimaryLaunchDelegate primaryLauncher = new PrimaryLaunchDelegate(configuration, mode, launch, monitor);
		
		/* Start the launcher */
		new Thread(primaryLauncher, "Pimary-Launcher").start();
		
		/* Wait until the launcher completes or we time-out */
		blockUntilCompletion(primaryLauncher);
		
		/* Throw any errors that may have occurred */
		if (primaryLauncher.getError() != null)
			throw primaryLauncher.getError();
	}
	
	public void run() 
	{
		ArrayList launchDelegates = null;
		try
		{
			String dataCollectors = configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_COLLECTOR_AND_ANALYSIS, (String)null);
			Hashtable selectedCollectors;
			
			/* For backward compatibility - check to see if there is an associated profiling set */
			if (dataCollectors == null)
			{	// commented to open profile dialog, bug #213201
				// otherwise default jvmpi profiling is started even for jvm 1.5 and 1.6 
//				String serializedSelection = LauncherUtility.serializeSelection(configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_PROFILING_SET, (String)null));
//				if (serializedSelection == null || (selectedCollectors = LauncherUtility.unserializeSelection(serializedSelection)) == null)
//				{
					openProfileDialog();
					return;
//				}
			}
			else
			{
				selectedCollectors = LauncherUtility.unserializeSelection (configuration);
			}				
			
			
			/* Parse the data collectors and delegate the launch */		
			if (selectedCollectors == null || selectedCollectors.size() <= 0)
			{
				openProfileDialog();
				return;
			}
				
			/* For backward compatibility, we'll need to invoke the launch method of all the profiling types that are selected */
			ProfileLaunchUtil.performProfilingTypesLaunch(configuration);
							
			
			/* Store all data collectors in the launch delegate list
		     * KEY = launch delegate, VALUE = data collector id */
			Hashtable delegateCollectorAssociation = LauncherUtility.validateLaunchItems(selectedCollectors, configuration, monitor);
			if (delegateCollectorAssociation == null)
				return;
			
			launchDelegates = new ArrayList(delegateCollectorAssociation.size());
			Enumeration launchDelegatesEnum = delegateCollectorAssociation.keys();
			while (launchDelegatesEnum.hasMoreElements()) launchDelegates.add(launchDelegatesEnum.nextElement());
			
			int listSize = launchDelegates.size();	
			
			/* The total work is divided by the number of launchers involved */
			monitor.beginTask(LauncherMessages.LAUNCHING_BEGIN_TASK, listSize);
			
			try
			{ 
				launchDelegates = LauncherUtility.delegateInit(launchDelegates, configuration, mode, launch, monitor);
				
				/* Update the list size and begin launching mutual launchers */
				listSize = launchDelegates.size();
				if (listSize <= 0)
				{
					finished = true;
					return;
				}
				
				/* Check to see if there are any process launchers.  If so, then create the execution process */
				Process process = null;
				IDataCollectorMutualLauncher mutualLauncher;
				IDataCollectorProcessLauncher processLauncher = null;
				for (int i = 0; i < listSize; i++)
				{					
					mutualLauncher = (IDataCollectorMutualLauncher)launchDelegates.get(i);
					
					if (i == 0)
					{
						node = mutualLauncher.createNode();
						nodeName = node == null ? configuration.getAttribute(IProfileLaunchConfigurationConstants.ATTR_HOSTNAME, CommonUITraceConstants.LOCAL_HOST) : node.getName();
					}
						
					
					if (mutualLauncher.isProcessLauncher())
					{
						processLauncher = (IDataCollectorProcessLauncher)mutualLauncher;
						process = processLauncher.createProcess();
						break;
					}
				}
				
				/* Create the model entities */
				createModelEntities(configuration, process);
				
				IDataCollectorProcessLauncher currentProcessLauncher;
				ProcessParameters processParameters = new ProcessParameters();
				processParameters.setFilterDuplicates(true);
				
				/* Continue with the launch of the mutual launchers */
				for (int i = 0; i < listSize; i++)
				{
					mutualLauncher = (IDataCollectorMutualLauncher)launchDelegates.get(i);
					boolean isProcessLauncher = mutualLauncher.isProcessLauncher(); 
					
					/* Collect the process parameters */
					if (isProcessLauncher && process != null)
					{
						currentProcessLauncher = (IDataCollectorProcessLauncher)mutualLauncher;
						ProcessParameters currentParameter = currentProcessLauncher.getProcessParameters(configuration);
						String applicationName = processParameters.getApplicationName();
						if (applicationName == null || applicationName.length() <= 0)
							processParameters.setApplicationName(currentParameter.getApplicationName());
						
						processParameters.add(currentParameter);
						currentProcessLauncher.configureProcess(process, trcProcessProxy);
						Agent agent = currentProcessLauncher.createAgent(process, trcProcessProxy);
						addAnalysisTypes(agent, selectedCollectors, currentProcessLauncher, delegateCollectorAssociation);
					}
					
					/* This should never happen */
					else if (isProcessLauncher && process == null)
					{
						LauncherUtility.openErrorWithDetail(LauncherMessages.LAUNCHER_COMMON_ERROR_TITLE, 
										NLS.bind(LauncherMessages.ERROR_DCM_INVALID_PROCESS, processLauncher == null ? null : processLauncher.getClass().getName()), null);
					}					
					
					/* This launcher doesn't create a process */
					else
					{
						Agent agent = ((IDataCollectorAgentLauncher)mutualLauncher).createAgent(trcProcessProxy);
						addAnalysisTypes(agent, selectedCollectors, mutualLauncher, delegateCollectorAssociation);
					}
				}
												
				/* If there is a process, then add the collected parameters and launch it */
				if (process != null)
				{
					PDCoreUtil.addProcessParameters(process, processParameters, true);
					launchProcess(process, launch);
				}
				
				/* Notify all launchers that the launch is complete */
				for (int i = 0; i < listSize; i++)
				{
					((IDataCollectorMutualLauncher)launchDelegates.get(i)).postLaunch(Status.OK_STATUS);
				}	
			}
			catch (Exception e)
			{
				/* A specific delegate has failed to do the launch.  Display a warning and continue 
				 * with the rest of the launches */
				if (e.getMessage() != null && e.getMessage().length() > 0)
					LauncherUtility.openWarningWithDetail(LauncherMessages.LAUNCHER_COMMON_WARNING_TITLE, 
							LauncherMessages.WARNING_DCM_FAILED_DELEGATE, e);
				
				if (trcProcessProxy != null)
				{
					Display.getDefault().syncExec(new Runnable(){

						public void run()
						{
							DeleteUtil.deleteProcess(trcProcessProxy, true, false, false, PDProjectExplorer.ID);							
						}
					});					
				}
				
				UIPlugin.getDefault().log(e);				
			}
		}
		catch (Throwable t)
		{
			try
			{
				if (launchDelegates != null)
				{
					for (int i = 0, listSize = launchDelegates.size(); i < listSize; i++)
					{
						if (((IDataCollectorBaseLauncher)launchDelegates.get(i)).isMutualLauncher())
							((IDataCollectorMutualLauncher)launchDelegates.get(i)).postLaunch(new Status(IStatus.ERROR, UIPlugin.getPluginId(), IStatus.ERROR, LauncherMessages.ERROR_DCM_LAUNCH, t));					
					}
				}
			}
			catch (Exception nestedException)
			{
				/* Quietly ignore any exception that may occur while 
				 * attempting to invoke the post launch methods */
			}
			
			monitor.setCanceled(true);
			
			/* Wrap the exception around a core exception */
			error = LauncherUtility.createCoreException(IStatus.ERROR, t.getMessage(), (t instanceof Exception ? (Exception)t : null));
		}
		
		finished = true;
		synchronized(this)
		{
			this.notify();
		}
	}
		
	
	private void openProfileDialog()
	{
		monitor.setCanceled(true);
		Display.getDefault().asyncExec(new Runnable() 
		{
			public void run() 
			{
				Shell shell = UIPlugin.getActiveWorkbenchShell();
				DebugUITools.openLaunchConfigurationDialog(shell, configuration, IDebugUIConstants.ID_PROFILE_LAUNCH_GROUP, new Status(Status.OK, UIPlugin.getPluginId(), RunLaunchProfileStatusHandler.CODE, "", null)); //$NON-NLS-1$
			}
		});
		finished = true;
	}

	private void addAnalysisTypes(Agent agent, Hashtable collectorAnalysisTypeAssociation, IDataCollectorMutualLauncher mutualLauncher, Hashtable delegateCollectorAssociation)
	{
		/* Associate the analysis types selected for the data collector that was just launched */
		String dataCollectorId = mutualLauncher == null ? null : (String)delegateCollectorAssociation.get(mutualLauncher);
		List analysisTypes = (dataCollectorId == null ? null : (List)collectorAnalysisTypeAssociation.get(DataCollectorManager.getInstance().getDataCollector(dataCollectorId)));
		
		if (analysisTypes == null)
			return;
		for (int i = 0, analysisTypeCount = analysisTypes.size(); i < analysisTypeCount; i++)		
		{
			Object currentAnalysisType = analysisTypes.get(i);
			if (currentAnalysisType instanceof AnalysisType)
			{
				LauncherUtility.associateAnalysisType(agent, ((AnalysisType)currentAnalysisType).getId());
			}
			else if (currentAnalysisType instanceof IProfilingSetType)
			{
				LauncherUtility.associateAnalysisType(agent, ((IProfilingSetType)currentAnalysisType).getId());
			}
		}
	}


	private void blockUntilCompletion(PrimaryLaunchDelegate primaryLauncher) 
	{
		/* Wait 60 seconds in increments of 100 millisecond before bailing out */
		final int INC = 100;
		final int TOTOAL_COUNT = 600;
		int counter = 0;
		while (counter < TOTOAL_COUNT && !primaryLauncher.isCompleted())
		{
			synchronized(primaryLauncher)
			{
				try 
				{
					primaryLauncher.wait(INC);
					counter++;
				} 
				catch (InterruptedException e) 
				{
					/* Ignore exception */
				}
			}			
		}
	}
	
	
	/**
	 * Indicates whether the launcher is completed
	 * @return a flag indicating the status
	 */
	public boolean isCompleted() 
	{	
		return finished;
	}

	/**
	 * Return any error that may have occurred during the launch processes
	 * @return Errors that may have occurred during the launch
	 */
	public CoreException getError()
	{
		return error;
	}

	/**
	 * Launch the process passed in.
	 * @param process The process to be launched
	 * @param launch 
	 */
	private void launchProcess(Process lprocess, ILaunch launch)
	{
        try 
        {        	
            lprocess.launch();
            
            final org.eclipse.debug.core.model.IProcess adapter = new ProcessAdapter(lprocess, launch, true, 1);
            UIPlugin.getDefault().registerLaunchProcess(adapter);
            lprocess.addProcessListener(new ProcessListener() 
            {               
				public void processLaunched(Process process)
				{
				}
				public void processExited(Process process)
				{
					UIPlugin.getDefault().deregisterLaunchProcess(adapter);
					DebugPlugin.getDefault().fireDebugEventSet(new DebugEvent[] {
							new DebugEvent(adapter, DebugEvent.TERMINATE)
					});
				}
            });
            
            // add the mapping between TRCProcessProxy -> IProcess
            int pID = Integer.parseInt(lprocess.getProcessId());
            Iterator iter = trcProcessProxy.getNode().getProcessProxies().iterator();
            while (iter.hasNext()) 
            {
               TRCProcessProxy proxy = (TRCProcessProxy)iter.next();
               if (pID == proxy.getPid() && (proxy.isActive()==true)) { /** bug 83130 **/	
                  ProcessMap.put(proxy, adapter);
                  break;
               }
            }
        } 
        
        catch (Exception e) 
        {
             String text = TraceMessages.PLERR_INFO_;
             final String msg = NLS.bind(TraceMessages.PL_ERROR_, lprocess.getExecutable());
             final Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.INTERNAL_ERROR, text, null);

             CommonPlugin.logError(e);
             
             Display.getDefault().syncExec(new Runnable() {
					public void run() {
		                ErrorDialog.openError(UIPlugin.getActiveWorkbenchShell(), TraceMessages.TRC_MSGT, msg, err);
					}
             });
         }
	}

	
	/**
	 * Create the model entities that should be displayed as a result of the
	 * launch.
	 * 
	 * @param conf The launch configuration
	 * @param process The process to be launched
	 * @throws CoreException If there is an error in creating the data model entities.
	 */
	private void createModelEntities(ILaunchConfiguration conf, Process process) throws CoreException
	{
		IPreferenceStore store = UIPlugin.getDefault().getPreferenceStore();
		
		/* Get the required attributes for the model entities */
		String projectName = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_PROJECT, store.getString(CommonUITraceConstants.TRACE_PROJECT_NAME));
		String monitorName = conf.getAttribute(IProfileLaunchConfigurationConstants.ATTR_DESTINATION_MONITOR, store.getString(CommonUITraceConstants.TRACE_MONITOR_NAME));
		
		
		/* Create the contiainer and the monitor */
        IContainer container = PDCoreUtil.createContainer(new Path(projectName));
        TRCMonitor monitor = PDCoreUtil.createMonitor(container, monitorName);

        /* create the node and the process */
        TRCNode trcNode = PDCoreUtil.createNode(monitor, nodeName, String.valueOf(LauncherUtility.getPort(conf)));
        
        /* If there is a process an actual process involved */
        
        
        if (process != null)
        {
        	trcProcessProxy = PDCoreUtil.createProcess(trcNode, process, false);
        	
        	/* Add an environment variable that would allow us to identify the launch item type
        	 * used to launch this process */
        	LauncherUtility.addLaunchConfigId(trcProcessProxy, conf.getType().getIdentifier());
        	process.addProcessListener(new ProcessListener(){

            	boolean invalidate = false;
    			public void processLaunched(Process process)
    			{
    				if (!invalidate)
    				{
    					trcProcessProxy.setRuntimeId(process.getUUID());
    					try
    					{
    						trcProcessProxy.setPid(Integer.parseInt(process.getProcessId()));
    					} catch (InactiveProcessException e)
    					{
    						/* Shouldn't happen */
    						e.printStackTrace();
    					}
    				}
    			}

    			public void processExited(Process process)
    			{
    				invalidate = true;
    				
    			}});
        }
        
        /* Attempt to create a dummy process */
        else if (node != null)
        {
        	process = ProcessFactory.createProcess(node);
        	trcProcessProxy = PDCoreUtil.createProcess(trcNode, process, false);
        }
        
        /* If all fails, then throw an exception */
        else
        {
        	throw LauncherUtility.createCoreException(IStatus.ERROR, LauncherMessages.ERROR_LAUNCH_MODEL_CREATION);
        }

	}

}
