/**********************************************************************
 * Copyright (c) 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.tptp.trace.ui.provisional.launcher;

import java.util.ArrayList;
import java.util.List;

/**
 * Represents the process parameters.  The parametes are divided into:
 * [executable parameters] [application name] [application parameters].
 * <p>Example:
 * -DsomeProperty=propertyValue HelloWorld "Hello World" 
 * </p>
 * 
 * @author Ali Mehregani
 *
 */
public class ProcessParameters 
{
	/* The executable parameters */
	private List executableParameters;
	
	/* The application name */
	private String applicationName;
	
	/* The application parameters */
	private List applicationParameters;
	
	/* Indicates whether duplicated executable/application parameters should be filtered */
	private boolean filterDuplicates;
	
	/**
	 * A constructor provided for convenience to set all three
	 * attributes at the stame time
	 * 
	 * @param executableParameters The executable parameters.  Expected elements are java.lang.String
	 * @param applicationName The application name
	 * @param applicationParameters The application parameters.  Expected elements are java.lang.String
	 */
	public ProcessParameters(String executableParameters, String applicationName, String applicationParameters) 
	{
		this();
		
		if (executableParameters != null && executableParameters.length() > 0)
			this.executableParameters.add(executableParameters);
		
		this.applicationName = applicationName;
		
		if (applicationParameters != null && applicationParameters.length() > 0)
			this.applicationParameters.add(applicationParameters);
	}

	
	/**
	 * Default constructor
	 */
	public ProcessParameters()
	{
		this.executableParameters = new ArrayList();
		this.applicationParameters = new ArrayList();
		this.applicationName = "";
	}



	/**
	 * Adds the executable/application parameters of the passed in
	 * argument to the parameters of this instance IF:
	 * this.getApplicationName().equals(processParameter.getApplicationName())
	 * 
	 * @param currentParameter The process parameter to be added
	 */
	public void add(ProcessParameters processParameter)
	{
		if (this.getApplicationName().equals(processParameter.getApplicationName()))
		{
			addExecutableParameter(processParameter.getExecutableParameters());
			addApplicationParameter(processParameter.getApplicationParameters());			
		}
	}

	
	/**
	 * @return the applicationName
	 */
	public String getApplicationName()
	{
		return applicationName;
	}


	/**
	 * @param applicationName the applicationName to set
	 */
	public void setApplicationName(String applicationName)
	{
		this.applicationName = applicationName;
	}

	
	/**
	 * @return the applicationParameters
	 */
	public List getApplicationParameters()
	{
		return applicationParameters;
	}

	
	/**
	 * Returns a string representation of the application
	 * parameters.
	 * 
	 * @return A string representation of application parameters
	 */
	public String getApplicationParametersStr()
	{
		return listToString(applicationParameters);
	}

	private String listToString(List list)
	{
		StringBuffer sb = new StringBuffer();
		for (int i = 0, listSize = list.size(); i < listSize; i++)
		{
			sb.append(" ");
			sb.append((String)list.get(i));			
		}
		return sb.toString();
	}


	/**
	 * @param applicationParameters the applicationParameters to set
	 * expected elements are java.lang.String
	 */
	public void setApplicationParameters(List applicationParameters)
	{
		this.applicationParameters = applicationParameters;
	}

	
	/**
	 * Add param to application parameters
	 * 
	 * @param param The parameter to be added
	 */
	public void addApplicationParameter(String param)
	{
		addStringToList(param, applicationParameters);
	}


	/**
	 * Add the parameters stored in the passed argument to
	 * application parameters
	 * 
	 * @param params The list of paramaters to be added
	 */
	public void addApplicationParameter(List params)
	{
		addListToList (applicationParameters, params);	
	}


	private void addStringToList(String param, List list)
	{
		if (param == null)
			return;
		
		if (filterDuplicates && list.contains(param))
			return;
		
		list.add(param);
	}

	
	private void addListToList(List list1, List list2)
	{
		if (list2 == null)
			return;
		for (int i = 0, listSize = list2.size(); i < listSize; i++)
		{
			addStringToList((String)list2.get(i), list1);
		}
	}
	
	/**
	 * @return the executableParameters
	 */
	public List getExecutableParameters()
	{
		return executableParameters;
	}

	
	/**
	 * Returns a string representation of the executable
	 * parameters.
	 * 
	 * @return A string representation of executable parameters
	 */
	public String getExecutableParametersStr()
	{
		return listToString(executableParameters);
	}
	
	

	/**
	 * @param executableParameters the executableParameters to set
	 * expected elements are java.lang.String
	 */
	public void setExecutableParameters(List executableParameters)
	{
		this.executableParameters = executableParameters;
	}

	/**
	 * Add param to executable parameters
	 * 
	 * @param param The parameter to be added
	 */
	public void addExecutableParameter(String param)
	{
		addStringToList(param, executableParameters);
	}


	/**
	 * Add the parameters stored in the passed argument to
	 * executable parameters
	 * 
	 * @param params The list of paramaters to be added
	 */
	public void addExecutableParameter(List params)
	{
		addListToList (executableParameters, params);	
	}
	
	
	/**
	 * @return the filterDuplicates
	 */
	public boolean isFilterDuplicates()
	{
		return filterDuplicates;
	}


	/**
	 * @param filterDuplicates the filterDuplicates to set
	 */
	public void setFilterDuplicates(boolean filterDuplicates)
	{
		this.filterDuplicates = filterDuplicates;
	}

}
