/**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ConnectUtilUI.java,v 1.5 2010/11/08 20:42:36 mreid Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.security.internal.util;

import java.security.KeyStore;
import java.security.cert.X509Certificate;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.hyades.security.internal.util.ui.CertificateDetails;
import org.eclipse.hyades.security.internal.util.ui.SaveCertificate;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.window.Window;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.PlatformUI;

public class ConnectUtilUI implements IConnectUtilUI {
	protected ConnectivityDialog _connectivityDialog;

	protected int btnPressed = 0;
	protected boolean okPressed = false;
	protected String alias = null;
	
	protected static Shell getValidShell() {
		Shell shell = null;
		IWorkbench workbench = PlatformUI.getWorkbench();
		if (workbench != null) {
			if (workbench.getActiveWorkbenchWindow() != null) {
				shell = workbench.getActiveWorkbenchWindow().getShell();
				if ((shell != null) && (!shell.isDisposed()))
					return shell;
			}

			if (workbench.getWorkbenchWindows().length > 0) {
				shell = workbench.getWorkbenchWindows()[0].getShell();
				if ((shell != null) && (!shell.isDisposed()))
					return shell;
			}
		}

		return null;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.security.internal.util.IConnectUtilUI#openQuestionDialog(java.lang.String, java.lang.String, java.lang.String[])
	 */
	public int openQuestionDialog(final String title, final String msg, final String[] btnLabels) {
		Display.getDefault().syncExec(new Runnable() {
			public void run() {
				if (btnLabels != null) {
					MessageDialog dialog = new MessageDialog(getValidShell(), title, null, msg, MessageDialog.QUESTION, btnLabels, 0);
					btnPressed = dialog.open();
				} else
					btnPressed = MessageDialog.openQuestion(getValidShell(), title, msg) == true ? 0 : 1;
			}
		});

		return btnPressed;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.security.internal.util.IConnectUtilUI#openCertificateDetailDialog(java.security.cert.X509Certificate)
	 */
	public void openCertificateDetailDialog(final X509Certificate certificate) {
		Display.getDefault().syncExec(new Runnable() {
			public void run() {
				CertificateDetails dlg = new CertificateDetails(getValidShell());
				dlg.init(certificate);
				dlg.open();
			}
		});
	}


	/* (non-Javadoc)
	 * @see org.eclipse.hyades.security.internal.util.IConnectUtilUI#openSaveCertificateDialog(java.security.KeyStore)
	 */
	public boolean openSaveCertificateDialog(final KeyStore ks) {
		Display.getDefault().syncExec(new Runnable() {
			public void run() {
				SaveCertificate dlg = new SaveCertificate(getValidShell(), ks);
				okPressed = dlg.open() == 0;
				alias = dlg.getAlias();
			}
		});
		return okPressed;
	}

	/**
	 * Get the user name and password the first time
	 * @return
	 */
	public ConnectivityDialog getUserDialog(String userId, String hostName) {
		return getUserDialog(true, userId, hostName);
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.security.internal.util.IConnectUtilUI#openDialog(org.eclipse.jface.dialogs.Dialog)
	 */
	public void openDialog(final Dialog dialog) {
		dialog.getShell().getDisplay().syncExec(new Runnable() {
			public void run() {
				dialog.open();
			}
		});
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.security.internal.util.IConnectUtilUI#openErrorDialog(java.lang.String, java.lang.String, org.eclipse.core.runtime.IStatus)
	 */
	public void openErrorDialog(final String title, final String message, final IStatus err){
		Display.getDefault().syncExec(new Runnable() {
			public void run() {
				ErrorDialog.openError(getValidShell(), title, message, err);
			}
		});
		
	}
	
	public IConnectUtilUser promptAuthentication(String hostName, String userName){
		ConnectivityDialog dialog = getUserDialog(userName, hostName);
		if((dialog != null) && (dialog.getReturnCode() == Window.OK)) {
			ConnectUtilUser user = new ConnectUtilUser(dialog.getUserId(), dialog.getPassword());
			return user;
		}
		else {
			return null;
		}
	}
	

	protected ConnectivityDialog getUserDialog(boolean isValidPassword, final String userId, final String _hostName) {
		if(isValidPassword) {
			Display.getDefault().syncExec(new Runnable() {
				public void run() {
					_connectivityDialog = new ConnectivityDialog(getValidShell(), _hostName, userId, (userId == null || userId.equals("")));
					_connectivityDialog.setDescription(NLS.bind(SecurityMessages._51, _hostName));
					_connectivityDialog.create();
					openDialog(_connectivityDialog);
				}
			});
		} else {
			Display.getDefault().syncExec(new Runnable() {
				public void run() {
					_connectivityDialog = new ConnectivityDialog(getValidShell(), _hostName, userId, (userId == null || userId.equals("")));
					_connectivityDialog.setDescription(NLS.bind(SecurityMessages._52, _hostName));
					_connectivityDialog.create();
					openDialog(_connectivityDialog);
				}
			});
		}

		if (_connectivityDialog.getReturnCode() != Window.OK) {
			return null; /* Bug 145071 */
		}

		return _connectivityDialog;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.security.internal.util.IConnectUtilUI#getAlias()
	 */
	public String getAlias() {
		return alias;
	}
}
