/********************************************************************** 
 * Copyright (c) 2006, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.trace.ui;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.Vector;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.loaders.util.HyadesResourceExtensions;
import org.eclipse.hyades.loaders.util.LoadersUtils;
import org.eclipse.hyades.log.ui.provisional.ILogUIHelper;
import org.eclipse.hyades.models.hierarchy.CorrelationContainerProxy;
import org.eclipse.hyades.models.hierarchy.HierarchyPackage;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.TRCAgentProxy;
import org.eclipse.hyades.models.hierarchy.TRCMonitor;
import org.eclipse.hyades.models.hierarchy.TRCNode;
import org.eclipse.hyades.models.hierarchy.TRCProcessProxy;
import org.eclipse.hyades.models.hierarchy.util.HierarchyResourceSetImpl;
import org.eclipse.hyades.models.hierarchy.util.HyadesExtendedResourceFactory;
import org.eclipse.hyades.models.hierarchy.util.IHyadesResourceExtension;
import org.eclipse.hyades.models.hierarchy.util.SaveUtil;
import org.eclipse.hyades.security.internal.util.ConnectUtil;
import org.eclipse.hyades.trace.ui.internal.util.TString;
import org.eclipse.hyades.trace.ui.provisional.ITraceUIHelper;
import org.eclipse.hyades.ui.extension.INavigatorItem;
import org.eclipse.hyades.ui.internal.extension.NavigatorExtensionUtil;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.hyades.ui.internal.navigator.Navigator;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.MessageDialogWithToggle;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIConstants;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUIBaseTraceConstants;
import org.eclipse.tptp.platform.common.ui.trace.internal.CommonUITraceMessages;
import org.eclipse.tptp.platform.common.ui.trace.internal.TraceUIManager;
import org.eclipse.ui.IViewPart;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;

import com.ibm.icu.text.DateFormat;
import com.ibm.icu.util.ULocale;

public class HyadesUtil
{
	/**
	 * Test the Agent Controller connection on a host with given port number
	 */
	public static final int CONNECTION_SUCCESS = 0;

	public static final int CONNECTION_CONTROLLER_ERROR = 1;

	public static final int CONNECTION_HOST_ERROR = 2;

	public static final int CONNECTION_PORT_ERROR = 3;

	protected static StringBuffer tmpBuffer = new StringBuffer();

	/**
	 * Returns the folder for the given selection
	 * 
	 * @param selection
	 *            an Object, the selection
	 * @return an IResource, the folder for the selection
	 */
	public static IResource getSelectionFolder(Object selection)
	{
		if (selection != null)
		{

			if (selection instanceof IResource)
				return (IResource) selection;

			if (selection instanceof EObject)
			{
				String strPath = TString.resourcePath(((EObject) selection).eResource().getURI());
				return ResourcesPlugin.getWorkspace().getRoot().findMember((new Path(strPath)).removeLastSegments(1));
			}
			if (selection instanceof INavigatorItem)
				return getSelectionFolder(((INavigatorItem) selection).getParent());
		}

		return null;
	}

	/**
	 * Returns the folder for the given viewer.
	 * 
	 * @param selection
	 *            an INavigator, the viewer
	 * @return an IResource, the folder for the viewer
	 */
	public static IResource getSelectionFolder(INavigator viewer)
	{
		if (!viewer.getStructuredSelection().isEmpty())
			return getSelectionFolder(viewer.getStructuredSelection().getFirstElement());

		return null;
	}

	/**
	 * Returns an appropriate label for the given agent. (i.e. <monitoring>,
	 * <monitoring...collecting>, etc)
	 * 
	 * @param a
	 *            TRCAgentProxy instance
	 * @return String the label
	 */
	public static String getAgentLabel(TRCAgentProxy a)
	{
		tmpBuffer.setLength(0);

		if (a.isActive() && a.isMonitored())
			tmpBuffer.append("<").append(CommonUITraceMessages.AG_MON);
		else if (a.isActive() && a.isAttached())
			tmpBuffer.append("<").append(CommonUITraceMessages.AG_AT);
		else if (a.isActive())
			tmpBuffer.append("<").append(CommonUITraceMessages.AG_DET);
		else if (isNotInitialized(a))
		{
			return "";
		} else
			tmpBuffer.append("<").append(CommonUITraceMessages.AG_TERM);

		if (a.isCollectionData())
			tmpBuffer.append(CommonUITraceMessages.COLLECT).append(">");
		else
			tmpBuffer.append(">");

		return tmpBuffer.toString();
	}

	protected static boolean isNotInitialized(TRCAgentProxy a)
	{
		return (a.getType() == null || (a.getType() != null && a.getType().equals(CommonUIConstants.PROFILE_AGENT_TYPE))) && !a.eIsSet(HierarchyPackage.eINSTANCE.getTRCAgentProxy_RuntimeId());
	}

	/**
	 * Returns the displayable agent name for the given agent.
	 * 
	 * @param agent
	 *            a TRCAgentProxy, the agent for which the displayable name is
	 *            returned.
	 * @return java.lang.String
	 */
	public static String getAgentName(TRCAgentProxy agent)
	{
		String type = agent.getType();
		String name = agent.getName();
		if (name == null)
			name = "";

		if (type.equals(CommonUIConstants.PROFILE_AGENT_TYPE))
		{
			if (!name.equals(CommonUIConstants.J2EE_AGENT_NAME))
				name = CommonUITraceMessages.TRCNM;
		} else if (type.equals(CommonUIConstants.LOG_AGENT_TYPE))
		{
			if (name.equals(""))
				name = CommonUITraceMessages.LGNM;
		}

		if (agent.getStartTime() > 0)
		{
			Date date = new Date((long) agent.getStartTime() * 1000);
			String timestamp = DateFormat.getDateTimeInstance(DateFormat.SHORT, DateFormat.MEDIUM).format(date);
			
			//use MEDIUM data format for Polish.
			//see https://bugs.eclipse.org/bugs/show_bug.cgi?id=286565
			ULocale locale = ULocale.getDefault();
			if (locale.getLanguage().equals("pl")) {
				timestamp = DateFormat.getDateTimeInstance(DateFormat.MEDIUM, DateFormat.MEDIUM).format(date);
			}
			
			name += " (" + timestamp + ")";
		}

		return name;
	}

	/**
	 * Returns the active navigator.
	 * 
	 * @return an INavigator, the active navigator
	 */
	public static INavigator getActiveNavigator()
	{
		CommonUIPlugin commonTraceUI = CommonUIPlugin.getDefault();

		if (commonTraceUI == null)
		{
			return null;
		}

		IWorkbench workbench = commonTraceUI.getWorkbench();
		if (workbench == null)
		{
			return null;
		}

		IWorkbenchWindow workbenchWindow = workbench.getActiveWorkbenchWindow();
		if (workbenchWindow == null)
		{
			return null;
		}

		IWorkbenchPage page = workbenchWindow.getActivePage();

		/*
		 * Bugzilla_77145 if(page == null ||
		 * !page.getPerspective().getId().equals(PDPerspective.ID_TRACE_PERSPECTIVE) )
		 */
		if (page == null)
			return null;

		IViewReference[] viewRefrences = page.getViewReferences();
		Navigator navigator = null;
		for (int i = 0; i < viewRefrences.length; i++)
		{
			IViewPart currentView = viewRefrences[i].getView(false);
			if (currentView instanceof Navigator && (navigator = (Navigator) currentView).isActiveNavigator())
				return navigator;
		}

		return null;

	}

	/**
	 * Returns a value specifying whether the given log agent proxies contain
	 * mixed resources.
	 * 
	 * @param agentProxies
	 *            an array of TRCAgentProxy instances, the agent proxies
	 * @return -1 if all agents are XMI serialized 0 if agents are mixed XMI and
	 *         DB 1 if all agents are DB serialized but are serizalized into
	 *         different databases. 2 if all agents are DB serialized into the
	 *         same database.
	 */
	public static short containsMixedResources(TRCAgentProxy[] agentProxies)
	{
		TRCAgentProxy agentProxy = null;
		TRCAgent agent = null;
		int xmiCount = 0;
		int dbCount = 0;
		int sameDB = 1;
		Properties p = null;
		String dbType = null;
		for (int i = 0; i < agentProxies.length; i++)
		{
			agentProxy = agentProxies[i];
			if (agentProxy == null)
			{
				continue;
			}
			agent = (TRCAgent) agentProxy.eGet(HierarchyPackage.eINSTANCE.getTRCAgentProxy_Agent(), false);
			if (agent == null)
			{
				continue;
			}
			if (EcoreUtil.getURI(agent).path().endsWith("xmi"))
			{
				++xmiCount;
			} else
			{
				++dbCount;
				URI uri = EcoreUtil.getURI(agentProxy.getAgent());
				Properties currentProperties = HyadesExtendedResourceFactory.getPropertiesFromURI(uri);
				String storeType = ((IHyadesResourceExtension) HyadesResourceExtensions.getInstance().get(LoadersUtils.getPostfix(uri.toString()))).getStoreType(LoadersUtils.getPostfix(uri.toString()));

				if (p != null && dbType != null && dbType.equals(storeType) && sameDB(p, currentProperties))
				{
					++sameDB;
				}
				p = currentProperties;
				dbType = storeType;

			}
		}
		if (dbCount == 0 && xmiCount == agentProxies.length)
		{
			return -1;
		} else if (dbCount == agentProxies.length && xmiCount == 0)
		{
			return sameDB == dbCount ? (short) 2 : 1;
		} else if (dbCount > 0 && xmiCount > 0)
		{
			return 0;
		}

		return -1;
	}

	protected static boolean sameDB(Properties p1, Properties p2)
	{
		String l1 = (String) p1.get("location");
		String l2 = (String) p2.get("location");
		if (l1 != null && l2 != null && l1.equals(l2))
		{
			return true;
		}
		return false;
	}

	/**
	 * Returns the log agents associated with the given object.
	 * 
	 * @param eObject
	 *            an EObject, the given object.
	 * @return a Collection, the log agents.
	 */
	public static Collection getLogAgents(EObject eObject)
	{

		List log_agents = new ArrayList();
		Collection processes = null;
		
		ITraceUIHelper traceUIHelper = ITraceUIHelper.INSTANCE;

		if (eObject instanceof CorrelationContainerProxy)
		{
			log_agents.addAll(((CorrelationContainerProxy) eObject).getCorrelatedAgents());
		}
		if (eObject instanceof TRCMonitor)
		{
			processes = getProcesses((TRCMonitor) eObject);
		} else if (eObject instanceof TRCNode)
		{
			processes = ((TRCNode) eObject).getProcessProxies();
		} else if (eObject instanceof TRCProcessProxy)
		{
			processes = new ArrayList(1);
			processes.add(eObject);
		} else if (eObject instanceof TRCAgentProxy && (((TRCAgentProxy) (eObject)).getType().equals(CommonUIConstants.LOG_AGENT_TYPE) || (traceUIHelper!=null && traceUIHelper.isViewForType(CommonUIConstants.LOG_VIEW_ID, ((TRCAgentProxy) (eObject)).getType()))))
		{
			log_agents.add(eObject);
		}
		if (processes != null)
		{
			TRCProcessProxy process = null;
			for (Iterator iter = processes.iterator(); iter.hasNext();)
			{
				process = (TRCProcessProxy) iter.next();
				if (process != null)
				{
					Collection agents = process.getAgentProxies();
					TRCAgentProxy agentProxy = null;
					for (Iterator iterator = agents.iterator(); iterator.hasNext();)
					{
						agentProxy = (TRCAgentProxy) iterator.next();
						if (agentProxy != null && (agentProxy.getType().equals(CommonUIConstants.LOG_AGENT_TYPE) || (traceUIHelper!=null && traceUIHelper.isViewForType(CommonUIConstants.LOG_VIEW_ID, agentProxy.getType()))))
						{
							log_agents.add(agentProxy);
						}
					}
				}
			}
		}
		return log_agents;
	}

	/**
	 * Returns the processes associated with the given monitor
	 * 
	 * @param monitor
	 *            a TRCMonitor, the given monitor
	 * @return a Collection, the processes
	 */
	public static Collection getProcesses(TRCMonitor monitor)
	{

		List nodes = monitor.getNodes();
		List processes = new ArrayList();
		int nodes_size = nodes.size();
		for (int i = 0; i < nodes_size; i++)
		{
			List list = ((TRCNode) nodes.get(i)).getProcessProxies();
			processes.addAll(list);
		}
		return processes;
	}

	/**
	 * Returns the resources associated with the given project.
	 * 
	 * @param project
	 *            an IProject, the given project
	 * @return a List, the resources
	 */
	public static List getResources(IProject project)
	{

		List resources = new ArrayList();
		IResource[] members;
		try
		{
			members = project.members();
			for (int idx = 0; idx < members.length; idx++)
			{
				IResource member = members[idx];
				if (member == null || !member.exists())
					continue;

				if (member instanceof IFile && ((IFile) member).getFileExtension() != null && ((IFile) member).getFileExtension().equals(CommonUIConstants.MONITOR_EXT))
				{
					resources.add(member);
				}

			}
		} catch (CoreException e)
		{
			e.printStackTrace();
		}

		return resources;
	}

	/**
	 * Returns the monitors associated with the given resouce and resource set.
	 * 
	 * @param resource
	 *            an IResource, the given resource
	 * @param resourceSet
	 *            a ResourceSet, the given resource set
	 * @return a List, the monitors
	 */
	public static List getMonitors(IResource resource, ResourceSet resourceSet)
	{
		List monList = new ArrayList();
		Resource res = null;
		try
		{
			String monPath = resource.getFullPath().toString();
			String name = monPath;

			res = resourceSet.getResource(SaveUtil.createURI("platform:/resource" + name + "#").trimFragment(), true);

			List ext = res.getContents();
			Iterator i = ext.iterator();
			while (i.hasNext())
			{
				Object obj = i.next();
				if (obj instanceof TRCMonitor)
				{
					monList.add(obj);
				}
			}

		} catch (Exception e)
		{
			e.printStackTrace();
		}

		return monList;

	}

	/**
	 * Changes all occurrences of oldPat to newPat, in the input String and
	 * returns the result.
	 * 
	 * @param in
	 *            a String, the input String
	 * @param oldPat
	 *            a String, the pattern to be replaced
	 * @param newPat
	 *            a String, the pattern replacing the old pattern
	 * @return a String, the changed input String
	 */
	public static String change(String in, String oldPat, String newPat)
	{
		if (oldPat.length() == 0)
			return in;
		if (oldPat.length() == 1 && newPat.length() == 1)
			return in.replace(oldPat.charAt(0), newPat.charAt(0));

		int lastIndex = 0;
		int newIndex = 0;
		tmpBuffer.setLength(0);

		for (;;)
		{
			newIndex = in.indexOf(oldPat, lastIndex);
			if (newIndex != -1)
			{
				tmpBuffer.append(in.substring(lastIndex, newIndex)).append(newPat);
				lastIndex = newIndex + oldPat.length();
			} else
			{
				tmpBuffer.append(in.substring(lastIndex));
				break;
			}
		}
		return tmpBuffer.toString();
	}

	/**
	 * Returns the selected trace or log object in the Profiling Monitors view
	 * or Log Navigator.
	 * 
	 * @return an EObject, the selected object
	 */
	public static EObject getMofObject()
	{
		Object sel = getNavigatorSelection(getActiveNavigator());
		if (sel instanceof EObject)
			return (EObject) sel;
		else if (sel instanceof INavigatorItem)
			return (EObject) ((INavigatorItem) sel).getData();
		else
			return null;
	}

	/**
	 * Returns the selection for the given navigator.
	 * 
	 * @param navigator
	 *            an INavigator instance, the given navigator
	 * @return an EObject, the selection
	 */
	public static EObject getNavigatorSelection(INavigator navigator)
	{

		if (navigator == null)
			return null;
		IStructuredSelection selection = navigator.getStructuredSelection();
		if (!selection.isEmpty())
		{
			Object obj = ((IStructuredSelection) selection).getFirstElement();

			if (obj != null)
			{
				if (obj instanceof EObject)
					return (EObject) obj;
				else if (obj instanceof INavigatorItem)
					return (EObject) ((INavigatorItem) obj).getData();
			}
		}

		return null;

	}

	/**
	 * Tests the connection of the given host and port returning the status of
	 * the test.
	 * 
	 * @param host
	 *            a String, the host for testing the connection
	 * @param port
	 *            a String, the port associated with the host
	 * @param message
	 *            a boolean if true, displays a popup message
	 * @param shell
	 *            a Shell instance the shell for any popup messages
	 * @return an int, the test connection result
	 */
	public static int testConnection(String host, String port, boolean message, Shell shell)
	{
		if (host == null || host.trim() == "" || port == null || port.trim() == "")
		{
			String text = CommonUITraceMessages.INVALID_HOST;
			Status err = new Status(Status.WARNING, ResourcesPlugin.PI_RESOURCES, IResourceStatus.WARNING, text, null);
			ErrorDialog.openError(shell, CommonUITraceMessages.TRC_MSGT, CommonUITraceMessages.CON_ERROR_, err);

			return CONNECTION_HOST_ERROR;
		}

		ITraceUIHelper traceUIHelper = ITraceUIHelper.INSTANCE;
		if(traceUIHelper!=null)
		{
			IPreferenceStore store = traceUIHelper.getTraceUIPreferenceStore();

			ConnectUtil util = new ConnectUtil(host, port, store.getString(CommonUIConstants.SECURE_USERID), traceUIHelper.getTraceUIApplication());
			int result = util.connect();
			if (result == ConnectUtil.CONNECTION_SUCCESS)
			{// store the user id
				store.setValue(CommonUIConstants.SECURE_USERID, util.getUserId());
	
				if (message)
				{
					String text = CommonUITraceMessages.CON_SUC;
					MessageDialog.openInformation(shell, CommonUITraceMessages.TRC_MSGT, text);
				}
	
			}
			return result;
		}
		
		//This should never happen
		return -1;
	}

	/**
	 * Returns the object that should be viewed for the given selection object.
	 * For example, if the process node is selected, the view should display the
	 * profiling data if trace collects profiling data. It will return the agent
	 * if there is only one log agent.
	 * 
	 * @param selObject
	 *            an EObject, the selected object
	 * @return an EObject, the object that should be viewed
	 */
	public static EObject getObjectToView(EObject selObject)
	{

		if (selObject == null)
			return selObject;

		List list = null;

		if (selObject instanceof TRCMonitor)
		{

			list = getLogObjectInMonitor((TRCMonitor) selObject);

		} else if (selObject instanceof TRCNode)
		{

			list = getLogObjectInNode((TRCNode) selObject);

		} else if (selObject instanceof TRCProcessProxy)
		{

			list = getLogAgentInProcess((TRCProcessProxy) selObject);

		}

		if (list != null && list.size() == 1)
		{
			return (EObject) list.get(0);
		}
		return selObject;
	}

	/**
	 * Returns a list of log objects in the given process.
	 * 
	 * @param selObject
	 *            an EObject, the given process
	 * @return a List, the list of log objects
	 */
	public static List getLogAgentInProcess(EObject selObject)
	{

		int nrOfAgents = 0;
		List agents = ((TRCProcessProxy) selObject).getAgentProxies();
		int size = agents.size();
		TRCAgentProxy a = null;
		List list = new ArrayList();

		ITraceUIHelper traceUIHelper = ITraceUIHelper.INSTANCE;

		for (int idx = 0; idx < size && nrOfAgents < 2; idx++)
		{
			a = (TRCAgentProxy) agents.get(idx);
			if (a == null || a.eIsProxy())
				continue;
			if (a.getType().equals(CommonUIConstants.LOG_AGENT_TYPE) || (traceUIHelper!=null && traceUIHelper.isViewForType(CommonUIConstants.LOG_VIEW_ID, a.getType())))
			{
				nrOfAgents++;
				list.add(a);
			}
		}

		return list;

	}

	/**
	 * Returns a list of log objects in the given node.
	 * 
	 * @param selObject
	 *            an EObject, the given node
	 * @return a List, the list of log objects
	 */
	public static List getLogObjectInNode(EObject selObject)
	{

		List processes = ((TRCNode) selObject).getProcessProxies();
		TRCProcessProxy process = null;
		int nrOfProcesses = 0;
		int size = processes.size();
		List list;
		List object_list = new ArrayList();

		for (int i = 0; i < size && nrOfProcesses < 2; i++)
		{
			process = (TRCProcessProxy) processes.get(i);
			if (process != null)
			{
				list = getLogAgentInProcess(process);
				if (list.size() >= 1)
				{
					nrOfProcesses++;
					object_list.add(process);
				}
			} else
			{
				continue;
			}
		}

		if (object_list.size() == 1)
		{
			list = getLogAgentInProcess((TRCProcessProxy) object_list.get(0));
			if (list.size() == 1)
			{
				object_list.clear();
				object_list.add(list.get(0));
			}

		}
		return object_list;
	}

	/**
	 * Returns a list of log objects in the given monitor.
	 * 
	 * @param selObject
	 *            an EObject, the given monitor
	 * @return a List, the list of log objects
	 */
	public static List getLogObjectInMonitor(EObject selObject)
	{

		java.util.List nodes = ((TRCMonitor) selObject).getNodes();
		TRCNode node = null;

		int size = nodes.size();
		int nrOfNodes = 0;
		List object_list = new ArrayList();
		List list;

		for (int i = 0; i < size && nrOfNodes < 2; i++)
		{
			node = (TRCNode) nodes.get(i);
			if (node == null)
			{
				continue;
			} else
			{

				list = getLogObjectInNode(node);
				if (list.size() >= 1)
				{
					nrOfNodes++;
					object_list.add(node);
				}
			}

		}

		if (object_list.size() == 1)
		{
			list = getLogObjectInNode((TRCNode) object_list.get(0));
			if (list.size() == 1)
			{
				object_list.clear();
				object_list.add(list.get(0));
			}
		}

		return object_list;
	}

	/**
	 * Returns the log agents associated with the given object and type of log.
	 * 
	 * @param eObject
	 *            an EObject, the given object.
	 * @param type
	 *            an int, the type of log (CommonUIConstants.XMI or
	 *            CommonUIConstants.SQL).
	 * @return a Collection, the log agents.
	 */
	public static Collection getLogAgents(EObject eObject, int type)
	{
		List result = new ArrayList();
		Collection logAgents = getLogAgents(eObject);
		TRCAgentProxy element = null;
		TRCAgent agent = null;
		for (Iterator iter = logAgents.iterator(); iter.hasNext();)
		{
			element = (TRCAgentProxy) iter.next();
			agent = (TRCAgent) element.eGet(HierarchyPackage.eINSTANCE.getTRCAgentProxy_Agent(), false);
			if (agent == null)
			{
				continue;
			}
			if (type == CommonUIConstants.XMI && EcoreUtil.getURI(agent).path().endsWith("xmi"))
			{
				result.add(element);
			} else if (type == CommonUIConstants.SQL && !EcoreUtil.getURI(agent).path().endsWith("xmi"))
			{
				result.add(element);
			}
		}
		return result;
	}

	/**
	 * 
	 * @param name
	 * @return the name to be displayed by the ui
	 */
	public static String resolveLogAgentDisplayName(String name)
	{
		if (name == null)
			return "";

		int idx = name.lastIndexOf("  #  ");
		if (idx != -1 && idx < name.length() - 1)
			return name.substring(idx + 5);

		// added for defect 158234 in order to allow path with
		// whitespace
		int bsidx = name.lastIndexOf("\\");
		int fsidx = name.lastIndexOf("/"); // added '/' to fix defect 165704
		idx = bsidx > fsidx ? bsidx : fsidx;
		if (idx != -1 && idx < name.length() - 1)
			return name.substring(idx + 1);

		return name;
	}

	/**
	 * 
	 * @param name
	 * @return the log agent type
	 */
	public static String resolveLogAgentType(String name)
	{
		if (name == null)
			return "";

		int idx = name.indexOf("  #  ");
		if (idx != -1)
			return name.substring(0, idx);
		//added for defect 158234 in order to allow for spaces in
		// log paths
		idx = name.indexOf(CommonUITraceMessages.LOG);
		if (idx > 0)
			return name.substring(0, idx)+ CommonUITraceMessages.LOG;

		return name;
	}

	/**
	 * Cleans up the resources for the given navigatorID
	 * 
	 * @param navigatorID
	 *            a String, the navigator ID
	 */
	public static void cleanupResources(String navigatorID)
	{

		ProfileEvent event = new ProfileEvent();
		event.setSource(null);
		event.setType(ProfileEvent.CLEAN_UP);
		TraceUIManager.getTraceUIManager().notifyProfileEventListener(event);

		IWorkbenchPage page = null;
		IWorkbenchWindow[] windows = CommonUIPlugin.getDefault().getWorkbench().getWorkbenchWindows();
		for (int i = 0; i < windows.length; i++)
		{
			IWorkbenchWindow window = windows[i];
			IWorkbenchPage[] pages = window.getPages();
			for (int j = 0; j < pages.length; j++)
			{
				page = pages[j];
				if (page != null && page.getPerspective() != null && page.getPerspective().getId().equals(CommonUIBaseTraceConstants.ID_TRACE_PERSPECTIVE))
				{
					break;
				}
			}
		}

		if (page != null)
		{
			if (navigatorID.equals(CommonUIBaseTraceConstants.ID_PD_NAVIGATOR_VIEW))
			{
				IViewPart part = page.findView(CommonUIBaseTraceConstants.ID_LOG_NAVIGATOR_VIEW);
				if (part != null)
					return;
			} else if (navigatorID.equals(CommonUIBaseTraceConstants.ID_LOG_NAVIGATOR_VIEW))
			{
				IViewPart part = page.findView(CommonUIBaseTraceConstants.ID_PD_NAVIGATOR_VIEW);

				if (part != null)
					return;
			}

		}

		try
		{

			boolean modified = false;
			boolean modifiedLoggingAgent = false;
			boolean modifiedProfilingAgent = false;
			if (HierarchyResourceSetImpl.getInstance().getResources().size() > 0)
			{

				Iterator docs = HierarchyResourceSetImpl.getInstance().getResources().iterator();
				while (docs.hasNext())
				{
					org.eclipse.emf.ecore.resource.Resource res = (org.eclipse.emf.ecore.resource.Resource) docs.next();
					if (res != null && res.isLoaded() && res.isModified())
					{
						modified = true;
						
			
						/* Navid Mehregani - bugzilla 163434: Check to see what kind of agent has NOT been saved. This is done
						 * so that the user is prompted for the right message when asked whether they want to save their data. */
						EList contents = res.getContents();
						for (int i=0; i < contents.size(); i++)
						{
							if (contents.get(i) instanceof TRCMonitor)
							{
								TRCMonitor monitor = (TRCMonitor)contents.get(i);
								Vector agentTypeList = new Vector();
								getAgentType(monitor, agentTypeList);
								
								for (int j=0; j < agentTypeList.size(); j++)
								{
									String agentType = (String) agentTypeList.get(j);
									
									/* Check the agent type */
									if (agentType.equalsIgnoreCase("Logging"))
										modifiedLoggingAgent = true;
									else if (agentType.equalsIgnoreCase("Profiler"))
										modifiedProfilingAgent = true;	
								}
							}
						}							
						
						/* Done for performance reasons */
						if (modifiedLoggingAgent && modifiedProfilingAgent)
							break;
						
						/* End of bugzilla 163434 */
					}
				}

				Collection external = NavigatorExtensionUtil.getAllModifiedItems(null);
				boolean externalModified = !external.isEmpty();

				if (modified || externalModified)
				{
					
					/* N.M - bugzilla 163434: Make sure the right message is displayed when prompting the user whether they
					 * want to save their data. */
					String title = CommonUITraceMessages.TRC_MSGT;
					String msg = CommonUITraceMessages.SAVE_PSQ;
					
					if ((modifiedLoggingAgent) && (modifiedProfilingAgent))
					{
						title = CommonUITraceMessages.PRO_LOG_MSG;
						msg = CommonUITraceMessages.SAVE_PSQ_LOG;
					}
					else if (modifiedLoggingAgent)
					{
						title = CommonUITraceMessages.LOG_MSG;
						msg = CommonUITraceMessages.SAVE_LOG;
					}
					/* End of bugzilla 163434 */
					
					String toggleMsg = CommonUITraceMessages.RDEC;

					IPreferenceStore store;
					ITraceUIHelper traceUIHelper = ITraceUIHelper.INSTANCE;
					if(traceUIHelper!=null)
					{
						store = traceUIHelper.getTraceUIPreferenceStore();
					}
					else
					{
						store = CommonUIPlugin.getDefault().getPreferenceStore();
					}
					
					String value = store.getString(CommonUIBaseTraceConstants.SAVE_ON_EXIT_KEY);
					if (value == null || "".equals(value))
					{
						value = MessageDialogWithToggle.PROMPT;
						store.setValue(CommonUIBaseTraceConstants.SAVE_ON_EXIT_KEY, value);
					}

					boolean saveRequested = false;
					if (MessageDialogWithToggle.PROMPT.equals(value))
					{
						saveRequested = MessageDialogWithToggle.openYesNoQuestion(null, title, msg, toggleMsg, false, store, CommonUIBaseTraceConstants.SAVE_ON_EXIT_KEY).getReturnCode() == IDialogConstants.YES_ID;
					}

					if (saveRequested || MessageDialogWithToggle.ALWAYS.equals(value))
					{
						if (modified)
						{
							SaveUtil.saveDocuments();
							IResource res = ResourcesPlugin.getWorkspace().getRoot();
							res.refreshLocal(IResource.DEPTH_INFINITE, null);
						}
						if (externalModified)
						{
							Iterator i = external.iterator();
							while (i.hasNext())
								((INavigatorItem) i.next()).save(false);
						}
					}
					else
					{
						/* Navid Mehregani
						 * bugzilla 101906: When the user chooses not to save their resources, make sure the corresponding project 
						 * is deleted when it's empty. (e.g. ProfileProject) An empty project will be useless to the user and will 
						 * just clutter their workspace  
						 */
						List resources = HierarchyResourceSetImpl.getInstance().getResources();
						Vector projectNames = new Vector();
						final String resourceString = ":/resource";
						String workspaceLocation = Platform.getLocation().toString();
						boolean isContainedInVector = false;
						String project = null;
						
						/* Go thorough the list of resources and save their corresponding project names */
						for (int i=0; i < resources.size(); i++)
						{
							Resource resource = (Resource) resources.get(i);
							project = resource.getURI().toString();
							project = project.substring(project.indexOf(resourceString) + resourceString.length(), project.lastIndexOf('/'));
							isContainedInVector = false;
							
							/* Was this project already added to our list? */
							for (int k=0; k < projectNames.size(); k++)
							{
								if (projectNames.get(k).equals(project))
								{
									isContainedInVector = true;
									break;
								}
							}
							
							if (!isContainedInVector)
								projectNames.add(project);
						}
						
						/* Go through all the projects and delete the empty ones */
						for (int i=0; i < projectNames.size(); i++)
						{
							try
							{							
								File file = new File(workspaceLocation + ((String)projectNames.get(i)));
								String[] fileList = file.list();
								boolean okToDelete = true; 
								if (fileList == null)
								{
									/* If fileList is null then file is not really pointing to a directory */
									okToDelete = false;
								}
								else
								{
									for (int k=0; k < fileList.length; k++)
									{
										/* We're not going to care if the project only contains a .project file */
										if (!fileList[k].equals(".project"))
										{
											okToDelete = false;
											break;
										}
									}	
								}								
								
								if (okToDelete)
								{
									try {
										/* Delete the empty project */
										ResourcesPlugin.getWorkspace().getRoot().getProject((String)projectNames.get(i)).delete(false, null);
										
									} catch (CoreException e) {
										IStatus status = new Status(IStatus.ERROR, CommonUIPlugin.getDefault().getBundle().getSymbolicName(), 0, "", e);
										CommonUIPlugin.getDefault().getLog().log(status);
									}
								}
								
							} catch (Exception e)
							{
								CommonPlugin.logError(e);
							}
						}
						
						/* End of bugzilla 101906 */
					}
				}

				// //clear documents list
				// SaveUtil.getDocuments().clear();
				List resources = HierarchyResourceSetImpl.getInstance().getResources();
				int s = resources.size();
				List documents = new ArrayList(s);
				documents.addAll(resources);
				for (int i = 0; i < s; i++)
				{
					Resource res = ((Resource) documents.get(i));

					if (!PlatformUI.getWorkbench().isClosing())
					{
						if (res.isLoaded())
						{
							try
							{
								res.unload();
							} catch (Exception exc)
							{
							}
						}
					} else
						res.setModified(false);
				}

				if (!PlatformUI.getWorkbench().isClosing())
				{
					resources.clear();
				}

			}
		} catch (Exception e)
		{
			CommonPlugin.logError(e);
		}
		
		ILogUIHelper logUIHelper = ILogUIHelper.INSTANCE;
		if(logUIHelper!=null)
		{
			//AnalysisHelper.getInstance().dispose();
			logUIHelper.disposeAnalysisHelper();
		}
		
		ITraceUIHelper traceUIHelper = ITraceUIHelper.INSTANCE;
		
		if(traceUIHelper!=null)
		{
			//TraceUIManager.getTraceUIManager().dispose();
			traceUIHelper.disposeTraceUIManager();
			//PDContentProvider.resetMonitors();
			traceUIHelper.resetMonitors();
		}

	}
	
	/**
	 * Will add the list of agent types, under this monitor, to the 'agentTypeList' Vector
	 * This method was added as a result of bugzilla 158564
	 * 
	 * @param monitor  The monitor that's of interest
	 * @param agentTypeList The vector to add the agent types to
	 */
	public static void getAgentType(TRCMonitor monitor, Vector agentTypeList)
	{
		EList nodeList = monitor.getNodes();
		
		for (int i=0; i < nodeList.size(); i++)
		{
			getAgentType((TRCNode)nodeList.get(i), agentTypeList);
		}
	}
	
	/**
	 * Will add the list of agent types, under this node, to the 'agentTypeList' Vector
	 * This method was added as a result of bugzilla 158564
	 * 
	 * @param node  The node that's of interest
	 * @param agentTypeList The vector to add the agent types to
	 */
	public static void getAgentType(TRCNode node, Vector agentTypeList)
	{
		EList processList = node.getProcessProxies();
		
		for (int i=0; i < processList.size(); i++)
		{
			getAgentType((TRCProcessProxy)processList.get(i), agentTypeList);
		}
	}
	
	/**
	 * Will add the list of agent types, under this process, to the 'agentTypeList' Vector
	 * This method was added as a result of bugzilla 158564
	 * 
	 * @param process  The process that's of interest
	 * @param agentTypeList The vector to add the agent types to
	 */
	public static void getAgentType(TRCProcessProxy process, Vector agentTypeList)
	{
		EList agentList = process.getAgentProxies();
		
		for (int i=0; i < agentList.size(); i++)
		{
			agentTypeList.add(getAgentType((TRCAgentProxy) agentList.get(i)));
		}
	}
	
	public static String getAgentType(TRCAgentProxy agent)
	{
		return agent.getType();
	}

	/**
	 * 
	 * @param type
	 * @param name
	 * @param userInput
	 * @return a concatenated value stored as the log agent name
	 */
	public static String codeLogAgentName(String type, String name, String userInput)
	{
		StringBuffer tmp = new StringBuffer(type);
		tmp.append("  #  ").append(name).append("  #  ").append(userInput);
		return tmp.toString();
	}
	
	/**
	 * 
	 * @param name
	 * @return the log agent path
	 */
	public static String resolveLogAgentPath(String name)
	{
		if(name == null)
			return "";
		
		int idx=name.indexOf("  #  ");
		if(idx!=-1)
		{
			name = name.substring(idx+5);
		
	        idx=name.indexOf("  #  ");
	        if(idx!=-1)
	        	return name.substring(0, idx);
		}
		else
		{
			// added for defect_158234 in order to accomodate whitespace in
			// Path names
			idx = name.indexOf(CommonUITraceMessages.LOG);
			if(idx!=-1 && idx<name.length()-1)
				if (name.startsWith(" "))
				{
					return name.substring(1, idx +3);
				}
				else
					return name.substring(idx+3);
		}
        
		return name;
	}
	
	
	/**
	 * Returns an appropriate label for the given process. 
	 * Form: processName at hostName [PID: processID]
	 * 
	 * @param process the process element for which to get the label.
	 * @return an appropriate label for the given process. 
	 */
	public static String getProcessName(TRCProcessProxy process)
	{
		/*
		 * Only show the process ID if it's a valid one (>0).
		 */
		if (process.getPid() > 0) {
			String msg = (process.getName() == null || process.getName().length() == 0) ? CommonUITraceMessages.UNK : process.getName();
			return NLS.bind(CommonUITraceMessages.PRC_LNG
					, new Object[] {msg, process.getNode().getName(), String.valueOf(process.getPid())});
		}
		else {
			String msg = (process.getName() == null || process.getName().length() == 0) ? CommonUITraceMessages.UNK : process.getName();
			return NLS.bind(CommonUITraceMessages.PRC_LNG_NID, new Object[]{ msg, process.getNode().getName()});
		}
	}
	
	
	//bug 259631
	/**
	 * Returns the local port number specified in the AC preference pages
	 * 
	 * @return local port number specified in the AC preference page that workbench would listen to. 
	 */
	public static String getLocalPortNumber(){
		
		 IPreferenceStore store = CommonUIPlugin.getDefault().getPreferenceStore();
		 if(store != null)
			 return store.getString(CommonUIConstants.LOCALHOST_PORT);       
		 else
			 return null;
	}
	
	//end of 259631

}
