/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: NamedElementPropertySource.java,v 1.4 2008/01/24 02:28:59 apnan Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.adapter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.hyades.ui.util.INamedElement;
import org.eclipse.osgi.util.TextProcessor;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIMessages;
import org.eclipse.ui.views.properties.IPropertyDescriptor;
import org.eclipse.ui.views.properties.IPropertySource;
import org.eclipse.ui.views.properties.PropertyDescriptor;

/**
 * Property source for {@link org.eclipse.hyades.ui.util.INamedElement} instances.
 * 
 * @author  Paul E. Slauenwhite
 * @author  Marcelo Paternostro
 * @version January 23, 2008
 * @since   Auguest 16, 2006
 */
public class NamedElementPropertySource 
implements IPropertySource
{
	/**
	 * Info category.
	 * @see IPropertySource
	 */
	public final static String PROP_INFO = CommonUIMessages._3;
	
	/**
	 * File category.
	 * @see IPropertySource
	 */
	public final static String PROP_RESOURCE = CommonUIMessages._51;

	/**
	 * Name property constant
	 * @see IPropertySource
	 */
	protected final static String PROP_NAME = CommonUIMessages._4;

	/**
	 * Description property constant
	 * @see IPropertySource
	 */
	protected final static String PROP_DESCRIPTION = CommonUIMessages._5;

	/**
	 * The named element that this property source applies to.
	 */
	protected INamedElement namedElement;
	
	/**
	 * The property source of the file associated to the named element.
	 */
	protected IPropertySource filePropertySource;
	
	/**
	 * Constructor for NamedElementPropertySource
	 * 
	 * <p>This constructor should be used only by subclasses that must to ensure 
	 * that the named element is set after the instantiation.
	 */
	public NamedElementPropertySource()
	{
	}

	/**
	 * Constructor for NamedElementPropertySource
	 * @param namedElement
	 */
	public NamedElementPropertySource(INamedElement namedElement)
	{
		setNamedElement(namedElement);
	}
	
	/**
	 * Sets the named element.
	 * @param namedElement
	 */
	protected void setNamedElement(INamedElement namedElement)
	{
		this.namedElement = namedElement;
		if(namedElement == null)
			return;
			
		IFile file = (IFile)namedElement.getAdapter(IFile.class);
		if(file != null)
			filePropertySource = (IPropertySource)file.getAdapter(IPropertySource.class);
	}
	
	/**
	 * Returns the named element.
	 * @return INamedElement
	 */
	public INamedElement getNamedElement()
	{
		return namedElement;
	}

	/**
	 * @see org.eclipse.ui.views.properties.IPropertySource#getPropertyDescriptors()
	 */
	public IPropertyDescriptor[] getPropertyDescriptors()
	{
		List propertyDescriptors = new ArrayList();
		
		PropertyDescriptor descriptor = new PropertyDescriptor(PROP_NAME, PROP_NAME.toLowerCase());
		descriptor.setCategory(PROP_INFO);
		descriptor.setAlwaysIncompatible(true);
		propertyDescriptors.add(descriptor);

		propertyDescriptors.addAll(Arrays.asList(addPropertyDescriptors()));

		descriptor = new PropertyDescriptor(PROP_DESCRIPTION, PROP_DESCRIPTION.toLowerCase());
		descriptor.setCategory(PROP_INFO);
		descriptor.setAlwaysIncompatible(true);
		propertyDescriptors.add(descriptor);
		
		if(filePropertySource != null)
		{
			IPropertyDescriptor[] filePropertyDescriptors = filePropertySource.getPropertyDescriptors();
			for(int i = 0, maxi = filePropertyDescriptors.length; i < maxi; i++)
			{
				if(filePropertyDescriptors[i] instanceof PropertyDescriptor)
					((PropertyDescriptor)filePropertyDescriptors[i]).setCategory(PROP_RESOURCE);
			}
			propertyDescriptors.addAll(Arrays.asList(filePropertyDescriptors));
		}
					
		return (IPropertyDescriptor[])propertyDescriptors.toArray(new IPropertyDescriptor[propertyDescriptors.size()]);
	}
	
	/**
	 * Adds property descriptors between the name and the description property.
	 * Subclasses may overwrite to provide detailed information.
	 * @return IPropertyDescriptor[]
	 */
	protected IPropertyDescriptor[] addPropertyDescriptors()
	{
		return new IPropertyDescriptor[0];
	}

	/**
	 * @see org.eclipse.ui.views.properties.IPropertySource#getPropertyValue(java.lang.Object)
	 */
	public Object getPropertyValue(Object id)
	{
		if(PROP_NAME.equals(id))
			return TextProcessor.process(namedElement.getName());

		if(PROP_DESCRIPTION.equals(id))
			return TextProcessor.process(namedElement.getDescription());
		
		if(filePropertySource != null)
			return filePropertySource.getPropertyValue(id);
			
		return null;
	}

	/**
	 * @see org.eclipse.ui.views.properties.IPropertySource#isPropertySet(java.lang.Object)
	 */
	public boolean isPropertySet(Object id)
	{
		return false;
	}

	/**
	 * @see org.eclipse.ui.views.properties.IPropertySource#getEditableValue()
	 */
	public Object getEditableValue()
	{
		return null;
	}

	/**
	 * @see org.eclipse.ui.views.properties.IPropertySource#resetPropertyValue(java.lang.Object)
	 */
	public void resetPropertyValue(Object id)
	{

	}

	/**
	 * @see org.eclipse.ui.views.properties.IPropertySource#setPropertyValue(java.lang.Object, java.lang.Object)
	 */
	public void setPropertyValue(Object id, Object value)
	{

	}
}
