/*******************************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IAssociationDescriptor.java,v 1.2 2006/09/20 19:49:11 ewchan Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.extension;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.swt.graphics.Image;

/**
 * An instance of this class represents an extension to be contributed
 * to the Hyades framework.  Examples of extensions:<OL>
 * <LI>Type descriptor</LI>
 * <LI>Editor definition</LI>
 * <LI>Analyse view</LI>
 * </OL>
 * 
 * <p>This interface is not intended to be implemented by clients.  Instances can 
 * be obtained by 
 * {@link org.eclipse.hyades.ui.extension.IAssociationMapping#createAssociationDescriptor()}.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public interface IAssociationDescriptor
extends IDisposable
{
	/**
	 * Returns the association mapping that contains this descriptor.
	 * @return IAssociationMapping
	 */
	public IAssociationMapping getAssociationMapping();
	
	/**
	 * Sets the name of this association descriptor.
	 * @return String
	 */
	public void setName(String name);

	/**
	 * Returns the name of this association descriptor.
	 * @return String
	 */
	public String getName();
	
	/**
	 * Sets the description of this association descriptor.
	 * @return String
	 */
	public void setDescription(String description);

	/**
	 * Returns the description of this association descriptor.
	 * @return String
	 */
	public String getDescription();
	
	/**
	 * Sets this test type's image descriptor.
	 * @param imageDescriptor
	 */
	public abstract void setImageDescriptor(ImageDescriptor imageDescriptor);
	
	/**
	 * Returns this association descriptor's image descriptor.
	 * @return imageDescriptor
	 */
	public abstract ImageDescriptor getImageDescriptor();
	
	/**
	 * Returns the image created from the image descriptor.
	 * @return Image
	 */
	public abstract Image getImage();
	
	/**
	 * Sets the configuration element.
	 * @param configurationElement
	 */
	public void setConfigurationElement(IConfigurationElement configurationElement);

	/**
	 * Sets the instance of the class that implements this association  
	 * descriptor.
	 * @param object
	 */
	public void setImplementationClass(Class extensionClass);

	/**
	 * Returns the class defined in this association 
	 * descriptor or <code>null</code> if there isn't any class defined.
	 * @return Object
	 */
	public Class getImplementationClass();

	/**
	 * Creates an instance of the Extension Class.  This method
	 * doesn't throw any exceptions returning null if the instance
	 * is not created.
	 * @return Object
	 */
	public Object createImplementationClassInstance();

	/**
	 * Returns the association descriptor's id.
	 * 
	 * <p>This information is used by the association mapping registry to store 
	 * the preferences related to this descriptor.
	 * 
	 * @return String.
	 */
	public String getId();

	/**
	 * Sets the id for this association descriptor.
	 * 
	 * <p>This information is used by the association mapping registry to store 
	 * the preferences related to this descriptor.
	 * 
	 * @param id
	 */
	public void setId(String id);

	/**
	 * Sets the extension for this association descriptor.
	 * @param string
	 */
	public void setExtension(String extension);

	/**
	 * Returns the extension defined for this association descriptor.
	 * @return String
	 */
	public String getExtension();

	/**
	 * Returns <code>true</code> if this descriptor applies to
	 * a given type or <code>false</code> otherwise.
	 * @param type
	 * @return boolean
	 */
	public boolean applyTo(String type);
	
	/**
	 * Returns <code>true</code> if this descriptor applies to
	 * a all type or <code>false</code> otherwise.
	 * @param type
	 * @return boolean
	 */
	public boolean applyToAllTypes();
	
	/**
	 * Adds a type that is associated to this instance.
	 * @return boolean <code>true</code> if the type was added or 
	 * <code>false</code> otherwise.
	 */
	public boolean addType(String type);	

	/**
	 * Removes a type that is associated to this instance.
	 * @return boolean <code>true</code> if the type was removed or 
	 * <code>false</code> otherwise.
	 */
	public boolean removeType(String type);	

	/**
	 * Returns the types that this association descriptor is related to.
	 * 
	 * <p>Clients should not use this method to check if a descriptor applies to
	 * a type.  The <code>apply</code> methods are the preferred way of doing this.
	 * 
	 * <p>This method doesn't follow the "get" naming convention to reduce the
	 * change of being used in the wrong scenario.
	 * 
	 * @return String[]
	 */
	public String[] types();

	/**
	 * Creates an association descriptor that has the same attributes
	 * values of this instance but that is not associated with the other objects 
	 * of the extension framework.
	 * @return IAssociationDescriptor
	 */
	public IAssociationDescriptor copy();
	
	/**
	 * Returns true is this view is the default view for this type of association 
	 * If the method returns true, this view will be opened when double click on an object having this type
	 * @return default value
	 */
	public boolean isPluginDefault();

	/**
	 * 
	 * @param value default value
	 */
	public void setPluginDefault(boolean value);
	
}
