/*******************************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: INavigatorItem.java,v 1.2 2006/09/20 19:49:11 ewchan Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.ui.extension;

import java.util.List;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.swt.graphics.Image;

/**
 * An item that can be displayed in the Hyades navigators
 * (Profiling Monitor, Log Navigator), but is not a built-
 * in type. Any client wishing to add an item to the
 * navigator(s) must implement this.
 * 
 * @see org.eclipse.hyades.ui.extension.INavigatorContribution
 * 
 * @author curtispd
 * @since 3.0
 */
public interface INavigatorItem extends IAdaptable {
	
	/**
	 * Deletes this item from the navigator. The associated
	 * {@link org.eclipse.hyades.ui.extension.INavigatorContribution}
	 * should no longer contribute this item.
	 *
	 * @param referentialIntegrity if <code>true</code>,
	 *    referential integrity should be enforced (e.g., remove
	 *    parent if it no longer has any children).
	 * @param deleteContents if <code>true</code>, the contents
	 *    on the filesystem should also be removed.
	 */
	public void delete(boolean referentialIntegrity, boolean deleteContents);

	/**
	 * Returns this item's children.
	 * 
	 * @return this item's children.
	 */
	public List getChildren();
	
	/**
	 * Returns the content of this item. This could be,
	 * for example, an EMF object containing trace data.
	 * 
	 * @return the content of this item.
	 */
	public Object getData();

	/**
	 * Returns the item's image icon. This is the icon
	 * displayed to the left of the item in the navigator.
	 * 
	 * @return the item's icon.
	 */
	public Image getImage();

	/**
	 * Returns this item's parent. If this is a top-level
	 * element, an
	 * {@link org.eclipse.core.resources.IWorkspaceRoot}
	 * should be returned.
	 * 
	 * @return this item's parent.
	 */
	public Object getParent();

	/**
	 * Returns the text to display in this item's label in
	 * the navigator.
	 *  
	 * @return the text to display in this item's label.
	 */
	public String getText();
	
	/**
	 * Returns the item's type. This is used to determine
	 * which views this item can be viewed with. You may use
	 * an existing type with views, or invent your own and
	 * contribute views via the
	 * <code>org.eclipse.hyades.ui.analyzeExtension</code>
	 * extension point.
	 * 
	 * @return the item's type.
	 */
	public String getType();

	/**
	 * Returns <code>true</code> if this item has children,
	 * otherwise <code>false</code>. This is equivalent to
	 * <code>(getChildren().size() > 0)</code>, but should
	 * be more efficient.
	 * 
	 * @return whether or not this item has children.
	 */
	public boolean hasChildren();

	/**
	 * Returns <code>true</code> if the delete action should
	 * be enabled on this item, otherwise <code>false</code>.
	 *
	 * @return whether or not the delete action should be
	 *    enabled on this item.
	 */
	public boolean isDeleteEnabled();
	
	/**
	 * Returns <code>true</code> if the save action should
	 * be enabled for this item, otherwise <code>false</code>.
	 * 
	 * @return whether or not the save action should be
	 *    enabled for this item.
	 */
	public boolean isSaveEnabled();

	/**
	 * Returns <code>true</code> if the unload action should
	 * be enabled for this item, otherwise <code>false</code>.
	 * 
	 * @return whether or not the unload action should be
	 *    enabled for this item.
	 */
	public boolean isUnloadEnabled();
	
	/**
	 * Saves the contents of this item to disk. These
	 * contents will be re-loaded on next startup.
	 * 
	 * @param referentialIntegrity if <code>true</code>,
	 *    referential integrity should be enforced.
	 */
	public void save(boolean referentialIntegrity);

	/**
	 * Unloads the contents of this item from memory.
	 * The next time the user opens this item, the
	 * contents should be re-loaded from disk.
	 * 
	 * @param referentialIntegrity if <code>true</code>,
	 *    referential integrity should be enforced (e.g.,
	 *    unload parent if it no longer has any children).
	 */
	public void unload(boolean referentialIntegrity);
}
