/**********************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IFilterAttribute.java,v 1.2 2006/09/20 19:49:12 ewchan Exp $
 *
 * Contributors: 
 * IBM - Initial API and implementation
 ***********************************************************************/

package org.eclipse.hyades.ui.filters;

import org.eclipse.hyades.models.hierarchy.extensions.WhereExpression;


/**
* This interface represents a filter attribute to be displayed in the advanced tab.
*/
public interface IFilterAttribute {
	
	/**
	 * Returns the unique ID, associated with this attribute.  The unique ID is used
	 * to later associate data with this filter attribute.
	 * 
	 * @return a String, the unique ID
	 */
	public String id();
	
	/**
	 * Returns the display text that will be visible in the advanced tab UI when
	 * adding/removing/editing attributes.
	 * 
	 * @return a String, the displayable UI text
	 */
	public String displayText();
	
	/**
	 * Validates that the input data is valid for the given displayable operator and
	 * displayable value.  If the data is valid it should return null.  If it is not valid,
	 * it should return the error message that will be displayed in the advanced tab dialog.
	 * 
	 * @param operator a String, the operator in the input that will be validated, this String would
	 * be among the Strings returned in the method operators()
	 * @param value a String, the value in the input that will be validated
	 * @return a String, null if the operator and value are valid for this filter attribute,
	 * and the displayable error message otherwise.
	 */
	public String validate(String operator, String value);
	
	/**
	 * Returns the set of operators associated with this attribute, these operators
	 * are Strings that are also displayable.  They are also returned to the methods
	 * validate(String operator, String value) and getWhereExpression(String operator, String value)
	 * for these methods to do their tasks.  These operators are shown in the combo box
	 * drop down when editing/adding attributes in the advanced tab.
	 * 
	 * @return an array of Strings, the possible operators that will be displayed for
	 * this attribute
	 */
	public String[] operators();
	
	/**
	 * Returns the WhereExpression associated with the given displayable operator
	 * and displayable value for this attribute. This WhereExpression is used to create the
	 * filter associated with the advanced tab.
	 * 
	 * @param operator a String, the operator to construct the WhereExpression from, this String would
	 * be among the Strings returned in the method operators()
	 * @param value a String, the value to construct the WhereExpression from
	 * @return a WhereExpression, the filter WhereExpression associated with this attribute
	 */
	public WhereExpression getWhereExpression(String operator, String value);
	
	/**
	 * Returns the displayable/editable value that will be in the advanced tab UI,
	 * for the given WhereExpression associated with this IFilterAttribute(matched by id).
	 * 
	 * @param whereExpression a WhereExpression associated with this attribute
	 * @return a String, the displayable value shown in the UI
	 */
	public String getValue(WhereExpression whereExpression);
	
	/**
	 * Returns the displayable/editable operator that will be in the advanced tab UI,
	 * for the given WhereExpression associated with this IFilterAttribute(matched by id).  This
	 * operator must be one of the operators returned from the method operators().
	 *  
	 * @param whereExpression a WhereExpression associated with this attribute
	 * @return a String, the displayable operator shown in the davanced tab UI, must be one of the
	 * operators from the method operators().
	 */
	public String getOperator(WhereExpression whereExpression);
	
	/**
	 * Returns the default value associated with this attribute.  This value will be displayed
	 * when this attribute is selected to be added.
	 * 
	 * @return a String, the default value for this attribute
	 */
	public String defaultValue();
}
