/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.ui.internal.action;

import java.util.Set;

import org.eclipse.hyades.ui.internal.extension.INavigatorLayout;
import org.eclipse.hyades.ui.internal.navigator.INavigator;
import org.eclipse.jface.action.Action;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Tree;
import org.eclipse.tptp.platform.common.ui.internal.util.UIUtil;

/**
 * The action that resets the navigator layout upon selection a new
 * layout (e.g. flat / hierarchical layout) in the navigator. The layouts
 * for a navigator, if any exist, should be displayed in the navigator's
 * view menu.
 * 
 * @author Curtis d'Entremont
 * @since 3.3
 */
public class LayoutAction extends Action {

	protected INavigator _navigator;
	protected INavigatorLayout _layout;
	protected ImageDescriptor _imageDescriptor;
	protected IPreferenceStore _store;
	protected String _preferenceKey;
	protected int _preferenceValue;
	
	/**
	 * Constructs a new LayoutAction for the specified navigator that
	 * applies the given layout when invoked.
	 * 
	 * @param navigator The navigator to apply the layout on.
	 * @param layout The layout to be applied for this action.
	 * @param label The action name, which is visible to the user.
	 * @param imageDescriptor The image for this action.
	 * @param store The preference store that holds the layout selection.
	 * @param preferenceKey The key in the preference store that holds the layout.
	 * @param preferenceValue The value of the preference for this particular action's layout.
	 */
	public LayoutAction(INavigator navigator, INavigatorLayout layout, String label, ImageDescriptor imageDescriptor, IPreferenceStore store, String preferenceKey, int preferenceValue) {
		super(label, AS_RADIO_BUTTON);
		_navigator = navigator;
		_layout = layout;
		_imageDescriptor = imageDescriptor;
		_store = store;
		_preferenceKey = preferenceKey;
		_preferenceValue = preferenceValue;
	}
	
	/**
	 * Returns the image descriptor for this layout action.
	 * 
	 * @return This menu action's image.
	 * @see org.eclipse.jface.action.Action#getImageDescriptor()
	 */
	public ImageDescriptor getImageDescriptor() {
		return _imageDescriptor;
	}
	
	/**
	 * Returns the INavigatorLayout that this action applies to the navigator.
	 * 
	 * @return the action's layout.
	 */
	public INavigatorLayout getLayout() {
		return _layout;
	}
	
	/**
	 * Performs the layout change for the navigator and layout given in the
	 * constructor. The rule is: Any node that was previously shown should be
	 * shown in new layout.
	 * 
	 * @see org.eclipse.jface.action.Action#run()
	 */
	public void run() {
		/*
		 * Called for all layouts; only execute the one that was clicked on.
		 */
		if (isChecked() && (_navigator.getLayout() != _layout)) {
			BusyIndicator.showWhile(Display.getDefault(), new Runnable() {
				public void run() {
					Tree tree = ((TreeViewer)_navigator.getViewer()).getTree();
					tree.setRedraw(false);
					Set visible = UIUtil.getVisibleViewerNodes((TreeViewer)_navigator.getViewer());
					_navigator.setLayout(_layout);
					updatePreferences();
					UIUtil.setVisibleViewerNodes((TreeViewer)_navigator.getViewer(), visible);
					tree.setRedraw(true);
				}
			});
		}
	}
	
	/**
	 * Updates the layout selection in preferences. The layout is persisted
	 * even if the workbench is restarted.
	 */
	protected void updatePreferences() {
		_store.setValue(_preferenceKey, _preferenceValue);
	}
}
