/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 * IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.context;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;

import org.eclipse.hyades.ui.provisional.context.ContextManager;
import org.eclipse.hyades.ui.provisional.context.ContextOpenSourceHandlerManager;
import org.eclipse.hyades.ui.provisional.context.IContextHandler;
import org.eclipse.hyades.ui.provisional.context.IContextLanguage;
import org.eclipse.hyades.ui.provisional.context.IContextOpenSourceHandler;
import org.eclipse.jface.preference.IPreferenceStore;

public class ContextMappingPreference {

	public class ContextProviders
	{
		protected String contextProviderId;
		protected String openSourceProviderId;
		
		public void contextProviderID(String contextProviderId_)
		{
			contextProviderId = contextProviderId_;
		}
		
		public void openSourceProviderID(String openSourceProviderId_)
		{
			openSourceProviderId = openSourceProviderId_;
		}
		
		public String contextProviderID()
		{
			return contextProviderId;
		}
		
		public String openSourceProviderID()
		{
			return openSourceProviderId;
		}
	}
	
	public static String CONTEXT_MAPPING_PREFERENCE = "hyades.ui.context.mapping.preference";
	protected Map mappingByContext;
	protected IPreferenceStore preferenceStore;
	protected String contextMappingStr;
	
	public ContextMappingPreference(IPreferenceStore preferenceStore_)
	{
		preferenceStore = preferenceStore_;
		contextMappingStr = null;
	}
	
	public void populateMappingByContext()
	{
		populateMappingFromPreferenceStore();

		IContextLanguage[] languages = ContextManager.getContextLanguages();
		String contextKey;
		
		for (int i = 0; i < languages.length; i++)
		{
			contextKey = languages[i].contextKey();

			if (getContextProviderId(contextKey) == null)
			{
				IContextHandler[] chs = ContextManager.getContextHandlers(contextKey);
				if (chs != null && chs.length > 0)
					setContextProviderId(contextKey, chs[0].id());
			}
			
			if (getContextOpenSourceProviderId(contextKey) == null)
			{
				IContextOpenSourceHandler[] coshs = ContextOpenSourceHandlerManager.getContextOpenSourceHandlers(contextKey);
				if (coshs != null && coshs.length > 0)
					setContextOpenSourceProviderId(contextKey, coshs[0].id());
			}
		}
	}
	
	public void populateMappingFromPreferenceStore()
	{
		contextMappingStr = getPreferenceStore().getString(CONTEXT_MAPPING_PREFERENCE);
		if (contextMappingStr == null)
			return;
		
		StringTokenizer st = new StringTokenizer(contextMappingStr, ":");
		String contextStr;
		StringTokenizer contextStrTokenizer;
		String key;
		String contextProviderId;
		String contextOpenSourceProviderId;
		
		while (st.hasMoreTokens())
		{
			contextStr = st.nextToken();
			contextStrTokenizer = new StringTokenizer(contextStr, ";");
			
			if (contextStrTokenizer.hasMoreTokens())
				key = contextStrTokenizer.nextToken();
			else
				key = null;
			
			if (contextStrTokenizer.hasMoreTokens())
				contextProviderId = contextStrTokenizer.nextToken();
			else
				contextProviderId = null;

			if (contextStrTokenizer.hasMoreTokens())
				contextOpenSourceProviderId = contextStrTokenizer.nextToken();
			else
				contextOpenSourceProviderId = null;
			
			if (key != null)
			{
				setContextProviderId(key, contextProviderId);
				setContextOpenSourceProviderId(key, contextOpenSourceProviderId);
			}
		}
	}
	
	public boolean writeMappingToPreferenceStore()
	{
		StringBuffer contextMappingStrB = new StringBuffer();
		
		Iterator keys = mappingByContext.keySet().iterator();
		String key;
		String contextProviderId;
		String contextOpenSourceProviderId;
		
		while (keys.hasNext())
		{
			key = (String)keys.next();
			contextProviderId = getContextProviderId(key);
			contextOpenSourceProviderId = getContextOpenSourceProviderId(key);
			
			contextMappingStrB.append(key);
			contextMappingStrB.append(";");
			if (contextProviderId != null)
				contextMappingStrB.append(contextProviderId);
			contextMappingStrB.append(";");
			if (contextOpenSourceProviderId != null)
				contextMappingStrB.append(contextOpenSourceProviderId);
			contextMappingStrB.append(":");
		}
		
		getPreferenceStore().setValue(CONTEXT_MAPPING_PREFERENCE, contextMappingStrB.toString());
		return true;
	}
	
	public String getContextProviderId(String contextKey)
	{
		return getContextProviders(contextKey).contextProviderID();
	}
	
	public String getContextOpenSourceProviderId(String contextKey)
	{
		return getContextProviders(contextKey).openSourceProviderID();
	}
	
	public void setContextProviderId(String contextKey, String id)
	{
		ContextProviders cps = getContextProviders(contextKey);
		cps.contextProviderID(id);
	}
	
	public void setContextOpenSourceProviderId(String contextKey, String id)
	{
		ContextProviders cps = getContextProviders(contextKey);
		cps.openSourceProviderID(id);
	}
	
	protected ContextProviders getContextProviders(String contextKey)
	{
		if (mappingByContext == null)
			mappingByContext = new HashMap();
		
		ContextProviders cps = (ContextProviders)mappingByContext.get(contextKey);
		if (cps == null)
		{
			cps = new ContextProviders();
			mappingByContext.put(contextKey, cps);
		}
		return cps;
	}
	
	public boolean changed()
	{
		String preferenceStr = getPreferenceStore().getString(CONTEXT_MAPPING_PREFERENCE);
		return preferenceStr == null || !preferenceStr.equals(contextMappingStr);
	}

	
	protected IPreferenceStore getPreferenceStore()
	{
        return preferenceStore;
	}

}
