/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.ui.internal.dbresource;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.hyades.loaders.util.HyadesResourceExtensions;
import org.eclipse.hyades.models.hierarchy.HierarchyFactory;
import org.eclipse.hyades.models.hierarchy.TRCAgent;
import org.eclipse.hyades.models.hierarchy.util.HierarchyResourceSetImpl;
import org.eclipse.hyades.models.hierarchy.util.HierarchyURIConverterImpl;
import org.eclipse.hyades.models.hierarchy.util.IHyadesResourceExtension;
import org.eclipse.hyades.models.util.ModelDebugger;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIMessages;

/**
 * @author slavescu
 */
public abstract class DBResourceSupportUICommon implements SelectionListener, ModifyListener {
	
	protected Text username;
	protected Text password;
	protected Composite container;
	protected Combo storeType;
	protected Text location;
	protected Text jdbcLocation;
	protected Button jdbcBrowse;
	protected Button jdbcTestConnection;
	protected Shell shell;
	protected Label jdbcLoc;
	protected Label lDBType;
	protected Label lUsername;
	protected Label lPassword;
	
	public void populateGroupControl(Composite lrsGroup) {
		if(container==null)
			container = lrsGroup;
		jdbcLoc = new Label(lrsGroup, SWT.NONE);
		jdbcLoc.setText(CommonUIMessages._54);
		Composite c = new Composite(lrsGroup, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		c.setLayout(layout);
		c.setLayoutData(GridDataUtil.createHorizontalFill());
		jdbcLocation = new Text(c, SWT.BORDER);
		jdbcLocation.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		jdbcBrowse = new Button(c, SWT.NULL);
		jdbcBrowse.setText(CommonUIMessages._55);
		lDBType = new Label(lrsGroup, SWT.NONE);
		lDBType.setText(CommonUIMessages._61);
		storeType = new Combo(lrsGroup, SWT.BORDER | SWT.READ_ONLY);
		storeType.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		Label lLocation = new Label(lrsGroup, SWT.NONE);
		lLocation.setText(CommonUIMessages._57);
		location = new Text(lrsGroup, SWT.BORDER);
		location.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		lUsername = new Label(lrsGroup, SWT.NONE);
		lUsername.setText(CommonUIMessages._58);
		username = new Text(lrsGroup, SWT.BORDER);
		username.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		lPassword = new Label(lrsGroup, SWT.NONE);
		lPassword.setText(CommonUIMessages._59);
		password = new Text(lrsGroup, SWT.BORDER);
		password.setEchoChar('*');
		password.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		Label lrsNote = new Label(lrsGroup, SWT.WRAP);
		lrsNote.setText(CommonUIMessages._60);
//		lrsNote.setForeground(new Color(getShell().getDisplay(),r,g,b));
//		lrsNote.setLayoutData(GridDataUtil.createHorizontalFill());
		GridData data = GridDataUtil.createHorizontalFill();
		data.heightHint = 60;
		data.widthHint = 60;
		data.horizontalSpan = 2;
		lrsNote.setLayoutData(data);

		jdbcBrowse.addSelectionListener(this);
		jdbcLocation.addModifyListener(this);
		location.addModifyListener(this);
		username.addModifyListener(this);
		password.addModifyListener(this);
				
		initializeValues(false);
		validateUserInput(true);
	}
	public abstract void initializeValues(boolean defaultValue);
	public abstract boolean storeValues();

	public Composite getControl() {
		return container;
	}
	
	public void setStoreType(List l) {
		storeType.removeAll();
		if (l != null) {
			Collections.sort(l);
			for (int i = 0; i < l.size(); i++) {
				String element = (String) l.get(i);
				storeType.add(element);
			}
			if (l.size() > 0)
				storeType.select(0);
		} else {
			storeType.removeAll();
		}
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e) {
		// do nothing
	}
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e) {
		if (e.widget == jdbcTestConnection) {
			if (shell == null)
				shell = container.getShell();
			Throwable exc = testConnection();
			if(exc==null)
			{
				MessageDialog.openInformation(shell,CommonUIMessages._56,CommonUIMessages._68);
			}
			else
			{
				MessageDialog.openError(shell,CommonUIMessages._56,CommonUIMessages._69+"\n"+exc.getLocalizedMessage());
			}
		}else if (e.widget == jdbcBrowse) {
			showJDBCLocationDialog();
			enableJDBCInputs();
			validateUserInput(false);
		}
	}
	
	protected Throwable testConnection(){
		storeValues();
		URI testUri = URI.createURI("testConnection-1234567890"+getAgentNamePostfix());
		File f = new File("testConnection-1234567890"+getAgentNamePostfix());
		Resource resource = null;
		try {
			f.delete();
			resource =  HierarchyResourceSetImpl.getInstance().createResource(testUri);
			TRCAgent agent=HierarchyFactory.eINSTANCE.createTRCAgent();
			resource.getContents().add(agent);
			resource.save(Collections.EMPTY_MAP);
		} catch (Throwable e) {
			return e;
		} finally {
		    if(resource !=null)
		    {
		        HierarchyResourceSetImpl.getInstance().getResources().remove(resource);
		    }
			f.delete();
//			deleteTestFile(testUri);
		}
		return null;

	}
	protected void deleteTestFile(URI testUri) {
		try {
			HierarchyURIConverterImpl c = new HierarchyURIConverterImpl();
			String filePath = c.toString(testUri);
			File file = new File(filePath);
			if(file.exists())
				file.delete();
		} catch (Exception e) {
			ModelDebugger.log(e);
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.internal.dbresource.DBResourceSupportUI#getStoreTypeNames(java.io.File)
	 */
	
	protected String getAgentNamePostfix() {
		String res=null;
		String activeDBType = storeType.getText();
		for (Iterator iter = HyadesResourceExtensions.getInstance().entrySet().iterator(); iter.hasNext();) {
			Map.Entry entry = (Map.Entry) iter.next();
			if(entry.getKey() instanceof String)
			{
				IHyadesResourceExtension hyadesResourceFactory = (IHyadesResourceExtension)entry.getValue();
				String postfix = (String)entry.getKey();
				if(postfix.endsWith(".trcadb") && hyadesResourceFactory.getStoreType(postfix).equals(activeDBType))
				{
					return postfix;
				}
			}
		}
		if(res==null)
			res = ".trcaxmi";
		return res;
	}	

	
	protected void enableGroup(boolean enable) {
		jdbcLocation.setEnabled(enable);
		jdbcBrowse.setEnabled(enable);
		storeType.setEnabled(enable);
//		jdbcTestConnection.setEnabled(enable);
		if (enable) {
			enableJDBCInputs();
		} else {
			location.setEnabled(enable);
			username.setEnabled(enable);
			password.setEnabled(enable);
			jdbcTestConnection.setEnabled(enable);
		}
	}
	protected abstract void enableJDBCTestConnection();
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
	 */
	public void modifyText(ModifyEvent e) {
		if (e.widget == jdbcLocation) {
			enableJDBCInputs();
			validateUserInput(false);
		} else {
			validateUserInput(true);
		}
	}
	protected abstract void validateUserInput(boolean skipJDBCcheck);
	public void enableJDBCInputs() {
		boolean validJDBC = isNetworkMode();
		location.setEnabled(validJDBC);
		username.setEnabled(validJDBC);
		password.setEnabled(validJDBC);
		if (!validJDBC) {
			location.setText("");
			username.setText("");
			password.setText("");
		}
		enableJDBCTestConnection();
	}
	protected boolean isNetworkMode() {
		return jdbcLocation.getText().endsWith(File.separator + "db2jcc.jar") || jdbcLocation.getText().endsWith(File.separator + "derbyclient.jar") || (jdbcLocation.getText().indexOf("mysql-connector-java")!=-1);
	}
	protected boolean isValidJDBCType() {
		List storeNames = new ArrayList();
		String jarLocation = jdbcLocation.getText();
		if (jarLocation == null || jarLocation.equals("")) {			
			storeNames.add(CommonUIMessages._63);
			setStoreType(storeNames);
			return false;
		}
		File trial = new File(jarLocation);
		if (trial == null || !trial.exists()) {			
			storeNames.add(CommonUIMessages._62);
			setStoreType(storeNames);
			return false;
		}
		storeNames = getStoreTypeNames(trial);
		if (storeNames.size() == 0) {
			storeNames.add(CommonUIMessages._62);
			setStoreType(storeNames);
			return false;
		} else {
			setStoreType(storeNames);
		}
		return true;
	}
	
	protected List getStoreTypeNames(File file) {
		List storeTypes = new ArrayList();
		Set processed = new HashSet();
		for (Iterator iter = HyadesResourceExtensions.getInstance().entrySet().iterator(); iter.hasNext();) {
			Map.Entry entry = (Map.Entry) iter.next();
			if (!processed.contains(entry.getKey()) && entry.getKey() instanceof String) {
				processed.add(entry.getKey());
				List types = ((IHyadesResourceExtension) entry.getValue()).getStoreTypeNames(file);
				if (types!=null && types.size()>0) {
					int s = types.size();
					for(int i=0;i<s;i++){						
						if(!storeTypes.contains(types.get(i))){
							storeTypes.add(types.get(i));
						}
					}					
				}
			}
		}
		return storeTypes;
	}

	
	protected void showJDBCLocationDialog() {
		if (shell == null)
			shell = container.getShell();
		FileDialog dlg = new FileDialog(shell);
		dlg.setFilterExtensions(new String[]{"*.jar;*.zip"});
		dlg.open();
		String classpath = dlg.getFileName();
		if (classpath != null && !classpath.equals("")) {
			classpath = dlg.getFilterPath() + File.separator + classpath;
			jdbcLocation.setText(classpath);
		}
	}
}