/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AssociationDescriptor.java,v 1.4 2010/05/07 14:47:34 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.extension;

import java.io.Serializable;
import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionRegistry;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.swt.graphics.Image;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.internal.util.UIUtil;


/**
 * <p>Implementation of the {@link org.eclipse.hyades.ui.extension.IAssociationDescriptor}
 * interface.</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Jerome Bozier
 * @author  Paul Slauenwhite
 * @version May 7, 2010
 * @since   August 16, 2006
 */
public class AssociationDescriptor implements IAssociationDescriptor, Serializable {

	protected static final long serialVersionUID = 3256725086991103289L;

	//Attributes that are not copied by the copy method
	protected AssociationMapping associationMapping;

	//Make sure that the method setAttributesOf is copying all the attributes
	protected String id;
	protected String name;
	protected String extension;
	protected String description;
	protected Class extensionClass;
	protected String imageKey;
	protected Set types;
	protected boolean isPluginDefault = false;
	protected IConfigurationElement configurationElement;
	private Object currentInstance = null;
	
	/**
	 * Constructor for AssociationDescriptor
	 * @param associationMapping The association mapping that contains this
	 * descriptor.
	 */
	public AssociationDescriptor(AssociationMapping associationMapping)
	{
		this.associationMapping = associationMapping;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		if(types != null)
			types.clear();
			
		configurationElement = null;
		extensionClass = null;
	}
	
	/**
	 * Returns the image registry that is used to store the image
	 * of this descriptor
	 * @return
	 */
	protected ImageRegistry getImageRegistry()
	{
		return associationMapping.getImageRegistry();	
	}
	
	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#getAssociationMapping()
	 */
	public IAssociationMapping getAssociationMapping()
	{
		return associationMapping;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#setName(java.lang.String)
	 */
	public void setName(String name)
	{
		this.name = name;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#getName()
	 */
	public String getName()
	{
		return name;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#setDescription(java.lang.String)
	 */
	public void setDescription(String description)
	{
		this.description = description;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#getDescription()
	 */
	public String getDescription()
	{
		return description;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#setImageDescriptor(org.eclipse.jface.resource.ImageDescriptor)
	 */
	public void setImageDescriptor(ImageDescriptor imageDescriptor)
	{
		String key = imageDescriptor.toString();
		if((getImageRegistry() != null) && (getImageRegistry().getDescriptor(key) == null))
		{
			getImageRegistry().put(key, imageDescriptor);
			imageKey = key;
		}
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#getImageDescriptor()
	 */
	public ImageDescriptor getImageDescriptor()
	{
		if((imageKey == null) || (getImageRegistry() == null))
			return null;
			
		ImageDescriptor imageDescriptor = getImageRegistry().getDescriptor(imageKey);
		if((imageDescriptor == null) && (configurationElement != null))
		{
			imageDescriptor = UIUtil.getImageDescriptorFromPlugin(Platform.getBundle(configurationElement.getDeclaringExtension().getNamespace()), imageKey);
			if(imageDescriptor != null)
				getImageRegistry().put(imageKey, imageDescriptor);
		}
		return imageDescriptor;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#getImage()
	 */
	public Image getImage()
	{
		if((imageKey == null) || (getImageRegistry() == null))
			return null;

		Image image = getImageRegistry().get(imageKey);
		if(image == null)
		{
			if(getImageDescriptor() != null)
				image = getImageRegistry().get(imageKey);
		}
		
		return image;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#setConfigurationElement(org.eclipse.core.runtime.IConfigurationElement)
	 */
	public void setConfigurationElement(IConfigurationElement configurationElement)
	{
		extensionClass = null;
		this.configurationElement = configurationElement;
	}
	
	/**
	 * Returns the configuration element.
	 * @return IConfigurationElement
	 */
	protected IConfigurationElement getConfigurationElement()
	{
		return configurationElement;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#setImplementationClass(Class)
	 */
	public void setImplementationClass(Class extensionClass)
	{
		this.extensionClass = extensionClass;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#getImplementationClass()
	 */
	public Class getImplementationClass()
	{
		currentInstance = null;
		if((extensionClass == null) && (configurationElement != null))
		{
			if(getValue(configurationElement, "class") != null)
			{
				try
				{
					currentInstance = configurationElement.createExecutableExtension("class");
					extensionClass = currentInstance.getClass();
				}
				catch (Exception e)
				{
					CommonPlugin.logError(e);
				}
			}
		}
			
		return extensionClass;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#createImplementationClassInstance()
	 */
	public Object createImplementationClassInstance()
	{
		Class cls = getImplementationClass();
		if (currentInstance != null) {
			return currentInstance;
		}
		if(cls != null)
		{
			try
			{
				return cls.newInstance();
			}
			catch (Exception e)
			{
				CommonPlugin.logError(e);
			}
		}
			
		return null;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#getId()
	 */
	public String getId()
	{
		return id;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#setId(java.lang.String)
	 */
	public void setId(String id)
	{
		this.id = id;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#setExtension(java.lang.String)
	 */
	public void setExtension(String fileExtension)
	{
		this.extension = fileExtension;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#getExtension()
	 */
	public String getExtension()
	{
		return extension;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#applyTo(java.lang.String)
	 */
	public boolean applyTo(String type)
	{
		if(applyToAllTypes())
			return true;
			
		return types.contains(type);
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#applyToAllTypes()
	 */
	public boolean applyToAllTypes()
	{
		return types == null;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#addType(String)
	 */
	public boolean addType(String type)
	{
		if(type == null)
			return false;
			
		if(types == null)
			types = new HashSet();

		if(types.add(type))
		{
			if(!associationMapping.register(type, this))
			{
				types.remove(type);
				return false;
			}
			
			if(types.size() == 1)
				associationMapping.deregister(this);
		}
		
		return true;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#removeType(java.lang.String)
	 */
	public boolean removeType(String type)
	{
		if(type == null)
			return false;
			
		if(types == null)
			return false;
		
		if(types.remove(type))
		{
			if(types.isEmpty())
				types = null;
			
			associationMapping.deregister(type, this);
			if(applyToAllTypes())
				associationMapping.register(this);
		}
		
		return true;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#types()
	 */
	public String[] types()
	{
		if(types == null)
			return new String[0];
			
		return (String[])types.toArray(new String[types.size()]);
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationDescriptor#copy()
	 */
	public IAssociationDescriptor copy()
	{
		AssociationDescriptor associationDescriptor = createClone();
		setAttributesOf(associationDescriptor);
		return associationDescriptor;
	}
		
	/**
	 * Creates the clone instance.  Subclasses should overwrite this method to properly
	 * support the clone feature.
	 * @return ExtensionDefinition
	 */
	protected AssociationDescriptor createClone()
	{
		return new AssociationDescriptor(null);
	}

	/**
	 * Copies this instance's attributes to the object specified in 
	 * the argument. Subclasses may overwrite this method to extend the cloning
	 * mechanism.
	 * @param extensionDefinition
	 */
	protected void setAttributesOf(AssociationDescriptor associationDescriptor)
	{
		associationDescriptor.name = name;
		associationDescriptor.description = description;
		associationDescriptor.id = id;
		associationDescriptor.extension = extension;
		associationDescriptor.configurationElement = configurationElement;
		associationDescriptor.extensionClass = extensionClass;
		associationDescriptor.imageKey = imageKey;
		
		if(types == null)
			associationDescriptor.types = null; 
		else
			associationDescriptor.types = new HashSet(types);
	}
	
	/**
	 * Loads the attributes of this descriptor from a given configuration
	 * element.
	 * @param configurationElement
	 */
	protected void loadFromConfigurationElement(IConfigurationElement configurationElement)
	{
		this.configurationElement = configurationElement;
		
		id = getValue(configurationElement, "id");
		name = getValue(configurationElement, "name");
		extension = getValue(configurationElement, "extension");
		description = getValue(configurationElement, "description");
		imageKey = getValue(configurationElement, "icon");
		
		String defaultValue = getValue(configurationElement, "isDefault");
		isPluginDefault = defaultValue != null && defaultValue.equalsIgnoreCase("true");
		
		String type = getValue(configurationElement, "type");
		if(type != null)
		{
			addType(type);
		}
		else
		{
			IConfigurationElement[] elements = configurationElement.getChildren("appliesTo");
			for(int i = 0, max = elements.length; i < max; i++)
				addType(getValue(elements[i], "type"));
		}
		
		/*
		 * Look for extension bindings. These allow you to link
		 * existing extensions to existing types. For example,
		 * if you wish to use an existing viewer for your new type.
		 */
		if (id != null) {
			IExtensionRegistry registry = Platform.getExtensionRegistry();
			IConfigurationElement[] elements = registry.getConfigurationElementsFor("org.eclipse.hyades.ui.extensionBinding");
			for (int i=0;i<elements.length;++i) {
				String extensionIdField = elements[i].getAttribute("extensionId");
				String typeField = elements[i].getAttribute("type");
				if (extensionIdField != null && typeField != null && id.equals(extensionIdField)) {
					addType(typeField);
				}
			}
		}
		
		if(applyToAllTypes())
		{
			associationMapping.register(this);	
			
			if(isPluginDefault)
			{
				associationMapping.setDefaultAssociationDescriptor(this);		
			}
		}
		else
		{
			if(isPluginDefault)
			{
				String[] types = types();
				for(int idx=0; idx<types.length; idx++)
				{
					associationMapping.setDefaultAssociationDescriptor(types[idx], this);	
				}
			}			
		}
			
	}
	
	/**
	 * Returns a value identified by a name in a given <code>IConfigurationElement</code>.  
	 * This value can correspond to an XML attribute or to a child element.  In the 
	 * last case there should be only one element with the given name.   
	 * @param configurationElement
	 * @param name
	 * @return String
	 */
	protected String getValue(IConfigurationElement configurationElement, String name)
	{
		String value = configurationElement.getAttribute(name);
		if(value == null)
		{
			IConfigurationElement[] childElements = configurationElement.getChildren(name);
			if(childElements.length == 1)
				value = childElements[0].getValue();
		}
		
		return value;
	}
	/**
	 * @return
	 */
	public boolean isPluginDefault() {
		return isPluginDefault;
	}

	/**
	 * @param value
	 */
	public void setPluginDefault(boolean value) {
		isPluginDefault = value;
	}

}
