/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AssociationDescriptorEntry.java,v 1.3 2008/01/24 02:29:08 apnan Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.extension;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.util.IDisposable;

/**
 * Entry for the association mappings.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class AssociationDescriptorEntry
implements IDisposable
{
	protected String type;
	protected Set typedAssociationDescriptors;
	protected List commonAssociationDescriptors;
	protected Set avoidedAssociationDescriptors;
	protected IAssociationDescriptor defaultAssociationDescriptor;
	
	/**
	 * Constructor for AssociationDescriptorEntry
	 * @param type
	 */
	public AssociationDescriptorEntry(List commonAssociationDescriptors, String type)
	{
		this.type = type;
		this.commonAssociationDescriptors = commonAssociationDescriptors;
		typedAssociationDescriptors = new HashSet();
		avoidedAssociationDescriptors = new HashSet();
	}
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		commonAssociationDescriptors = null;
		defaultAssociationDescriptor = null;
		avoidedAssociationDescriptors.clear();
		// TODO [refactor] non-UI class shouldn't extens IDisposable. need new mechnism.
		//CoreUtil.dispose(typedAssociationDescriptors);
	}

	/**
	 * Returns this entry's type.
	 * @return
	 */
	public String getType()
	{
		return type;
	}
	
	/**
	 * Adds an association descriptor to this entry.
	 * @param associationDescriptor
	 * @return <code>true</code> if the association descriptor was added.
	 */
	public boolean addTypedAssociationDescriptor(IAssociationDescriptor associationDescriptor)
	{
		if(associationDescriptor == null)
			return false;

		avoidedAssociationDescriptors.remove(associationDescriptor);		
		return typedAssociationDescriptors.add(associationDescriptor);
	}
	
	/**
	 * Removes an association descriptor of this entry.
	 * @param associationDescriptor
	 * @return <code>true</code> if the association descriptor was removed.
	 */		
	public boolean removeTypedAssociationDescriptor(IAssociationDescriptor associationDescriptor)
	{
		if(associationDescriptor == null)
			return false;
			
		avoidedAssociationDescriptors.remove(associationDescriptor);
		typedAssociationDescriptors.remove(associationDescriptor);
		if(defaultAssociationDescriptor == associationDescriptor)
			defaultAssociationDescriptor = null;
			
		return true;
	}

	/**
	 * Removes all the association descriptors of this entry.
	 */		
	public void removeAllTypedAssociationDescriptor()
	{
		if(typedAssociationDescriptors.contains(defaultAssociationDescriptor))
			defaultAssociationDescriptor = null;
			
		avoidedAssociationDescriptors.clear();
		typedAssociationDescriptors.clear();
	}

	/**
	 * Returns an unmodifiable set with the association descriptors of this entry.
	 * @return Set
	 */
	protected Set getTypedAssociationDescriptor()
	{
		return Collections.unmodifiableSet(typedAssociationDescriptors);
	}

	/**
	 * Returns all the association descriptors that apply to this entry's
	 * type.  The default descriptor is always the 1st element of the array.
	 * @return IAssociationDescriptor[]
	 */
	public IAssociationDescriptor[] getAssociationDescriptors()
	{
		ArrayList arrayList = new ArrayList(typedAssociationDescriptors);
		arrayList.addAll(commonAssociationDescriptors);
		if(arrayList.isEmpty())
			return new IAssociationDescriptor[0];
		arrayList.removeAll(avoidedAssociationDescriptors);
		
		IAssociationDescriptor associationDescriptor = getDefault();
		if(associationDescriptor != null)
		{
			arrayList.remove(associationDescriptor);
			arrayList.add(0, associationDescriptor);
		}
	
		return (IAssociationDescriptor[])arrayList.toArray(new IAssociationDescriptor[arrayList.size()]);
	}
	
	/**
	 * Sets the default association descriptor for this entry's
	 * type.
	 * @return IAssociationDescriptor
	 */
	public void setDefault(IAssociationDescriptor associationDescriptor)
	{
		if(associationDescriptor.applyTo(type) && !avoidedAssociationDescriptors.contains(associationDescriptor))
			defaultAssociationDescriptor = associationDescriptor;
	}
	
	/**
	 * Returns the default association descriptor for this entry's
	 * type.
	 * @return IAssociationDescriptor
	 */
	public IAssociationDescriptor getDefault()
	{
		if((defaultAssociationDescriptor == null) || (!defaultAssociationDescriptor.applyTo(type)) || avoidedAssociationDescriptors.contains(defaultAssociationDescriptor))
		{
			defaultAssociationDescriptor = null;
			for (Iterator i = typedAssociationDescriptors.iterator(); i.hasNext();)
			{
				IAssociationDescriptor descriptor = (IAssociationDescriptor)i.next();
				if(!avoidedAssociationDescriptors.contains(descriptor))
				{
					defaultAssociationDescriptor = descriptor;
					return descriptor;
				}
			}

			for (Iterator i = commonAssociationDescriptors.iterator(); i.hasNext();)
			{
				IAssociationDescriptor descriptor = (IAssociationDescriptor)i.next();
				if(!avoidedAssociationDescriptors.contains(descriptor))
				{
					defaultAssociationDescriptor = descriptor;
					return descriptor;
				}
			}
		}

		return defaultAssociationDescriptor;
	}
	
	/**
	 * Adds an association descriptor to the set of descriptors that should not
	 * be used with this entry's type.
	 * @param associationDescriptor
	 * @return <code>true</code> if the <code>associationDescriptor</code> was added 
	 * or <code>false</code> otherwise.
	 */
	public boolean addToAvoidedSet(IAssociationDescriptor associationDescriptor)
	{
		if((associationDescriptor == null) || (!associationDescriptor.applyTo(type)))
			return false;
			
		if(avoidedAssociationDescriptors.add(associationDescriptor))
		{
			if(defaultAssociationDescriptor == associationDescriptor)
				defaultAssociationDescriptor = null;
				
			return true;
		}
		
		return false;
	}

	/**
	 * Removes an association descriptor from the set of descriptors that should not
	 * be used with this entry's type.
	 * @param associationDescriptor
	 * @return <code>true</code> if the <code>associationDescriptor</code> was removed 
	 * or <code>false</code> otherwise.
	 */
	public boolean removeFromAvoidedSet(IAssociationDescriptor associationDescriptor)
	{
		if((associationDescriptor == null) || (!associationDescriptor.applyTo(type)))
			return false;
			
		return avoidedAssociationDescriptors.remove(associationDescriptor);
	}

	/**
	 * Removes all the association descriptors from the set of descriptors that 
	 * should not be used with this entry's type.
	 */
	public void removeAllFromAvoidedSet()
	{
		avoidedAssociationDescriptors.clear();
	}
	
	/**
	 * Returns an unmodifiable set with the association descriptors to be avoided.
	 * @return Set
	 */
	public Set avoidedAssociationDescriptors()
	{
		return Collections.unmodifiableSet(avoidedAssociationDescriptors);
	}
}