/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AssociationMappingRegistry.java,v 1.6 2009/06/03 14:52:46 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.extension;

import java.util.Map;
import java.util.TreeMap;

import org.eclipse.core.runtime.ListenerList;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationDescriptorFilter;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.hyades.ui.extension.IAssociationMappingRegistry;
import org.eclipse.hyades.ui.internal.util.XMLUtil;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIMessages;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;
import org.eclipse.ui.IPropertyListener;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * <p>Implementation of the {@link org.eclipse.hyades.ui.extension.IAssociationMappingRegistry}
 * interface.</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Jerome Bozier
 * @author  Paul Slauenwhite
 * @version June 3, 2009
 * @since   August 16, 2006
 */
public class AssociationMappingRegistry implements IAssociationMappingRegistry {
	
	protected static final String PREFERENCE_PREFIX_KEY = CommonUIPlugin.PLUGIN_ID + ".AssociationMappingRegistry."; 
	
	protected Map associationMappingByExtensionPoint;
	
	protected IAssociationDescriptorFilter filter;
	protected String objectCategory;
	protected IPreferenceStore preferenceStore;
	protected ImageRegistry imageRegistry;
	protected ListenerList commitChangeListeners;
	protected String xmlBuffer; 
	
	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#initialize(String, IAssociationDescriptorFilter, IPreferenceStore)
	 */
	public void initialize(String objectCategory, IAssociationDescriptorFilter filter, IPreferenceStore preferenceStore)
	throws IllegalArgumentException
	{
		if(objectCategory == null)
			throw new IllegalArgumentException(NLS.bind(CommonUIMessages._9, "objectCategory"));
			
		this.objectCategory = objectCategory;
		this.filter = filter;
		this.preferenceStore = preferenceStore;
		
		load(preferenceStore);
	}

	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		//TODO [refactor] non-UI class shouldn't extens IDisposable. need new mechnism.
		//CoreUtil.dispose(associationMappingByExtensionPoint);
		if(commitChangeListeners != null)
			commitChangeListeners.clear();
		
		filter = null;
		preferenceStore = null;
		xmlBuffer = null;
		imageRegistry = null;		
	}

	/**
	 * Resolves the image registry used to store the image
	 * of the mappings.
	 * 
	 * @return The image registry used to store the image of the mappings.
	 */
	public ImageRegistry getImageRegistry(){
		
		if(imageRegistry == null){
			
			boolean isDisplayCreated = false;
			
			//Wait for a display to be created on this thread to a maximum of 1000 ms (or 1 s) to avoid a race condition when Eclipse is starting:
			for (int counter = 0; counter < 10; counter++) {
			
				Display currentDisplay = Display.getCurrent();
				
				if((currentDisplay != null) && (!currentDisplay.isDisposed()) && (currentDisplay == Display.getDefault())) {
					
					isDisplayCreated = true;
					
					break;
				}
				else{
					
					//Sleep for 100 ms:
					try {
						Thread.sleep(100);
					} 
					catch (InterruptedException e) {
						//Ignore.
					}
				}
			}
			
			//Only create the image registry if the display is created:
			if(isDisplayCreated){

				try{
					imageRegistry = new ImageRegistry();
				}
				catch(RuntimeException e){
					CommonPlugin.logError(e);
				}
			}
		}
			
		return imageRegistry;
	}
	
	/**
	 * Sets the image registry that is used to store the mappings'
	 * images.
	 * @param imageRegistry
	 */
	public void setImageRegistry(ImageRegistry imageRegistry)
	{
		this.imageRegistry = imageRegistry;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#getObjectCategory()
	 */
	public String getObjectCategory()
	{
		return objectCategory;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#getPreferenceStore()
	 */
	public IPreferenceStore getPreferenceStore()
	{
		return preferenceStore;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#getFilter()
	 */
	public IAssociationDescriptorFilter getFilter()
	{
		return filter;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#retrieveAssociationMapping(String, String)
	 */
	public synchronized IAssociationMapping retrieveAssociationMapping(String extensionPoint, String pluginId)
	throws IllegalArgumentException
	{
		if(extensionPoint == null)
			throw new IllegalArgumentException(NLS.bind(CommonUIMessages._9, "extensionPoint"));
			
		AssociationMapping associationMapping = null;
		if(associationMappingByExtensionPoint == null)
			associationMappingByExtensionPoint = new TreeMap();
		else
			associationMapping = (AssociationMapping)associationMappingByExtensionPoint.get(extensionPoint);

		if(associationMapping == null)
		{
			associationMapping = new AssociationMapping(this, extensionPoint, pluginId);
			associationMappingByExtensionPoint.put(extensionPoint, associationMapping);
		}
		
		return associationMapping;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#getAssociationMapping(java.lang.String)
	 */
 	public synchronized IAssociationMapping getAssociationMapping(String extensionPoint)
	{
		if((extensionPoint == null) || (associationMappingByExtensionPoint == null))
			return null;
			
		return (AssociationMapping)associationMappingByExtensionPoint.get(extensionPoint);
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#getAssociationMappings()
	 */
	public IAssociationMapping[] getAssociationMappings()
	{
		if(associationMappingByExtensionPoint == null)
			return new IAssociationMapping[0];
			
		return (AssociationMapping[])associationMappingByExtensionPoint.values().toArray(new AssociationMapping[associationMappingByExtensionPoint.values().size()]);
	}
	
	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#checkpoint()
	 */
	public synchronized boolean checkpoint()
	{
		xmlBuffer = write();
		return true;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#clearCheckpoint()
	 */
	public synchronized void clearCheckpoint()
	{
		xmlBuffer = null;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#commit()
	 */
	public synchronized boolean commit()
	{
		checkpoint();
		if(write(preferenceStore))
			notifyCommitChangeListeners();
			
		return true;
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#rollback()
	 */
	public synchronized boolean rollback()
	{
		if(xmlBuffer != null)
			load(xmlBuffer);
		return true;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#addCommitChangeListener(org.eclipse.ui.IPropertyListener)
	 */
	public void addCommitChangeListener(IPropertyListener propertyListener)
	{
		if(commitChangeListeners == null)
			commitChangeListeners = new ListenerList();
		commitChangeListeners.add(propertyListener);
	}

	/**
	 * @see org.eclipse.hyades.ui.extension.IAssociationMappingRegistry#removeCommitChangeListener(org.eclipse.ui.IPropertyListener)
	 */
	public void removeCommitChangeListener(IPropertyListener propertyListener)
	{
		if(commitChangeListeners != null)
			commitChangeListeners.remove(propertyListener);
	}
	
	/**
	 * Notify all the commit listeners.  This implementation ensures that
	 * an exception thrown by a listener doesn't affect the notification
	 * process.
	 */
	protected void notifyCommitChangeListeners()
	{
		if(commitChangeListeners == null)
			return;
			
		Object[] listeners = commitChangeListeners.getListeners();
		for(int i = 0, maxi = listeners.length; i < maxi; i++)
		{
			IPropertyListener propertyListener = (IPropertyListener)listeners[i];
			try
			{
				propertyListener.propertyChanged(this, PROP_CONTENTS);	
			}
			catch(RuntimeException e)
			{
			}
		}
	}

	/**
	 * Returns the preference key to retrieve this registry settings.
	 * @return String
	 */
	protected String getPreferenceKey()
	{
		StringBuffer key = new StringBuffer(PREFERENCE_PREFIX_KEY);
		key.append(objectCategory);
		return key.toString();
	}
	
	/**
	 * Reads this registry from a preference store.
	 * @param preferenceStore
	 * @return <code>true</code> if the registry was read or 
	 * <code>false</code> otherwise.
	 */
	protected boolean load(IPreferenceStore preferenceStore)
	{
		if(preferenceStore == null)
			return false;
			
		String value = preferenceStore.getString(getPreferenceKey());
		if((value == null) || (value.length() == 0))
			return false;
			
		return load(value);
	}
	
	/**
	 * Reads this registry from a xml string.
	 * @param xml
	 * @return <code>true</code> if the registry was read or 
	 * <code>false</code> otherwise.
	 */
	protected boolean load(String xml)
	{
		XMLUtil.setLogger(CommonPlugin.getLogger());
		Element registryElement = XMLUtil.loadDom(xml, "AssociationMappingRegistry");
		if(registryElement == null)
			return false;
			
		NodeList nodeList = XMLUtil.getChildrenByName(registryElement, "AssociationMapping");
		for(int i=0, maxi=nodeList.getLength(); i<maxi; i++)
		{
			if(nodeList.item(i) instanceof Element)
				loadMapping((Element)nodeList.item(i));
		}
		
		return true;
	}
	
	/**
	 * Writes this registry to a preference store.
	 * @param preferenceStore
	 * @return <code>true</code> if the registry was written to the prefernce or 
	 * <code>false</code> otherwise.  If the information to be stored is already
	 * in the preference store this method returns false.
	 */
	protected boolean write(IPreferenceStore preferenceStore)
	{
		if(preferenceStore == null)
			return false;
		
		String xmlString = write();
		if(xmlString.equals(preferenceStore.getString(getPreferenceKey())))
			return false;
			
		preferenceStore.setValue(getPreferenceKey(), xmlString);
		return true;
	}
	
	/**
	 * Writes this registry to a xml string.
	 * @return <code>true</code> if the registry was written or <code>false</code> 
	 * otherwise.
	 */
	protected String write()
	{
		StringBuffer xml = new StringBuffer();
		
		//Header
		xml.append("<?xml");
		xml.append(XMLUtil.createXMLAttribute("version","1.0", false)).append(XMLUtil.createXMLAttribute("encoding","UTF-8", false));
		xml.append("?>");
		
		//Body
		xml.append("<AssociationMappingRegistry>");
		IAssociationMapping[] mappings = getAssociationMappings();
		for(int i = 0, maxi = mappings.length; i < maxi; i++)
			writeMapping(xml, mappings[i]);
		xml.append("</AssociationMappingRegistry>");
		
		return xml.toString();
	}
	
	/**
	 * Loads the preference setting located in in XML element.
	 * @param mappingElement
	 */
	protected void loadMapping(Element mappingElement)
	{
		String extensionPoint = XMLUtil.getValue(mappingElement, "extensionPoint");
		if(extensionPoint == null)
			return;
		String pluginId = XMLUtil.getValue(mappingElement, "pluginId");
			
		IAssociationMapping mapping = retrieveAssociationMapping(extensionPoint, pluginId);
		
		String defaultId = XMLUtil.getValue(mappingElement, "defaultId");
		if(defaultId != null)
			mapping.setDefaultAssociationDescriptor(mapping.getAssociationDescriptor(defaultId));
			
		NodeList typeDetailNodeList = XMLUtil.getChildrenByName(mappingElement, "TypeDetail");
		for(int i=0, maxi=typeDetailNodeList.getLength(); i<maxi; i++)
		{
			if(!(typeDetailNodeList.item(i) instanceof Element))
				continue;
				
			Element typeDetailElement = (Element)typeDetailNodeList.item(i);
			String type = XMLUtil.getValue(typeDetailElement, "type");
			if(type == null)
				continue;
			
			defaultId = XMLUtil.getValue(typeDetailElement, "defaultId");
			if(defaultId != null)
				mapping.setDefaultAssociationDescriptor(type, mapping.getAssociationDescriptor(type, defaultId));
			
			mapping.removeAllFromAvoidSet(type);
			NodeList avoidedDescriptorNodeList = XMLUtil.getChildrenByName(typeDetailElement, "AvoidedDescriptor");
			for(int j=0, maxj=avoidedDescriptorNodeList.getLength(); j<maxj; j++)
			{
				if(!(avoidedDescriptorNodeList.item(j) instanceof Element))
					continue;
				
				Element avoidedDescriptorElement = (Element)avoidedDescriptorNodeList.item(j);
				String id = XMLUtil.getValue(avoidedDescriptorElement, "id");
				if(id != null)
					mapping.addToAvoidedSet(type, mapping.getAssociationDescriptor(type, id));
			}
		}
	}

	/**
	 * Writes the preferences of the specified mapping in an XML format.  This method
	 * is <b>NOT</b> writing the mapping itself.  It is concerned only with the 
	 * preference settings.
	 * @param xml
	 * @param mapping
	 */
	protected void writeMapping(StringBuffer xml, IAssociationMapping mapping)
	{
		StringBuffer tempXML = new StringBuffer();
		tempXML.append("<AssociationMapping");
		tempXML.append(XMLUtil.createXMLAttribute("extensionPoint",mapping.getExtensionPoint(), false));
		tempXML.append(XMLUtil.createXMLAttribute("pluginId",mapping.getPluginId(), false));

		IAssociationDescriptor defaultDescriptor = mapping.getDefaultAssociationDescriptor();
		if(defaultDescriptor != null)
			tempXML.append(XMLUtil.createXMLAttribute("defaultId", defaultDescriptor.getId(), false));

		tempXML.append(">");
			
			
		boolean hasContent = false;
		String[] types = mapping.getTypes();
		for(int i = 0, maxi = types.length; i < maxi; i++)
		{
			defaultDescriptor = mapping.getDefaultAssociationDescriptor(types[i]);
			IAssociationDescriptor[] avoidedDescriptors = mapping.getAvoidedAssociationDescriptors(types[i]);
			
			if(((defaultDescriptor == null) || (defaultDescriptor.getId() == null)) && (avoidedDescriptors.length == 0))
				continue;
				
			hasContent = true;
			tempXML.append("<TypeDetail");
			tempXML.append(XMLUtil.createXMLAttribute("type", types[i], false));
			if(defaultDescriptor != null)
				tempXML.append(XMLUtil.createXMLAttribute("defaultId", defaultDescriptor.getId(), false));
			tempXML.append(">");
			for(int j = 0, maxj = avoidedDescriptors.length; j < maxj; j++)
			{
				if(avoidedDescriptors[j].getId() == null)
					continue;
					
				tempXML.append("<AvoidedDescriptor");
				tempXML.append(XMLUtil.createXMLAttribute("id", avoidedDescriptors[j].getId(), false));
				tempXML.append("/>");
			}
			tempXML.append("</TypeDetail>");					
		}
		tempXML.append("</AssociationMapping>");
		
		if(hasContent)
			xml.append(tempXML);
	}
}
