 /**********************************************************************
 * Copyright (c) 2007, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: NavigatorActionExtensionUtil.java,v 1.3 2008/01/24 02:29:08 apnan Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.ui.internal.extension;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.ui.provisional.navigator.action.INavigatorActionContributor;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIMessages;

/**
 * Utility class for retrieving navigator action contributions.<br>
 * See extension point : <code>org.eclipse.tptp.platform.common.ui.NavigatorActions</code>
 * 
 * @author Eugene Chan
 * @author Alex Nan
 * @since 4.4
 */
public class NavigatorActionExtensionUtil {

	/**
	 * Extension point for navigator action contributions
	 */
	protected static final String EXT_PT_NAVIGATOR_ACTIONS = "org.eclipse.tptp.platform.common.ui.NavigatorActions";
		
	/**
	 * Defined group values
	 */
	public static final String GP_IMPORT_LOG = "import.log";
	public static final String GP_EXPORT_LOG = "export.log";
	public static final String GP_IMPORT_SDB = "import.sdb";
	public static final String GP_EXPORT_SDB = "export.sdb";
	public static final String GP_ANALYSIS_LOG = "analysis.log";
	public static final String GP_SDB_EDITOR = "editor.sdb";
	public static final String GP_GENERIC = "generic";
	
	public static Set groups = null;
	
	protected NavigatorAction[] contributors;
	protected Map groupMap = new HashMap();
	
	public class NavigatorAction {
		
		protected String id;
		protected String group;
		protected INavigatorActionContributor actionContributor;
		
		public String getId() {
			return id;
		}
		public void setId(String id) {
			this.id = id;
		}
		public String getGroup() {
			return group;
		}
		public void setGroup(String group) {
			this.group = group;
		}
		public INavigatorActionContributor getActionContributor() {
			return actionContributor;
		}
		public void setActionContributor(INavigatorActionContributor actionContributor) {
			this.actionContributor = actionContributor;
		}
	}
	
	protected void loadContributors()
	{
		if(groups == null){
			initGroups();
		}
		if (contributors == null)
		{
			ArrayList contributorList = new ArrayList();
			IExtensionPoint extensionPt = Platform.getExtensionRegistry().getExtensionPoint(EXT_PT_NAVIGATOR_ACTIONS);
			IExtension[] extensions = extensionPt == null ? new IExtension[0] : extensionPt.getExtensions();
			IConfigurationElement[] configElements = null;
			int l = extensions.length;
			int cel = 0;
			for (int i = 0; i < l; i++)
			{
				configElements = extensions[i].getConfigurationElements();
				cel = configElements.length;
				if(configElements == null || cel == 0){
					continue;
				}
				for(int j = 0; j < cel; j++){
					try
					{
						NavigatorAction adopter = new NavigatorAction();
						adopter.setId(configElements[j].getAttribute("id"));
						
						String group = configElements[j].getAttribute("group");
						//validate group value or set to GENERIC group if none is supplied
						adopter.setGroup(isValidGroup(group)? group : GP_GENERIC);
						
						Object execExt = configElements[j].createExecutableExtension("class");
						if(execExt instanceof INavigatorActionContributor){
							adopter.setActionContributor((INavigatorActionContributor)execExt);	
						}else {
							CommonPlugin.logError(new Exception(CommonUIMessages.invalidNavExtAction));
							continue;
						}
						
						contributorList.add(adopter);
						
						//add contribution to the group map  
						Object list = groupMap.get(group);
						ArrayList alist;
						if (list==null) {
							alist = new ArrayList();						
						}
						else {
							alist = (ArrayList)list;
						}
						alist.add(adopter);
						groupMap.put(group, alist);
						
					}
					catch (Throwable t)
					{
						CommonPlugin.logError(t);
					}
				}
			}
			
			contributors = (NavigatorAction[])contributorList.toArray(new NavigatorAction[contributorList.size()]);
		}
	}
	/**
	 * Initialize action group set.
	 */
	protected void initGroups(){
		groups = new HashSet();
		groups.add(GP_IMPORT_LOG);
		groups.add(GP_EXPORT_LOG);
		groups.add(GP_IMPORT_SDB);
		groups.add(GP_EXPORT_SDB);		
		groups.add(GP_ANALYSIS_LOG);
		groups.add(GP_SDB_EDITOR);
		groups.add(GP_GENERIC);			
	}
	/**
	 * Validate group string among the known group values.
	 * @param group group name
	 * @return
	 */
	protected boolean isValidGroup(String group) {
		if (group==null) return false;
		
		return groups.contains(group);
	}
	
	/**
	 * Retrieve the list of contributors of a certain group
	 * @param group group name
	 * @return
	 */
	public List getGroup(String group){
		if (contributors==null) loadContributors();
		return (List)groupMap.get(group);
	}
	
	/**
	 * Retrieve a list of contribution of extension point
	 * @return
	 */
	public NavigatorAction[] getContributions() {
		if (contributors==null) loadContributors();
		return contributors;
	}
}
