/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: NavigatorExtensionUtil.java,v 1.4 2008/01/24 02:29:08 apnan Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/

package org.eclipse.hyades.ui.internal.extension;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.ui.extension.INavigatorContribution;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIConstants;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;

/**
 * An internal utility class to facilitate dealing with navigator extensions.
 * Allows getting all the extensions, as well as various convenience methods.
 * 
 * @author curtispd
 * @since 3.0
 */
public class NavigatorExtensionUtil
{
	protected static final String ALL = "all";

	protected static final Map _contributions = new HashMap();

	/**
	 * Delegates the filling of the context menu to all navigator extensions.
	 * 
	 * @param menu
	 *            the menu to populate.
	 * @param selection
	 *            the selected items.
	 * @param navigatorID
	 *            the navigator's unique ID
	 */
	public static void fillContextMenu(IMenuManager menu,
			IStructuredSelection selection, String navigatorID)
	{
		INavigatorContribution[] e = NavigatorExtensionUtil
				.getContributions(navigatorID);
		for (int i = 0; i < e.length; ++i)
			e[i].fillContextMenu(menu, selection);
	}

	/**
	 * Returns all navigator contributions for a specific navigator.
	 * 
	 * @param navigatorID
	 *            the navigator's unique ID
	 * @return the contributions for the given navigator.
	 */
	public static INavigatorContribution[] getContributions(String navigatorID)
	{
		INavigatorContribution[] cached = (INavigatorContribution[]) _contributions
				.get(navigatorID == null ? ALL : navigatorID);

		if (cached == null)
		{
			IExtension[] commonUIPluginExtension = Platform.getExtensionRegistry().getExtensionPoint(CommonUIPlugin.PLUGIN_ID, CommonUIPlugin.EP_NAVIGATOR_EXTENSIONS).getExtensions();
			
			/* Ali M.: For backward compatibility */
			IExtension[] exts = null;
			IExtensionPoint extPoint = Platform.getExtensionRegistry().getExtensionPoint(CommonUIConstants.HYADES_UI_PLUGIN_ID, CommonUIPlugin.EP_NAVIGATOR_EXTENSIONS);
			if (extPoint != null)
			{
				IExtension[] hyadesUIPluginExtension = extPoint.getExtensions();
				if (hyadesUIPluginExtension != null && hyadesUIPluginExtension.length > 0)
				{
					exts = new IExtension[commonUIPluginExtension.length + hyadesUIPluginExtension.length];
					System.arraycopy(commonUIPluginExtension, 0, exts, 0, commonUIPluginExtension.length);
					System.arraycopy(hyadesUIPluginExtension, 0, exts, commonUIPluginExtension.length, hyadesUIPluginExtension.length);					
				}
			}
			if (exts == null)
				exts = commonUIPluginExtension;
			
			List list = new ArrayList();
			for (int i = 0; i < exts.length; ++i)
			{
				IConfigurationElement[] elements = exts[i]
						.getConfigurationElements();
				for (int j = 0; j < elements.length; ++j)
				{
					if ("navigatorContribution".equals(elements[j].getName()))
					{
						String nav = elements[j].getAttribute("navigatorID");
						if (navigatorID == null || navigatorID.equals(nav))
						{
							try
							{
								String className = elements[j]
										.getAttribute("class");
								Class classObj = Platform.getBundle(
										exts[i].getNamespace()).loadClass(
										className);
								list.add(classObj.newInstance());
							} catch (Exception e)
							{
								CommonPlugin.logError(e);
							}
						}
					}
				}
			}
			cached = new INavigatorContribution[list.size()];
			list.toArray(cached);
			_contributions.put(navigatorID == null ? ALL : navigatorID, cached);
		}
		return cached;
	}

	/**
	 * Returns all children contributed by all extensions, for the specific
	 * <code>Object</code>.
	 * 
	 * @param item
	 *            the item whose children we want.
	 * @param navigatorID
	 *            the navigator's unique ID
	 * @return the item's children.
	 */
	public static List getAllChildren(Object item, String navigatorID)
	{
		INavigatorContribution[] e = NavigatorExtensionUtil
				.getContributions(navigatorID);
		List list = new ArrayList();
		for (int i = 0; i < e.length; ++i)
			list.addAll(e[i].getChildren(item));
		return list;
	}

	/**
	 * Returns <code>true</code> if the item has any children contributed by
	 * extensions.
	 * 
	 * @param item
	 *            the parent item.
	 * @param navigatorID
	 *            the navigator's unique ID
	 * @return true if there are any children.
	 */
	public static boolean hasChildren(Object item, String navigatorID)
	{
		INavigatorContribution[] e = NavigatorExtensionUtil
				.getContributions(navigatorID);
		for (int i = 0; i < e.length; ++i)
			if (e[i].hasChildren(item))
				return true;
		return false;
	}

	/**
	 * Returns all <code>INavigatorItem</code>s that need to be saved. This
	 * is used to determine which items should be saved when the workbench is
	 * shut down.
	 * 
	 * @param navigatorID
	 *            the navigator's unique ID
	 * @return all external items that need to be saved.
	 */
	public static Collection getAllModifiedItems(String navigatorID)
	{
		INavigatorContribution[] e = NavigatorExtensionUtil
				.getContributions(navigatorID);
		Collection c = new HashSet();
		for (int i = 0; i < e.length; ++i)
			c.addAll(e[i].getModifiedItems());
		return c;
	}
	
	
	/**
	 * Disposes all navigator extensions.
	 */
	public static void disposeAll() {
		Iterator iter = _contributions.values().iterator();
		while (iter.hasNext()) {
			INavigatorContribution[] e = (INavigatorContribution[])iter.next();
			for (int i=0;i<e.length;++i) {
				e[i].dispose();
				e[i] = null;
			}
		}
		_contributions.clear();
	}
}
