/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: BasicFeatureFilterAttribute.java,v 1.3 2007/11/27 04:52:11 apnan Exp $
 *
 * Contributors: 
 * IBM - Initial API and implementation
 ***********************************************************************/

package org.eclipse.hyades.ui.internal.filters;

import java.util.Vector;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.hierarchy.extensions.BinaryExpression;
import org.eclipse.hyades.models.hierarchy.extensions.ExtensionsFactory;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalExpression;
import org.eclipse.hyades.models.hierarchy.extensions.LogicalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.RelationalOperators;
import org.eclipse.hyades.models.hierarchy.extensions.SimpleOperand;
import org.eclipse.hyades.models.hierarchy.extensions.WhereExpression;
import org.eclipse.hyades.ui.filters.IFilterAttribute;
import org.eclipse.hyades.ui.util.ICommonUIHelper;
import org.eclipse.osgi.util.NLS;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIMessages;

/**
 * Constructs a basic filter attribute for the advanved tab for view filters.
 * 
 * Values entered can be separated with '|' characters, to mean OR.  So Package_name() may be the
 * feature given with relational operators = and like, these will displayed.  A user could then enter
 * [feature] [=] [A | B] and this will handle it as feature == A || feature == B. 
 *
 */
public class BasicFeatureFilterAttribute implements IFilterAttribute {

	public static String OR_SEPARATOR = "|";
	
	protected String _id;
	protected String _displayText;
	protected RelationalOperators[] _operators; 
	
	protected EStructuralFeature _feature;
	
	protected boolean _useNotLikeOperator;
	
	/**
	 * The contstructor, initializes an object of this type.
	 * @param id String The unique id for the advanced tab associated with this attribute.
	 * @param displayText String The text displayed in the attribute tab.
	 * @param feature EStructuralFeature The feature that we filter by in this attribute.
	 * @param operators RelationalOperators[] The list of relational operators that will be displayed in the advanced tab.
	 * @param useNotLikeOperator Whether to show a "Not like" operator or not.
	 */
	public BasicFeatureFilterAttribute(String id, String displayText, EStructuralFeature feature, RelationalOperators[] operators, boolean useNotLikeOperator)
	{
		_id = id;
		_displayText = displayText;
		_operators = operators;
		
		_feature = feature;
		_useNotLikeOperator = useNotLikeOperator;
	}
	
	public EStructuralFeature feature()
	{
		return _feature;
	}
	
	/**
	 * @see IFilterAttribute#id()
	 */
	public String id() {
		return _id;
	}

	/**
	 * @see IFilterAttribute#displayText()
	 */
	public String displayText() {
		return _displayText;
	}

	/**
	 * @see IFilterAttribute#validate(String, String)
	 */	
	public String validate(String operator, String value) {
		String values[] = getValues(value);
		
		for (int i = 0; i < values.length; i++)
		{
			if (_feature instanceof EAttribute)
			{
				try
				{
					EcoreUtil.createFromString(((EAttribute)_feature).getEAttributeType(), values[i]);
				}
				catch (Exception e)
				{
					return NLS.bind(CommonUIMessages._118, new Object[] {values[i], displayText()});
				}
			}
		}
			
		return null;
	}

	/**
	 * @see IFilterAttribute#operators()
	 */	
	public String[] operators() {
		String[] operators = new String[_operators.length + (_useNotLikeOperator?1:0)];
		
		int j = 0;
		
		for (int i = 0; i < _operators.length; i++)
		{
			operators[j++] = getOperatorString(_operators[i]);
			if (RelationalOperators.get(RelationalOperators.LIKE).equals(_operators[i]) && _useNotLikeOperator)
				operators[j++] = CommonUIMessages._93;
		}
		
		if (_useNotLikeOperator && j < operators.length)
			operators[j] = CommonUIMessages._93;			
		
		return operators;
	}
	
	/**
	 * Returns the value from the given entered display value
	 */	
	protected String getValueFromDisplayValue(String value)
	{
		return value;
	}

	protected WhereExpression getSubWhereExpression(EStructuralFeature feature, RelationalOperators operator, String value, boolean notExpression)
	{
		if (notExpression)
		{
			LogicalExpression logicalExpression = ExtensionsFactory.eINSTANCE.createLogicalExpression();
			logicalExpression.setOperator(LogicalOperators.NOT_LITERAL);
			
			logicalExpression.getArguments().add(ICommonUIHelper.INSTANCE.createBinaryExpression(feature, operator, getValueFromDisplayValue(value), false));
			
			return logicalExpression;
		}
		else
			return ICommonUIHelper.INSTANCE.createBinaryExpression(feature, operator, getValueFromDisplayValue(value), false);
	}

	/**
	 * @see IFilterAttribute#getWhereExpression(String, String)
	 */	
	public WhereExpression getWhereExpression(String operator, String value) {
		
		String[] values = getValues(value);
		if (values.length == 1)
		{
			RelationalOperators relationOperator = getOperator(operator);
			
			if (relationOperator != null)
				return getSubWhereExpression(_feature, relationOperator, values[0], false);
			else if (CommonUIMessages._93.equals(operator))
				return getSubWhereExpression(_feature, RelationalOperators.get(RelationalOperators.LIKE), values[0], true);
			else
				return null;
		}
		else
		{
			LogicalExpression logicalExpression = ExtensionsFactory.eINSTANCE.createLogicalExpression();
			logicalExpression.setOperator(LogicalOperators.OR_LITERAL);				
			
			for (int j = 0; j < values.length; j++)
			{
				RelationalOperators relationOperator = getOperator(operator);
				
				if (relationOperator != null)
					logicalExpression.getArguments().add(getSubWhereExpression(_feature, relationOperator, values[j], false));
				else if (CommonUIMessages._93.equals(operator))
					logicalExpression.getArguments().add(getSubWhereExpression(_feature, RelationalOperators.get(RelationalOperators.LIKE), values[j], true));
			}
			
			if (logicalExpression.getArguments().size() > 0)
				return logicalExpression;
			else
				return null;
		}
	}

	/**
	 * Returns the display value from the given value.
	 */		
	protected String getDisplayValue(String value)
	{
		return value;
	}	

	/**
	 * @see IFilterAttribute#getValue(WhereExpression)
	 */			
	public String getValue(WhereExpression whereExpression) {
		if (whereExpression instanceof BinaryExpression)
		{
		 	BinaryExpression be = (BinaryExpression)whereExpression;
		 	return getDisplayValue(((SimpleOperand)be.getRightOperands().get(0)).getRawValue()); 	
		}
		else if (whereExpression instanceof LogicalExpression)
		{
			String values = "";
			EList arguments = ((LogicalExpression)whereExpression).getArguments();
			
			for (int i = 0; i < (arguments.size() - 1); i++)
			 	values = values + getValue((WhereExpression)arguments.get(i)) + " " + OR_SEPARATOR + " ";
			
		 	values = values + getValue((WhereExpression)arguments.get(arguments.size()-1));		 	
		 	
			return values;
		}
		else
			return null;

	}

	/**
	 * @see IFilterAttribute#getOperator(WhereExpression)
	 */			
	public String getOperator(WhereExpression whereExpression) {
		if (whereExpression instanceof BinaryExpression)
		{
		 	BinaryExpression be = (BinaryExpression)whereExpression;
		 	return getOperatorString(be.getOperator());
		}
		else if (whereExpression instanceof LogicalExpression)
		{
			if (LogicalOperators.NOT_LITERAL.equals(((LogicalExpression)whereExpression).getOperator()))
				return CommonUIMessages._93;
			
			EList arguments = ((LogicalExpression)whereExpression).getArguments();
			if (arguments.size() > 0)
				return getOperator((WhereExpression)arguments.get(0));
			else
				return null;
		}
		else
			return null;
	}

	/**
	 * @see IFilterAttribute#defaultValue()
	 */			
	public String defaultValue() {
		return "";
	}
	
	/**
	 * Returns the set of values separated by OR separators ('|').
	 * @param valuesStr
	 * @return
	 */
	protected String[] getValues(String valuesStr)
	{
		String[] values;
		Vector valuesV = new Vector();
		
		int idx = valuesStr.indexOf(OR_SEPARATOR);
		while (idx >= 0)
		{
			valuesV.add(valuesStr.substring(0, idx).trim());
			valuesStr = valuesStr.substring(idx+1);
			idx = valuesStr.indexOf(OR_SEPARATOR);
		}
		valuesV.add(valuesStr.trim());		
		
		values = new String[valuesV.size()];
		valuesV.toArray(values);
		return values;
	}	
	
	/**
	 * Get the RelationalOperator for the given operator String
	 * @param operator
	 * @return
	 */
	public static RelationalOperators getOperator(String operator)
	{
		if (CommonUIMessages._92.equals(operator))
			return RelationalOperators.get(RelationalOperators.LIKE);
		else if (CommonUIMessages._94.equals(operator))
			return RelationalOperators.get(RelationalOperators.EQ);			
		else if (CommonUIMessages._95.equals(operator))
			return RelationalOperators.get(RelationalOperators.NEQ);			
		else if (CommonUIMessages._96.equals(operator))
			return RelationalOperators.get(RelationalOperators.GT);			
		else if (CommonUIMessages._97.equals(operator))
			return RelationalOperators.get(RelationalOperators.LT);			
		else if (CommonUIMessages._98.equals(operator))
			return RelationalOperators.get(RelationalOperators.GE);
		else if (CommonUIMessages._99.equals(operator))
			return RelationalOperators.get(RelationalOperators.LE);
		
		return null;
	}
	
	/**
	 * Get the operator String for the given RelationalOperators
	 * @param operator
	 * @return
	 */	
	public static String getOperatorString(RelationalOperators operator)
	{
		if (RelationalOperators.get(RelationalOperators.LIKE).equals(operator))
			return CommonUIMessages._92;
		else if (RelationalOperators.get(RelationalOperators.EQ).equals(operator))
			return CommonUIMessages._94;			
		else if (RelationalOperators.get(RelationalOperators.NEQ).equals(operator))
			return CommonUIMessages._95;
		else if (RelationalOperators.get(RelationalOperators.GT).equals(operator))
			return CommonUIMessages._96;
		else if (RelationalOperators.get(RelationalOperators.LT).equals(operator))
			return CommonUIMessages._97;
		else if (RelationalOperators.get(RelationalOperators.GE).equals(operator))
			return CommonUIMessages._98;
		else if (RelationalOperators.get(RelationalOperators.LE).equals(operator))
			return CommonUIMessages._99;
		
		return null;
	}	
}
