/*******************************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LogicalFolder.java,v 1.3 2008/01/24 02:28:51 apnan Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.logicalfolder;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.ui.adapter.NamedElementPropertySource;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.hyades.ui.util.INamedElement;
import org.eclipse.osgi.util.NLS;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIMessages;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.IPersistableElement;
import org.eclipse.ui.views.properties.IPropertyDescriptor;
import org.eclipse.ui.views.properties.IPropertySource;
import org.eclipse.ui.views.properties.PropertyDescriptor;



/**
 * The logical folder is a UI element that can be used to group elements 
 * that are presented in, for example, tree viewers.
 * 
 * <p>There are 2 ways of defining the children of a logical folder:<OL>
 * <LI>By accessing {@link #getChildren()} and obtaining a list defined by the logical folder</LI>
 * <LI>By invoking the {@link #linkChildren(List)} method and defining the child list.</LI> 
 * </OL>
 * When the second approach is used, the list returned by {@link #getChildren()} is
 * an unmodifiable list.  Clients should check if the list {@link #isLinked() is linked}
 * before changing it.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class LogicalFolder
extends NamedElementPropertySource implements	Comparable, IDisposable, 
												IAdaptable, INamedElement, IPersistableElement
{
	/**
	 * Element count property constant
	 * @see IPropertySource
	 */
	protected final static String PROP_CHILD_COUNT = CommonUIMessages._6;

	protected String name;
	protected String description;
	
	protected Object parent;
	protected List children;
	protected boolean isLinked;
	protected Object data;
	
	/**
	 * Constructor for LogicalFolder
	 */
	protected LogicalFolder()
	{
		setNamedElement(this);
		isLinked = false;
	}
	
	/**
	 * Constructor for LogicalFolder
	 * @param name The name associated with this logical folder
	 * @throws IllegalArgumentException if the name is null or zero-length.
	 */
	public LogicalFolder(String name)
	throws IllegalArgumentException
	{
		this();
		if((name == null) || (name.length() == 0))
			throw new IllegalArgumentException(CommonUIMessages._10);
		this.name = name;
	}

	/**
	 * Constructor for LogicalFolder
	 * @param name The name associated with this logical folder
	 * @param parent The parent of this logical folder.
	 * @throws IllegalArgumentException if the name or parent is <code>null</code> 
	 * or if the* name is zero-length.
	 */
	public LogicalFolder(Object parent, String name)
	throws IllegalArgumentException
	{
		this(name);
		if(parent == null)
			throw new IllegalArgumentException(NLS.bind(CommonUIMessages._9, "parent"));
		this.parent = parent;
	}

	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		/*
		 * The children list cannot be cleared if it is a linked list.
		 */
		if(!isLinked())
			children.clear();
			
		data = null;
		parent = null;
	}

	/**
	 * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
	 */
	public Object getAdapter(Class adapter)
	{
		if(adapter == IPropertySource.class)
			return this;
			
		if(adapter == IPersistableElement.class)
			return this;
			
		return Platform.getAdapterManager().getAdapter(this, adapter);
	}

	/**
	 * @see org.eclipse.hyades.ui.util.INamedElement#setName(String)
	 */
	public void setName(String name)
	{
		throw new UnsupportedOperationException();
	}

	/**
	 * @see org.eclipse.hyades.ui.util.INamedElement#getName()
	 */
	public String getName()
	{
		return name;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.util.INamedElement#setDescription(java.lang.String)
	 */
	public void setDescription(String description)
	{
		this.description = description;
	}

	/**
	 * @see org.eclipse.hyades.ui.util.INamedElement#getDescription()
	 */
	public String getDescription()
	{
		return description;
	}
	
	/**
	 * Returns the children of this logical folder.
	 * @return List.
	 */
	public boolean hasChildren()
	{
		return (children != null) && (!children.isEmpty());
	}
	
	/**
	 * Sets a given list as the child list of this logical folder.  This method
	 * is not using the setter convention to reduce the chances of misuse.
	 * @param childrenList
	 */
	public void linkChildren(List childrenList)
	{
		isLinked = true;
		children = Collections.unmodifiableList(childrenList);
	}
	
	/**
	 * Returns whether the child list is linked to another list.
	 * @return boolean
	 */
	public boolean isLinked()
	{
		return isLinked;
	}
	
	/**
	 * Returns the children of this logical folder.
	 * @return List.
	 */
	public List getChildren()
	{
		if(children == null)
			children = createChildren();
		return children;
	}
	
	/**
	 * Creates the list that will store the children of this logical folder.  Subclasses 
	 * can overwrite.
	 * @param children
	 */
	protected List createChildren()
	{
		return new ArrayList();
	}
	
	/**
	 * Returns the parent of this logical folder.
	 * @return Object
	 */
	public Object getParent()
	{
		return parent;
	}
	
	/**
	 * Sets the data that is associated with this logical folder.  There is no 
	 * semantics related to the data value.
	 * @param data
	 */
	public void setData(Object data)
	{
		this.data = data;
	}
	
	/**
	 * Returns the data
	 * @return Object
	 */
	public Object getData()
	{
		return data;
	}
	
	/**
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo(Object o)
	{
		if(this == o)
			return 0;	
		
		LogicalFolder viewerFolder = (LogicalFolder)o;
		
		//is this at the root level?
		if(parent == null)
		{
			if(viewerFolder.parent == null)
				return name.compareTo(viewerFolder.name);
				
			//viewerFolder has a parent so it is a "sub" logical folder
			return -1;
		}

		//is viewerFolder is at the root level?
		if(viewerFolder.parent == null)
			return 1;
			
		//are they siblings?
		if(parent.equals(viewerFolder.parent))
			return name.compareTo(viewerFolder.name);
			
		//this and viewFolder have different parents
		return parent.hashCode()-viewerFolder.parent.hashCode();	
	}
	
	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	public boolean equals(Object o)
	{
		if(!(o instanceof LogicalFolder))
			return false;

		return (compareTo(o) == 0);
	}
	
	/**
	 * @see java.lang.Object#hashCode()
	 */	
	public int hashCode()
	{
		if(parent != null)
			return 3*parent.hashCode() + name.hashCode();
		return name.hashCode();
	}
	
	/**
	 * @see org.eclipse.hyades.ui.adapter.NamedElementPropertySource#addPropertyDescriptors()
	 */
	protected IPropertyDescriptor[] addPropertyDescriptors()
	{
		if((children != null) && (children.size() > 0))
		{
			PropertyDescriptor[] propertyDescriptors = new PropertyDescriptor[1];
			propertyDescriptors[0] = new PropertyDescriptor(PROP_CHILD_COUNT, PROP_CHILD_COUNT.toLowerCase());
			propertyDescriptors[0].setCategory(PROP_INFO);
			propertyDescriptors[0].setAlwaysIncompatible(true);
			return propertyDescriptors;
		}		

		return super.addPropertyDescriptors();
	}

	/**
	 * @see IPropertySource#getPropertyValue(Object)
	 */
	public Object getPropertyValue(Object id)
	{
		if(PROP_CHILD_COUNT.equals(id))
			return Integer.toString(getChildren().size());
				
		return super.getPropertyValue(id);
	}
	
	/**
	 * @see org.eclipse.ui.IPersistableElement#getFactoryId()
	 */
	public String getFactoryId()
	{
		return LogicalFolderFactory.class.getName();
	}

	/**
	 * @see org.eclipse.ui.IPersistableElement#saveState(org.eclipse.ui.IMemento)
	 */
	public void saveState(IMemento memento)
	{
		LogicalFolderFactory.save(memento, this);
	}	
}
