/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: HyadesReportExtensionsWizard.java,v 1.4 2009/03/27 17:10:31 paules Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.ui.internal.wizard;

import java.io.InputStream;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.ui.report.IReportGenerator;
import org.eclipse.hyades.ui.report.ReportGeneratorWizard;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIMessages;
import org.eclipse.tptp.platform.common.ui.internal.util.UIUtil;
import org.eclipse.tptp.platform.common.ui.wizard.LocationPage;

/**
 * <p>This wizard is used for the migration from the reportExtensions to the new reportGenerators extension point.</p>
 * 
 *  
 * @author  Patrick Nedelec
 * @author  Paul E. Slauenwhite
 * @version March 27, 2009
 * @since   August 16, 2006
 */
public class HyadesReportExtensionsWizard extends ReportGeneratorWizard {
    protected LocationPage locationPage;
    protected IReportGenerator generator;

    /**
     * 
     */
    public HyadesReportExtensionsWizard(IReportGenerator generator) {
        super();
        setWindowTitle(CommonUIMessages._152); //$NON-NLS-1$
        //setDefaultPageImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_WIZBAN_REPORT));
        this.generator = generator;
    }

    /**
     * @see org.eclipse.hyades.ui.internal.wizard.HyadesWizard#initPages()
     */
    protected void initPages() {
        locationPage = new LocationPage("location", adjustLocation()); //$NON-NLS-1$
        locationPage.setTitle(CommonUIMessages._152); //$NON-NLS-1$
        locationPage.setDescription(CommonUIMessages._154); //$NON-NLS-1$
        String fileExtension = getFileExtension();
        if (fileExtension != null) {
            locationPage.setFileExtension(fileExtension);
            locationPage.setFileName("*." + fileExtension); //$NON-NLS-1$
        }
    }

    protected String getFileExtension() {
        return null;
    }

    protected IStructuredSelection adjustLocation() {
        IStructuredSelection structuredSelection = getSelection();
        Object selection = structuredSelection.getFirstElement();
        if (selection instanceof IResource) {
            IResource res = (IResource) selection;
            IContainer parent = res.getParent();
            if ((parent != null) && parent.exists()) {
                structuredSelection = new StructuredSelection(parent);
            }
        }
        return structuredSelection;
    }

    /**
     * @see org.eclipse.hyades.ui.report.ReportGeneratorWizard#addReportPages()
     */
    protected void addReportPages() throws Exception {
        addPage(locationPage);
    }

    /**
     * @see org.eclipse.hyades.ui.report.ReportGeneratorWizard#generate(org.eclipse.core.runtime.IProgressMonitor)
     */
    public IFile generate(IProgressMonitor monitor) throws Exception {
        IFile file = null;
        IPath path = locationPage.getContainerFullPath();
        IResource res = createContainer(path);

        //String fileExtension = TestUI.REPORT_FILE_EXTENSION;
        //if (generator != null && generator.getReportFileExtension(getSelection()) != null) fileExtension = generator.getReportFileExtension(getSelection());

        //path = new Path(locationPage.getFileName() + "." + fileExtension); //$NON-NLS-1$
        file = ((IContainer) res).getFile(path);
        if (file == null) {
            throw new Exception("Exception caught when retrieving the file"); //$NON-NLS-1$
        }

        if (file.exists()) {
            if (!MessageDialog.openQuestion(getShell(), CommonUIMessages._158, //$NON-NLS-1$
                    CommonUIMessages._159)) //$NON-NLS-1$
                return null;
        }

        InputStream inputStream = null;
        try {
            if (generator != null) {
                inputStream = generator.generate(file, getSelection());
            }
        } catch (Exception e) {
        	CommonPlugin.logError(e);

            String error = e.getLocalizedMessage();
            if ((error == null) || (error.length() == 0)) error = e.getMessage();
            if ((error == null) || (error.length() == 0)) error = e.getClass().getName();

            MessageDialog.openError(getShell(), CommonUIMessages._8, //$NON-NLS-1$
            		NLS.bind(CommonUIMessages._160, error)); //$NON-NLS-1$
            return null;
        }

        if (inputStream == null) {
            MessageDialog.openError(getShell(), CommonUIMessages._8, //$NON-NLS-1$
                    CommonUIMessages._161); //$NON-NLS-1$
            return null;
        }

        try {
            if (file.exists()) {
                file.delete(true, true, null);
            }
            file.create(inputStream, true, null);
        } catch (CoreException e) {
        	CommonPlugin.logError(e);
            UIUtil.openSaveFileErrorDialog(getShell(), file.getFullPath().toString(), e);
            return null;
        }
        return file;
    }

    protected IContainer createContainer(IPath path) {
        IContainer container = null;
        IWorkspace workbench = ResourcesPlugin.getWorkspace();
        int segCount = path.segmentCount();

        for (int idx = 0; idx < segCount; idx++) {
            if (idx == 0) {
                //project
                IProject project = workbench.getRoot().getProject(path.uptoSegment(idx + 1).toString());
                if (project == null || !project.exists()) {
                    //create the project
                    try {
                        project.create(null);
                    } catch (Exception e) {
                        CommonPlugin.logError(e);
                        e.printStackTrace();
                    }
                }
                try {
                    project.open(null);
                } catch (Exception e) {
                }
                container = project;
            } else // (idx > 1)
            { //folder
                IFolder folder = workbench.getRoot().getFolder(path.uptoSegment(idx + 1));
                if (folder == null || !folder.exists()) {
                    //create the folder
                    try {
                        folder.create(false, true, null);
                    } catch (Exception e) {
                    	CommonPlugin.logError(e);
                        e.printStackTrace();
                    }
                }
                container = folder;
            }
        }
        try {
            container.getProject().refreshLocal(IResource.DEPTH_INFINITE, null);
        } catch (CoreException exc) {
        	CommonPlugin.logError(exc);
            exc.printStackTrace();
        }
        return container;
    }

    /**
     * @see org.eclipse.hyades.ui.report.ReportGeneratorWizard#isAvailable(org.eclipse.jface.viewers.ISelection)
     */
    public boolean isAvailable(ISelection selection) {
        return true;
    }

}
