/**********************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: HyadesReportWizard.java,v 1.6 2009/08/05 21:03:55 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.ui.internal.wizard;

import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.ui.internal.report.ReportDescription;
import org.eclipse.hyades.ui.internal.report.ReportManager;
import org.eclipse.hyades.ui.internal.wizard.selection.IWizardElement;
import org.eclipse.hyades.ui.internal.wizard.selection.WizardElement;
import org.eclipse.hyades.ui.report.IReportDescription;
import org.eclipse.hyades.ui.report.IReportGenerator;
import org.eclipse.hyades.ui.report.ReportGeneratorWizard;
import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.wizard.IWizard;
import org.eclipse.osgi.util.NLS;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIMessages;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;
import org.eclipse.tptp.platform.common.ui.internal.util.UIUtil;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.wizards.newresource.BasicNewResourceWizard;
import org.osgi.framework.Bundle;

/**
 * <p>This wizard allows the user to generate a report from its workbench selection.</p>
 * 
 * 
 * @author  Patrick Nedelec
 * @author  Paul E. Slauenwhite
 * @version March 27, 2009
 * @since   August 16, 2006
 */
public class HyadesReportWizard extends HyadesWizard {
    protected static String REPORT_WIZARD_KEY = "HyadesReportWizard"; //$NON-NLS-1$

    HyadesReportSelectionWizardPage reportSelectionPage;
    protected String contextHelpID = null;
    /**
     * Constructor for HyadesReportWizard
     */
    public HyadesReportWizard() {
        super();
        setWindowTitle(CommonUIMessages._152); //$NON-NLS-1$
        //setDefaultPageImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_WIZBAN_REPORT));

        IDialogSettings dialogSettings = CommonUIPlugin.getDefault().getDialogSettings();
        IDialogSettings settings = dialogSettings.getSection(REPORT_WIZARD_KEY);
        if (settings == null) settings = dialogSettings.addNewSection(REPORT_WIZARD_KEY);
        setDialogSettings(settings);
        setForcePreviousAndNextButtons(true);
    }
    
    public HyadesReportWizard(String contextHelpID) {
    	this();    	
    	this.contextHelpID = contextHelpID;        
    }


    protected void initPages() {
        reportSelectionPage = contextHelpID != null ? new HyadesReportSelectionWizardPage("report", getWorkbench(), getSelection(), getWizardElements(), CommonUIMessages._156, contextHelpID) : new HyadesReportSelectionWizardPage("report", getWorkbench(), getSelection(), getWizardElements(), CommonUIMessages._156); //$NON-NLS-1$ //$NON-NLS-2$
        reportSelectionPage.setTitle(CommonUIMessages._152); //$NON-NLS-1$
        reportSelectionPage.setDescription(CommonUIMessages._180); //$NON-NLS-1$
    }

    /**
     * @return
     */
    protected IWizardElement[] getWizardElements() {
        List wizardElems = new LinkedList();
        try {
            //- Get all the contributors to reportGenerators
            Iterator it = HyadesReportGeneratorsExtensionRegistryReader.getInstance().getWizardCategoryElements().iterator();
            while (it.hasNext()) {
                IConfigurationElement elem = (IConfigurationElement) it.next();
                ReportGeneratorWizard wiz = (ReportGeneratorWizard) elem.createExecutableExtension("class"); //$NON-NLS-1$
                if (wiz.isAvailable(getSelection())) {
                    WizardElement wizardElement = new WizardElement(wiz); //$NON-NLS-1$
                    wizardElement.setId(elem.getAttribute(HyadesReportGeneratorsExtensionRegistryReader.ID));
                    wiz.setId(wizardElement.getId());                    
                    wizardElement.setLabel(elem.getAttribute(HyadesReportGeneratorsExtensionRegistryReader.NAME));
                    wizardElement.setDescription(elem.getAttribute(HyadesReportGeneratorsExtensionRegistryReader.DESCRIPTION));
                    ImageDescriptor imageDescriptor = null;
                    String symbolicName = elem.getDeclaringExtension().getNamespace();
                    Bundle bundle = Platform.getBundle(symbolicName);
                    if (bundle != null && bundle.getState() == Bundle.ACTIVE) {
                        String icon_path = elem.getAttribute(HyadesReportGeneratorsExtensionRegistryReader.ICON);
                        if (icon_path != null) {
                            URL url = new URL(bundle.getEntry("/"), icon_path); //$NON-NLS-1$
                            imageDescriptor = ImageDescriptor.createFromURL(url);
                        }
                    }
                    wizardElement.setImageDescriptor(imageDescriptor);
                    wizardElems.add(wizardElement);
                }
            }

            //- Ensure that the old extension point named reportExtensions still read
            ReportManager manager = ReportManager.getInstance();
            IReportDescription[] list = manager.getReportDescriptions();
            for (int idx = 0; idx < list.length; idx++) {
                ReportDescription data = (ReportDescription) list[idx];
                IReportGenerator gen = null;
                gen = data.getReportGenerator();
                if (gen != null && gen.isAvailable(getSelection())) {
                    WizardElement wizardElement = new WizardElement(new HyadesReportExtensionsWizard(gen));
                    wizardElement.setId(data.getName());
                    wizardElement.setLabel(data.getName());
                    wizardElement.setDescription("wizard.Report.Description"); //$NON-NLS-1$
//                    wizardElement.setImageDescriptor(TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_REPORT));
                    wizardElems.add(wizardElement);
                }
            }
        } catch (Exception e) {
            CommonPlugin.logError(e);
        }
        IWizardElement[] arrayOfWizardElement = new IWizardElement[wizardElems.size()];
        wizardElems.toArray(arrayOfWizardElement);
        return arrayOfWizardElement;
    }

    public void addPages() {
        addPage(reportSelectionPage);
    }

    public boolean performFinish() {
        try {
            FinishOperation op = new FinishOperation();
            getContainer().run(false, false, op);
            IFile reportFile = op.getReportFile();
            String editorid = op.getEditorId();
            
            reportSelectionPage.saveWidgetValues();
            if (reportFile == null) {
                return false;
            }
            BasicNewResourceWizard.selectAndReveal(reportFile, getWorkbench().getActiveWorkbenchWindow());
            open(reportFile, editorid);
        } catch (InterruptedException e) {
        	CommonPlugin.logError(e);
            return false;
        } catch (InvocationTargetException e) {
            Throwable exception = e.getTargetException();
            CommonPlugin.logError(exception);

            String error = exception.getLocalizedMessage();
            if ((error == null) || (error.length() == 0)) error = exception.getMessage();
            if ((error == null) || (error.length() == 0)) error = exception.getClass().getName();

            MessageDialog.openError(getShell(), CommonUIMessages._8, //$NON-NLS-1$
            		NLS.bind(CommonUIMessages._160, error)); //$NON-NLS-1$                
            return false;
        }
        return true;
    }

    protected void open(IFile file, String editorid) {
        IPreferenceStore store = CommonUIPlugin.getDefault().getPreferenceStore();
        store.setValue("REPORT_OPEN_EDITOR", reportSelectionPage.openEditor());
        if (reportSelectionPage.openEditor()) UIUtil.openEditor(file, editorid, false);
    }

    class FinishOperation extends WorkspaceModifyOperation {
        protected IFile indexReport = null;
        protected String editorid = null;

        /**
         * @see org.eclipse.ui.actions.WorkspaceModifyOperation#execute(org.eclipse.core.runtime.IProgressMonitor)
         */
        protected void execute(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {
            try {
                IWizard selectedWizard = reportSelectionPage.getSelectedNode().getWizard();
                if (selectedWizard instanceof ReportGeneratorWizard) {
                    monitor.beginTask("", 1); //$NON-NLS-1$             
                    indexReport = ((ReportGeneratorWizard) selectedWizard).generate(new SubProgressMonitor(monitor, 1, SubProgressMonitor.PREPEND_MAIN_LABEL_TO_SUBTASK));
                    editorid = ((ReportGeneratorWizard) selectedWizard).getEditorId();
                }
            } catch (Throwable e) {
                throw new InvocationTargetException(e);
            } finally {
                monitor.done();
            }
        }

        public IFile getReportFile() {
            return indexReport;
        }

        public String getEditorId(){
        	return editorid;
        }
    }

}
