/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SelectionPage.java,v 1.6 2009/03/27 17:10:31 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.wizard.selection;


import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.wizard.IWizard;
import org.eclipse.jface.wizard.IWizardNode;
import org.eclipse.jface.wizard.WizardSelectionPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIMessages;
import org.eclipse.tptp.platform.common.ui.internal.util.UIUtil;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchWizard;
import org.eclipse.ui.model.WorkbenchContentProvider;
import org.eclipse.ui.model.WorkbenchLabelProvider;
import org.eclipse.ui.model.WorkbenchViewerSorter;

/**
 * <p>Selection page.</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Paul Slauenwhite
 * @version	March 27, 2009
 * @since   August 16, 2006
 */
public class SelectionPage 
extends WizardSelectionPage implements ISelectionChangedListener, IDoubleClickListener
{
	protected static final String SET_SELECTED_WIZARD_ID = "SelectionPage.STORE_SELECTED_WIZARD_ID"; //$NON-NLS-1$
	protected final static int SIZING_LISTS_HEIGHT = 200;
	
	protected IWorkbench workbench;
	protected IStructuredSelection structuredSelection;
	protected IWizardElement[] wizardElements;
	protected String message;
	
	protected TableViewer wizardSelectionViewer;

	/**
	 * Constructor for SelectionPage
	 * @param pageName
	 */
	public SelectionPage(String pageName, IWorkbench workbench, IStructuredSelection structuredSelection, IWizardElement[] wizardElements, String message)
	{
		super(pageName);
		
		this.workbench = workbench;
		this.structuredSelection = structuredSelection;
		this.wizardElements = wizardElements;
		this.message = message;
		
		setTitle(CommonUIMessages._49); 
		setDescription(CommonUIMessages._50); 
	}

	/**
	 * @see org.eclipse.jface.dialogs.IDialogPage#dispose()
	 */
	public void dispose()
	{
		workbench = null;
		structuredSelection = null;
		wizardElements = null;
		super.dispose();
	}

	/**
	 * Returns the workbench definied in this page's constructor.
	 * @return IWorkbench
	 */
	public IWorkbench getWorkbench()
	{
		return workbench;
	}

	/**
	 * Returns the selection definied in this page's constructor.
	 * @return IWorkbench
	 */
	public IStructuredSelection getStructuredSelection()
	{
		return structuredSelection;
	}
	
	/**
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent)
	{
		Font font = parent.getFont();

		// create composite for page.
		Composite outerContainer = new Composite(parent, SWT.NONE);
		outerContainer.setLayout(new GridLayout());
		outerContainer.setLayoutData(new GridData(GridData.FILL_BOTH));
		outerContainer.setFont(font);

		Label messageLabel = new Label(outerContainer,SWT.NONE);
		messageLabel.setText(message);
		messageLabel.setFont(font);

		//Create a table for the list
		Table table = new Table(outerContainer, SWT.BORDER);
		GridData data = new GridData(GridData.FILL_BOTH);
	
		int availableRows = UIUtil.availableRows(parent);
	
		//Only give a height hint if the dialog is going to be too small
		if(availableRows > 50){
			data.heightHint = SIZING_LISTS_HEIGHT;
		}
		else{
			data.heightHint = availableRows * 3;
		}
		
		table.setLayoutData(data);
		table.setFont(font);

		// the list viewer		
		wizardSelectionViewer = new TableViewer(table);
		wizardSelectionViewer.setContentProvider(new WorkbenchContentProvider());
		wizardSelectionViewer.setLabelProvider(new WorkbenchLabelProvider());
		wizardSelectionViewer.setSorter(new WorkbenchViewerSorter());
		wizardSelectionViewer.addSelectionChangedListener(this);
		wizardSelectionViewer.addDoubleClickListener(this);

		wizardSelectionViewer.setInput(new WizardElementWorkbenchAdapter(wizardElements));
		restoreWidgetValues();
		
		table.setFocus();
        
        
        //- Adds a custom part
        addCustomPart(outerContainer);
        
		setControl(outerContainer);			
	}
	
	/**
     * Classes that extend SelectionPage may add their own widgets to the page.
     * @param parent the parent composite
     * 
     * @author pnedelec
     */
    protected void addCustomPart(Composite parent) {
    }

    /**
	 * Notes the newly-selected wizard element and updates the page accordingly.
	 * @param event the selection changed event
	 */
	public void selectionChanged(SelectionChangedEvent event)
	{
		setErrorMessage(null);
		IStructuredSelection selection = (IStructuredSelection) event.getSelection();
		IWizardElement wizardElement = (IWizardElement)selection.getFirstElement();
		if (wizardElement == null)
		{
			setMessage(null);
			setSelectedNode(null);
			return;
		}

		setSelectedNode(createWizardNode(wizardElement));
		setMessage(wizardElement.getDescription());
	}
	
	/**
	 * @see org.eclipse.jface.viewers.IDoubleClickListener#doubleClick(org.eclipse.jface.viewers.DoubleClickEvent)
	 */
	public void doubleClick(DoubleClickEvent event)
	{
		selectionChanged(new SelectionChangedEvent(wizardSelectionViewer, wizardSelectionViewer.getSelection()));
		getContainer().showPage(getNextPage());
	}
	
	/**
	 * Since Finish was pressed, write widget values to the dialog store so that they
	 * will persist into the next invocation of this wizard page
	 */
	public void saveWidgetValues()
	{
		if(getDialogSettings() == null)
			return;
	
		// since the user is able to leave this page then exactly one wizard
		// must be currently selected
		IStructuredSelection sSelection = (IStructuredSelection)wizardSelectionViewer.getSelection();
		// We are losing the selection going back
		if(!sSelection.isEmpty())
		{
			IWizardElement wizardElement = (IWizardElement)sSelection.getFirstElement();
			if(wizardElement.getId() != null)
				getDialogSettings().put(SET_SELECTED_WIZARD_ID, wizardElement.getId());
		}
	}

	/**
	 * Uses the dialog store to restore widget values to the values that they held
	 * last time this wizard was used to completion.
	 */
	protected void restoreWidgetValues()
	{
		if(getDialogSettings() == null)
			return;
			
		String wizardElementId = getDialogSettings().get(SET_SELECTED_WIZARD_ID);
		if(wizardElementId == null)
			return;
			
		IWizardElement wizardElement = findWizardElement(wizardElementId);
		if (wizardElement == null)
			return;

		StructuredSelection selection = new StructuredSelection(wizardElement);
		wizardSelectionViewer.setSelection(selection);
		selectionChanged(new SelectionChangedEvent(wizardSelectionViewer,selection));
	}
	
	/**
	 *	Answer the wizard object corresponding to the passed id, or null
	 *	if such an object could not be found
	 *
	 *	@return IWizardElement
	 *	@param searchPath java.lang.String
	 */
	protected IWizardElement findWizardElement(String wizardElementId)
	{
		for (int i = 0; i < wizardElements.length; ++i)
		{
			if(wizardElementId.equals(wizardElements[i].getId()))
				return wizardElements[i];
		}
	
		return null;
	}	
	
	/**
	 * Returns an <code>IWizardNode</code> representing the specified wizard element
	 * which has been selected by the user.  <b>Subclasses</b> may override this
	 * implementation.
	 *
	 * @param element the wizard element that an <code>IWizardNode</code> is needed for
	 * @return org.eclipse.jface.wizards.IWizardNode
	 */
	protected IWizardNode createWizardNode(IWizardElement wizardElement)
	{
		IWizardNode wizardNode = (IWizardNode)wizardElement.getAdapter(IWizardNode.class);
		if((wizardNode != null) && (!wizardNode.isContentCreated()))
		{
			IWizard wizard = wizardNode.getWizard();
			if(wizard instanceof IWorkbenchWizard)
			{
				((IWorkbenchWizard)wizard).init(getWorkbench(), getStructuredSelection());
				wizard.addPages();
			}
		}
		return wizardNode;
	}
	
	/**
	 * Sets the array of wizard elements.
	 * @param elements
	 */
	public void setWizardElements(IWizardElement[] elements) {
	    this.wizardElements = elements;
	    wizardSelectionViewer.setInput(new WizardElementWorkbenchAdapter(wizardElements));
		restoreWidgetValues();
	}
}
