/********************************************************************** 
 * Copyright (c) 2005, 2008  IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 * $Id: ContextManager.java,v 1.2 2008/01/24 02:28:55 apnan Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.ui.provisional.context;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.ui.internal.context.ContextHandlerImpl;
import org.eclipse.hyades.ui.internal.context.ContextLanguageImpl;
import org.eclipse.hyades.ui.internal.context.ContextMappingPreference;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;

/**
 * This class manages TPTP context language resources. It manages all registered 
 * context language as well as context hanlder extension. It allows them to be accessed
 * context language by key or Id, and context handler using context key.
 * 
 * @since 4.1
 */
public class ContextManager {

	//instance of context manager
	protected static ContextManager instance;
	
	//extension points and elements Id
	protected final String CONTEXTS_EXTENTION_POINT = "contexts";
	protected final String CONTEXT_ELEMENT = "context";
	protected final String CONTEXT_PROVIDERS_EXTENTION_POINT = "contextProviders";
	protected final String CONTEXT_PROVIDER_ELEMENT = "contextProvider";

	protected Map contextLanguageByKey;
	protected Map contextHandlersByKey;
	
	protected ContextMappingPreference contextMappingPreference;

	protected ContextManager()
	{
		contextMappingPreference = new ContextMappingPreference(CommonUIPlugin.getDefault().getPreferenceStore());
	}
	
	/**
	 * Return an instance of the context manager.
	 * @return instance of context manager
	 */
	public static ContextManager instance()
	{
		if (instance == null)
			instance = new ContextManager();
		
		return instance;
	}
	
	/**
	 *  Initialization of context related resources.
	 */ 
	protected void initialize() 
	{
		initializeContextLanguage();
		initializeContextProviders();
	}
	
	/**
	 * Initialization of context launguage registrated from extension point.
	 */
	protected void initializeContextLanguage()
	{
		contextLanguageByKey = new HashMap();		
		
		IExtensionPoint point = Platform.getExtensionRegistry().getExtensionPoint("org.eclipse.hyades.ui", CONTEXTS_EXTENTION_POINT);

		if (point != null) {
			IConfigurationElement[] elements = point.getConfigurationElements();
			
			for (int i = 0; i < elements.length; i++) {
				
				IConfigurationElement elem = elements[i];
				
				String pluginID = elem.getDeclaringExtension().getNamespace();
				
				if (elem.getName().equals(CONTEXT_ELEMENT)) {
					String id = elem.getAttribute("id");	
					String key = elem.getAttribute("contextKey");
					String name = elem.getAttribute("name");
					String description = elem.getAttribute("description");				
					
					if(id == null || key == null || name == null)
						 continue;
	
					String iconPath = elem.getAttribute("icon");
					ImageDescriptor imageDescriptor = CommonUIPlugin.imageDescriptorFromPlugin(pluginID,iconPath);
										
					IContextLanguage cl = new ContextLanguageImpl(id, key, name, description, imageDescriptor);
					
					contextLanguageByKey.put(cl.contextKey(), cl);				
				}
			}
		}
	}

	/**
	 * Initialization of context handler registrated from extension point.
	 */
	protected void initializeContextProviders()
	{
		contextHandlersByKey = new HashMap();
		
		IContextLanguage[] contexts = getContextLanguages();

		IExtensionPoint point = Platform.getExtensionRegistry().getExtensionPoint("org.eclipse.hyades.ui", CONTEXT_PROVIDERS_EXTENTION_POINT);
		if (point != null) {
			IConfigurationElement[] elements = point.getConfigurationElements();
			
			for (int i = 0; i < elements.length; i++) {
				
				IConfigurationElement elem = elements[i];
				
				if (elem.getName().equals(CONTEXT_PROVIDER_ELEMENT)) {

					String id = elem.getAttribute("id");	
					
					IContextProvider contextProvider = null;
					try
					{
						if (!"".equals(elem.getAttribute("class"))) {
							Object clz = elem.createExecutableExtension("class");
							if (clz instanceof IContextProvider)
								contextProvider = (IContextProvider)clz;
							else
								CommonPlugin.logError("Class in extension point org.eclipse.hyades.ui"+CONTEXT_PROVIDERS_EXTENTION_POINT+" must implement IContextProvider");
						}
					}
					catch (Exception e)
					{
						e.printStackTrace();
					}
	
					String excludeContext = elem.getAttribute("excludeContext");					
					
					IConfigurationElement[] contextsElems = elem.getChildren("contexts");
					Vector contextsV1 = new Vector();
					for (int k = 0; k < contextsElems.length; k++)
						contextsV1.addElement(contextsElems[k].getChildren()[0].getAttribute("refID"));
					String[] contextsA = new String[contextsV1.size()];
					contextsV1.toArray(contextsA);				
	
					if (id != null && contextProvider != null)
					{
						IContextHandler ch = new ContextHandlerImpl(id, contextProvider, "true".equals(excludeContext), contextsA.length == 0? null : contextsA);
						
						for(int j = 0; j < contexts.length; j++)
						{
							Vector contextsV = (Vector)contextHandlersByKey.get(contexts[j].contextKey());
							if (contextsV == null)
							{
								contextsV = new Vector();
								contextHandlersByKey.put(contexts[j].contextKey(), contextsV);							
							}
							
							if (keyInContextHandler(contexts[j].contextKey(), ch))
								contextsV.addElement(ch);
						}
						//TODO: log error if context handler not in any context
					}
				}
				else {
					//TODO log error on unrecognize element
				}
			}
		}
	}
	
	/**
	 * Validate if given context key is a support context of the given context handler 
	 * @param key context key for validation
	 * @param ch conttext handler for validation
	 * @return true if context key is support by context handler, false otherwise
	 */
	protected boolean keyInContextHandler(String key, IContextHandler ch)
	{
		if (ch.contexts() == null)
			return !ch.excludeContexts();
		
		for (int i = 0; i < ch.contexts().length; i++)
		{
			if (key.equals(ch.contexts()[i]))
				return !ch.excludeContexts();
		}
		
		return ch.excludeContexts();
	}
	
	protected IContextLanguage internalGetContextLanguage(String contextKey)
	{
		if (contextLanguageByKey == null)
			initialize();
		return (IContextLanguage) contextLanguageByKey.get(contextKey);
	}

	/**
	 * Retrieve the context language from the given context key.
	 * @param contextKey the context key 
	 * @return an IContextLanguage instance
	 */
	public static IContextLanguage getContextLanguage(String contextKey)
	{
		return ContextManager.instance().internalGetContextLanguage(contextKey);
	}
	
	protected IContextLanguage[] internalGetContextLanguages()
	{
		if (contextLanguageByKey == null)
			initialize();
		IContextLanguage[] contexts = new IContextLanguage[contextLanguageByKey.size()];
		contextLanguageByKey.values().toArray(contexts);
		return contexts;
	}
	
	/**
	 * Retrieve an array of all the context lanaguages registered.
	 * @return IContextLanguage[] an array of IContextLanguage instances
	 */
	public static IContextLanguage[] getContextLanguages()
	{
		return ContextManager.instance().internalGetContextLanguages();
	}
	
	protected IContextHandler getContextHandlerById(String contextKey, String id)
	{
		IContextHandler[] ch = internalGetContextHandlers(contextKey);
		
		for (int i = 0; i < ch.length; i++)
		{
			if (ch[i].id().equals(id))
				return ch[i];
		}
		
		return null;
	}
	
	protected Vector getContextHandlersByKey(String key)
	{
		Vector v = (Vector)contextHandlersByKey.get(key);
		if (v!=null)
			return v;
		else
			return new Vector();
	}

	protected IContextHandler[] internalGetContextHandlers(String contextKey)
	{
		if (contextHandlersByKey == null)
			initialize();
		
		Vector v;
		if (contextKey != null){
			v = getContextHandlersByKey(contextKey);
		}
		else
		{
			v = new Vector();
			Iterator i = contextHandlersByKey.keySet().iterator();
			while (i.hasNext())
				v.addAll(getContextHandlersByKey((String)i.next()));				
		}
		
		IContextHandler[] contextHandlers = new IContextHandler[v!=null?v.size():0];
		if (v != null)
			v.toArray(contextHandlers);		
		
		return contextHandlers;
	}	
	
	/**
	 * Returns an array of context handlers mapped with the given context key
	 * @param contextKey context key
	 * @return IContextHandler[] an array of IContextHandlers instances
	 */
	public static IContextHandler[] getContextHandlers(String contextKey)
	{
		return ContextManager.instance().internalGetContextHandlers(contextKey);
	}
	
	protected IContextProvider internalGetContextProvider(String contextKey)
	{
		if (contextKey == null)
			return null;
		
		if (contextHandlersByKey == null)
			initialize();
		
		if (contextMappingPreference.changed())
			contextMappingPreference.populateMappingByContext();
		
		IContextHandler handler = getContextHandlerById(contextKey, contextMappingPreference.getContextProviderId(contextKey));
		if (handler != null)
			return handler.getContextProvider();
		else
			return null;
	}
	
	/**
	 * Retrieve the context provider for the given context key, this returns the selected
	 * context provider in the context selection preference page.
	 * @param contextKey the context key 
	 * @return IContextProvider supporting the given context key
	 */
	public IContextProvider getContextProvider(String contextKey)
	{
		return ContextManager.instance().internalGetContextProvider(contextKey);
	}
	
	/**
	 * Retrieve the default context provider. Default context provider is used
	 * when context key of resource is not provided.  It will return the selected
	 * IContextProvider for the default context.
	 * @return IContextProvider of the default context
	 */
	public IContextProvider getDefautProvider() 
	{
		return getContextProvider(getDefautContext());
	}
	
	/**
	 * Retrieve the default context key. Default context key is used
	 * when context key of resource is not provided
	 * @return String the context key of the default context
	 */
	public static String getDefautContext() 
	{
		return "Java";
	}

	protected boolean supportsAttribute(IContextProvider cp, String attribute){
		String[] attrs = cp.getSupportAttributes();
		for (int i=0;i<attrs.length;i++) {
			if (attrs[i].equals(attribute)) return true;
		}
		return false;
	}
	
	protected boolean supportsMode(IContextLabelFormatProvider clfp, int mode){
		int[] modes = clfp.getSupportModes();
		for (int i=0;i<modes.length;i++) {
			if (modes[i] == mode) return true;
		}
		return false;
	}
	
	protected IContextLabelFormatProvider internalGetContextLabelFormatProvider(String contextKey, String attribute, int mode) {
		
		IContextProvider cp = getContextProvider(contextKey);
		
		while (cp != null)
		{
			if (supportsAttribute(cp,attribute)) {
				IContextLabelFormatProvider clfp = cp.getContextLabelFormatProvider(attribute);
				
				if (supportsMode(clfp, mode))
					return clfp;
			}
			
			cp = getContextProvider(cp.getBaseProvider());
		}
				
		//return the default one if nothing found
		return getDefautProvider().getContextLabelFormatProvider(attribute);
	}
	
	/**
	 * Returns the label format provider for the given context key, attribute, and mode.  If a particular context does not
	 * have a formatter for a particular attribute and/or mode, it will use the formatter of its parent context
	 * language.
	 * @param contextKey String the context key
	 * @param attribute String the context attribute
	 * @param mode int the mode
	 * @return IContextLabelFormatProvider returns the formatter
	 */
	public static IContextLabelFormatProvider getContextLabelFormatProvider(String contextKey, String attribute, int mode) {
		return ContextManager.instance().internalGetContextLabelFormatProvider(contextKey, attribute, mode);
	}
}
