/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ContextOpenSourceHandlerManager.java,v 1.3 2008/01/24 02:28:55 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.ui.provisional.context;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Vector;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.ui.internal.context.ContextMappingPreference;
import org.eclipse.hyades.ui.internal.context.ContextOpenSourceHandler;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;


/**
 * This class manages TPTP context sensitive open resources action. It manages all 
 * registered context open source. It allows them to be accessed by context key.
 * 
 * @since 4.1
 */
public class ContextOpenSourceHandlerManager {

	//instance of manager
	protected static ContextOpenSourceHandlerManager instance;
	
	//extension point and element id
	protected final String CONTEXT_OPEN_SRC_HANDLER_EXTENTION_POINT = "contextOpenSourceProviders"; //$NON-NLS-1$
	protected final String CONTEXT_OPEN_SRC_HANDLER_ELEMENT = "contextOpenSourceProvider"; //$NON-NLS-1$
	
	protected Map contextOpenSourceHandlers;
	
	protected ContextMappingPreference contextMappingPreference;
	
	protected ContextOpenSourceHandlerManager()
	{
		contextMappingPreference = new ContextMappingPreference(CommonUIPlugin.getDefault().getPreferenceStore());
	}
	
	/**
	 * Return an instance of the manager.
	 * @return instance of manager
	 */
	public static ContextOpenSourceHandlerManager instance()
	{
		if (instance == null)
			instance = new ContextOpenSourceHandlerManager();
		
		return instance;
	}
	
	/**
	 *  Initialization of context open source resources.
	 */ 
	protected void initialize()
	{
		contextOpenSourceHandlers = new HashMap();
		
		IContextLanguage[] contexts = ContextManager.getContextLanguages();

		IExtensionPoint point = Platform.getExtensionRegistry().getExtensionPoint("org.eclipse.hyades.ui", CONTEXT_OPEN_SRC_HANDLER_EXTENTION_POINT); //$NON-NLS-1$

		if (point != null) {
			IConfigurationElement[] elements = point.getConfigurationElements();
			
			for (int i = 0; i < elements.length; i++) {
				
				IConfigurationElement elem = elements[i];
				
				if (elem.getName().equals(CONTEXT_OPEN_SRC_HANDLER_ELEMENT)) {

					String id = elem.getAttribute("id"); //$NON-NLS-1$	
					
					IContextOpenSourceProvider contextOpenSourceProvider = null;
					try
					{
						if (!"".equals(elem.getAttribute("class"))) { //$NON-NLS-1$
							Object cls = elem.createExecutableExtension("class"); //$NON-NLS-1$
							if (cls instanceof IContextOpenSourceProvider)
								contextOpenSourceProvider = (IContextOpenSourceProvider)cls;
							else
								CommonPlugin.logError("Class in extension point org.eclipse.hyades.ui"+CONTEXT_OPEN_SRC_HANDLER_EXTENTION_POINT+" must implement IContextOpenSourceProvider"); //$NON-NLS-1$
						}		
							
					}
					catch (Exception e)
					{
						e.printStackTrace();
					}
	
					String excludeContext = elem.getAttribute("excludeContext"); //$NON-NLS-1$					
					
					IConfigurationElement[] contextsElems = elem.getChildren("contexts"); //$NON-NLS-1$
					
					Vector contextsV1 = new Vector();
					
					for (int k = 0; k < contextsElems.length; k++)
						contextsV1.addElement(contextsElems[k].getChildren()[0].getAttribute("refID")); //$NON-NLS-1$
					
					String[] contextsA = new String[contextsV1.size()];
					contextsV1.toArray(contextsA);				
	
					if (id != null && contextOpenSourceProvider != null)
					{
						IContextOpenSourceHandler ch = new ContextOpenSourceHandler(id, contextOpenSourceProvider, "true".equals(excludeContext), contextsA.length == 0? null : contextsA); //$NON-NLS-1$
						
						for(int j = 0; j < contexts.length; j++)
						{
							Vector contextsV = (Vector)contextOpenSourceHandlers.get(contexts[j].contextKey());
							if (contextsV == null)
							{
								contextsV = new Vector();
								contextOpenSourceHandlers.put(contexts[j].contextKey(), contextsV);							
							}
							
							if (keyInContextHandler(contexts[j].contextKey(), ch))
								contextsV.addElement(ch);
						}
						//TODO: log error if context handler not in any context
					}
				}
				else {
					//TODO log error on unrecognize element
				}
			}
		}
	}
	
	
	/**
	 * Validate if given context key is a support context of the given context open source handler 
	 * @param key context key for validation
	 * @param ch conttext open source handler for validation
	 * @return true if context key is support by context open source handler, false otherwise
	 */
	protected boolean keyInContextHandler(String key, IContextOpenSourceHandler ch)
	{
		if (ch.contexts() == null)
			return !ch.excludeContexts();
		
		for (int i = 0; i < ch.contexts().length; i++)
		{
			if (key.equals(ch.contexts()[i]))
				return !ch.excludeContexts();
		}
		
		return ch.excludeContexts();
	}

	protected IContextOpenSourceProvider internalGetContextOpenSourceProvider(String contextKey)
	{
		if (contextKey == null)
			return null;
		
		if (contextOpenSourceHandlers == null)
			initialize();
		
		if (contextMappingPreference.changed())
			contextMappingPreference.populateMappingByContext();
		
		IContextOpenSourceHandler handler = getContextOpenSourceHandlerById(contextKey, contextMappingPreference.getContextOpenSourceProviderId(contextKey));
		if (handler != null)
			return handler.getContextOpenSourceProvider();
		else
			return null;
	}
	
	/**
	 * Return a context open source provider from the given context key.  This returns the selected
	 * IContextProvider for the given context, as selected in the context selection preference page.
	 * @param contextKey the context key
	 * @return IContextOpenSourceProvider the context open source provider
	 */
	public static IContextOpenSourceProvider getContextOpenSourceProvider(String contextKey)
	{
		return ContextOpenSourceHandlerManager.instance().internalGetContextOpenSourceProvider(contextKey);
	}
	
	protected IContextOpenSourceHandler getContextOpenSourceHandlerById(String contextKey, String id)
	{
		IContextOpenSourceHandler[] ch = getContextOpenSourceHandlers(contextKey);
		
		for (int i = 0; i < ch.length; i++)
		{
			if (ch[i].id().equals(id))
				return ch[i];
		}
		
		return null;
	}
	
	protected Vector getContextOpenSourceHandlersByKey(String key)
	{
		Vector v = (Vector)contextOpenSourceHandlers.get(key);
		if (v != null)
			return v;
		else
			return new Vector();
	}
	
	protected IContextOpenSourceHandler[] internalGetContextOpenSourceHandlers(String contextKey)
	{
		if (contextOpenSourceHandlers == null)
			initialize();
		
		Vector v;
		if (contextKey != null)
		{
			v = getContextOpenSourceHandlersByKey(contextKey);
		}
		else
		{
			v = new Vector();
			Iterator i = contextOpenSourceHandlers.keySet().iterator();
			while (i.hasNext())
				v.addAll(getContextOpenSourceHandlersByKey((String)i.next()));				
		}

		IContextOpenSourceHandler[] contextOpenSourceHandlersResult = new IContextOpenSourceHandler[v.size()];
		v.toArray(contextOpenSourceHandlersResult);
		
		return contextOpenSourceHandlersResult;
	}
	
	/**
	 * Return an array of context open source handlers for the given context key.  It returns all
	 * the context handlers associated with the given context.
	 * @param contextKey the context key
	 * @return IContextOpenSourceHandler[] an array of IContextOpenSourceHandler instances
	 */
	public static IContextOpenSourceHandler[] getContextOpenSourceHandlers(String contextKey)
	{
		return ContextOpenSourceHandlerManager.instance().internalGetContextOpenSourceHandlers(contextKey);
	}
}
