/**********************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: IColumnLabel.java
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.ui.provisional.context;

import org.eclipse.swt.graphics.Image;


public interface IContextLabelFormatProvider {
	
	/**
	 * Below is a listing of modes.  These modes are used when retreiving the display strings
	 * and display images.  The mode determines which area the String/Image is being retrieved
	 * for so that the getDisplayStringFromElement and getDisplayImageByElement methods, can
	 * determine which String to use.
	 * 
	 * Example:
	 * For the IContextAttributes.PACKAGE_NAME attribute the following modes returns the following
	 * in the basic java context provider, the default provider.
	 * MODE_COLUMN_HEADER: "Package" the header visible in a statistical view
	 * MODE_COLUMN_CONTENT: "java.util." a formatter for the content shown in the cells in the stats views.
	 * MODE_VIEW_FILTER: "Package name" the text displayed in the filter
	 * MODE_VIEW_ACTION: "" Not implemented and not used for this attribute.
	 */
	
	/**
	 * The mode associated with the column header names in the statistical views.
	 */
	public static final int MODE_COLUMN_HEADER = 0;
	
	/**
	 * The mode associated with the column cell content in the statistical views.
	 */
	public static final int MODE_COLUMN_CONTENT = 1;
	
	/**
	 * The mode associated with an action.
	 */
	public static final int MODE_VIEW_ACTION = 2;
	
	/**
	 * The mode associated with view filter strings.
	 */
	public static final int MODE_VIEW_FILTER = 3;
	
	/**
	 * This returns the display string given the value to be displayed, the element this
	 * String is associated with, and the mode to determine which String to show.
	 * @param value the value to be displayed
	 * @param element the element associated with this String
	 * @param mode the mode to determine which String to display
	 * @return String the display String
	 */
	public String getDisplayStringFromElement(Object value, Object element, int mode);
	
	/**
	 * This returns the display image given the value to be displayed, the element this
	 * String is associated with, and the mode to determine which image to show.
	 * @param value the value to be displayed
	 * @param element the element associated with this String
	 * @param mode the mode to determine which String to display
	 * @return Image the Image associated with this element
	 */
	public Image getDisplayImageByElement(Object value, Object element, int mode);
	
	/**
	 * Returns the display description if a display description is shown for the attribute.
	 */
	public String getDisplayDescription();
	
	/**
	 * Returns the display tool tip if a tool tip is shown for the attribute.
	 */
	public String getDisplayToolTip();

	/**
	 * Converts the given model value text to the displayed text in the filter dialog in
	 * the context associated with this IContextLabelFormatProvider.  This
	 * does not apply to the Object Collected filter attribute.
	 * @param text String the text to be converted
	 * @return the converted text
	 */
	public String convertModelValueToDisplayString(String text);
	
	/**
	 * Converts the given display text to be displayed/entered in the filter dialog to
	 * the model value in the context associated with this IContextLabelFormatProvider.  This
	 * does not apply to the Object Collected filter attribute. 
	 * @param text String the text to be converted
	 * @return the converted text
	 */
	public String convertDisplayStringToModelValue(String text);
	
	/**
	 * Returns the supported modes of this IContextLabelFormatProvider.  If a mode is requested
	 * that is not supported the base provider will be used.  
	 * @return int[] an array of integers, the modes applicable.
	 */
	public int[] getSupportModes();
}
