/********************************************************************** 
 * Copyright (c) 2006, 2009 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html         
 *  
 * 
 * Contributors: 
 * 		pnedelec (IBM) - Initial API and implementation 
 **********************************************************************/
package org.eclipse.hyades.ui.util;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Path;

/** 
 * <p>Contains utilities to validate the location of an EMF resource.</p>
 * 
 * @author  Patrick Nedelec
 * @author  Paul Slauenwhite
 * @version October 27, 2009
 * @since   January 24, 2007
 */
public class LocationValidator {
    
	public static final String INVALID_PATH = "Invalid Path"; //$NON-NLS-1$
    
    /**
     * This method validates the file name to determine if it conforms to 
     * OS file name restrictions.
     * 
     * At this point we are testing for valid file names on Windows and Linux 
     * systems only.  In the future more system restrictions may need to be 
     * added.
     * 
     * @param valueStr The file name string being passed in
     * @return (The invalid character | INVALID_PATH | null)  
     */
    public static String validateOSFileName(String valueStr) {
        if (valueStr.startsWith("-")) { //$NON-NLS-1$
            return new Character(valueStr.charAt(0)).toString();
        }
        if (!Path.ROOT.isValidPath(valueStr)) {
            return INVALID_PATH;
        }
        IWorkspace workspace = ResourcesPlugin.getWorkspace();
        IStatus result = workspace.validateName(valueStr, IResource.FILE);
        if (!result.isOK()) {
            return INVALID_PATH;
        }
        return validateEMFSegment(valueStr);
    }   
    
    /**
     * Validate the path of an EMF resource.
     * <p>
     * Note: A path having exactly 1 segment will be validated as a project.
     * It it has more segments, the path will be validated as a folder.
     * </p>
     * @see IWorkspace#validatePath(String, int);
     * @param path the absolute path relative to the workspace
     * @return (The invalid character | INVALID_PATH | null)
     */
    public static String validatePath(IPath path) {
        int type = path.segmentCount() == 1 ? IResource.PROJECT : IResource.FOLDER;
        return validatePath(path, type);
    }
    
    /**
     * Validate the path of an EMF resource.
     * @see IWorkspace#validatePath(String, int);
     * @param path the absolute path relative to the workspace
     * @param type bitwise-or of the resource type constants (
     * <code>FILE</code>,<code>FOLDER</code>,<code>PROJECT</code>, or
     * <code>ROOT</code>) indicating expected resource type(s)
     * @return (The invalid character | INVALID_PATH | null)
     */
    public static String validatePath(IPath path, int type) {
        IWorkspace workspace = ResourcesPlugin.getWorkspace();
        IStatus result = workspace.validatePath(path.toString(), type);
        if (!result.isOK()) {
            return INVALID_PATH;
        }
        if (!path.isValidPath(path.toString())) {
            return INVALID_PATH;
        }
        //- Now check if the path does not contain EMF reserved character
        String[] segments = path.segments();
        for (int i = 0, max = segments.length; i < max; i++) {
            //- Second validate that the segment does not contain reserved EMF character
            String invalidChar = validateEMFSegment(segments[i]);
            if (invalidChar != null) {
                return invalidChar;
            }
        }
        return null;
    }
    
    /**
	 * <p>Determines if the EMF segment is valid.</p>
	 *  
	 * <p>Validity is based on the following conditions:</p>
	 * 
	 * <ul>
	 * <li>The EMF segment is not <code>null</code>.</li>
	 * <li>The EMF segment does not contain any of the following characters:
	 * <ul>
	 *   <li><</li>
	 *   <li>></li>
	 *   <li>+</li>
	 *   <li>|</li>
	 *   <li>{</li>
	 *   <li>}</li>
	 *   <li>[</li>
	 *   <li>]</li>
	 *   <li>:</li>
	 *   <li>;</li>
	 *   <li>,</li>
	 *   <li>"</li>
	 *   <li>/</li>
	 *   <li>\\</li>
	 *   <li>`</li> 
	 *   <li>=</li> 
	 *   <li>?</li>
	 *   <li>#</li>
	 *   <li>%</li>
	 *   <li>@</li>
	 *   <li>&</li>
	 *   <li>$</li>
	 *   <li>^</li>
	 * </ul>
	 * </ul>
	 * 
	 * <p>This validation follows the syntax defined by <a href="http://www.ietf.org/rfc/rfc2396.txt">RFC 2396</a>.</p>
	 * 
	 * @param  name The EMF segment.
	 * @return <code>null</code> if all of the validity conditions are satisfied (see above), otherwise the invalid character.
	 * @provisional As of TPTP V4.5.6, this is stable provisional API (see http://www.eclipse.org/tptp/home/documents/process/development/api_contract.html).
	 */
    public static String validateEMFSegment(String name) {
        for (int i = 0; i < name.length(); i++) {
            char ch = name.charAt(i);
            switch (ch) {
                case '<':
                    return new Character(ch).toString();
                case '>':
                    return new Character(ch).toString();
                case '+':
                    return new Character(ch).toString();
                case '|':
                    return new Character(ch).toString();
                case '{':
                    return new Character(ch).toString();
                case '}':
                    return new Character(ch).toString();
                case '[':
                    return new Character(ch).toString();
                case ']':
                    return new Character(ch).toString();
                case ':':
                    return new Character(ch).toString();
                case ';':
                    return new Character(ch).toString();
                case ',':
                    return new Character(ch).toString();
                case '"':
                    return new Character(ch).toString();
                case '/':
                    return new Character(ch).toString();
                case '\\':
                    return new Character(ch).toString();
                case '\'': 
                    return new Character(ch).toString();
                case '=': 
                    return new Character(ch).toString();
                case '?':
                    return new Character(ch).toString();
                case '#':
                    return new Character(ch).toString();
                case '%':
                    return new Character(ch).toString();
                case '@':
                    return new Character(ch).toString();
                case '&':
                    return new Character(ch).toString();
                case '$':
                    return new Character(ch).toString();
                case '^':
                    return new Character(ch).toString();
            }
        }
        return null;
    }
}
