/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: GraphNode.java,v 1.3 2008/01/24 02:28:49 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.ui.core;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.uml2sd.ui.drawings.IGC;
import org.eclipse.hyades.uml2sd.ui.drawings.ISDPreferences;
import org.eclipse.hyades.uml2sd.util.DebugUtil;



/**
 * The base class used for all UML2 graph nodes displayed in the Sequence Diagram SDWidget.
 * 
 * @author sveyrier
 * @version 1.0
 */

public abstract class GraphNode {
	
	
	protected int startEventOccurrence = 0;
	
	protected int endEventOccurrence = 0;
	
	/**
	 * Preference ColorId to use to draw font
	 */
	public String prefId = ISDPreferences.PREF_SYNC_MESS;
	
	/**
	 *  The selection state of the graph node.
	 */
	protected boolean selected = false;
	
	/**
	 *  The focus state of the graph node.
	 */
	protected boolean focused = false;
	
	
	protected boolean hasChilds=false;
	
	/**
	 * The graph node name used to label the graph node in the View.
	 */
	protected String 	name	 = ""; //$NON-NLS-1$
	
	protected HashMap nodes;
	protected HashMap fnodes;
	protected HashMap bnodes;
	
	protected HashMap indexes;	
	protected HashMap fSort;
	protected HashMap bSort;
	
	/**
	 * Reset the internal index of the first visible GraphNode for
	 * each ordered GraphNode lists
	 *
	 */
	public void resetIndex()
	{
		if (!hasChilds)
			return;
		Iterator it=indexes.keySet().iterator();
		while (it.hasNext())
		{
			Object nodeType=it.next();
			indexes.put(nodeType,new Integer(0));
		}
	}
	
	/**
	 * Add a GraphNode into the receiver
	 * @param nodeToAdd the node to add
	 */
	public void addNode (GraphNode nodeToAdd)
	{
		if (!hasChilds)
		{
			nodes=new HashMap(2);
			fnodes=new HashMap(2);
			bnodes=new HashMap(2);
			indexes=new HashMap(2);
			bSort=new HashMap(2);
			fSort=new HashMap(2);
			hasChilds=true;
		}
		
		//Nothing to add
		if (nodeToAdd == null)
			return;
	
		if (nodes.get(nodeToAdd.getArrayId())==null)
		{
			nodes.put(nodeToAdd.getArrayId(),new ArrayList(1));
			indexes.put(nodeToAdd.getArrayId(),new Integer(0));
			fnodes.put(nodeToAdd.getArrayId(),new ArrayList(1));
			fSort.put(nodeToAdd.getArrayId(),new Boolean(false));
			if (nodeToAdd.getBackComparator()!=null)
			{
				bnodes.put(nodeToAdd.getArrayId(),new ArrayList(1));
				bSort.put(nodeToAdd.getArrayId(),new Boolean(false));
			}
		}
		
		List fNodeList=(List)fnodes.get(nodeToAdd.getArrayId());
		List bNodeList=null;
		if (bnodes!=null)
			bNodeList=(List)bnodes.get(nodeToAdd.getArrayId());
		if (fNodeList!=null&&fNodeList.size()>0)
		{
			//check if the nodes are added y ordered
			//if not, tag the list to sort it later (during draw)
			GraphNode node=(GraphNode)fNodeList.get(fNodeList.size()-1);
			Comparator fcomp = nodeToAdd.getComparator();
			Comparator bcomp = nodeToAdd.getBackComparator();
			if ((fcomp!=null)&&(fcomp.compare(node,nodeToAdd)==1))
			{
				fSort.put(nodeToAdd.getArrayId(),new Boolean(true));
			}
			if ((bcomp!=null)&&(bcomp.compare(node,nodeToAdd)==1))
			{
				bSort.put(nodeToAdd.getArrayId(),new Boolean(true));
			}
		}
		fNodeList.add(nodeToAdd);
		nodes.put(nodeToAdd.getArrayId(),fNodeList);
		fnodes.put(nodeToAdd.getArrayId(),fNodeList);
		if (nodeToAdd.getBackComparator()!=null)
		{	
			bNodeList.add(nodeToAdd);
			bnodes.put(nodeToAdd.getArrayId(),bNodeList);
		}
	}
	
	/**
	 * Set the graph node name.<br>
	 * It is the name display in the view to label the graph node.
	 * @param the name to set
	 */
	public void setName(String nodeName)
	{
		name = nodeName;
	}

	/**
	 * Returns the graph node name.<br>
	 * It is the name display in the view to label the graph node.
	 * @return the graph node name
	 */
	public String getName()
	{
		return name;
	}
	
	
	/**
	 * Tags the the graph node has selected.<br>
	 * WARNING: This method is only used to draw the 
	 * graph node using the system selection colors. <br>
	 * To use the complete
	 * SDViewer selection mechanism (selection management, notification, etc..)
	 * see SDWidget class
	 * @see org.eclipse.hyades.uml2sd.ui.view.SDWidget#addSelection(GraphNode)
	 * @see org.eclipse.hyades.uml2sd.ui.view.SDWidget#removeSelection(GraphNode)
	 * @see org.eclipse.hyades.uml2sd.ui.view.SDWidget#clearSelection()
	 * @param  selection - true to set selected, false to set unselected
	 */
	public void setSelected(boolean selection)
	{
		selected = selection;
	}
	
	/**
	 * Tags the the graph node as focused.<br>
	 * WARNING: This method is only used to draw the 
	 * graph node using the system focus style. <br>
	 * To use the complete
	 * SDViewer focus mechanism
	 * see SDWidget class
	 * @see org.eclipse.hyades.uml2sd.ui.view.SDWidget#addSelection(GraphNode)
	 * @see org.eclipse.hyades.uml2sd.ui.view.SDWidget#removeSelection(GraphNode)
	 * @see org.eclipse.hyades.uml2sd.ui.view.SDWidget#clearSelection()
	 * @param  focus - true to set focued, false otherwise
	 */
	public void setFocused(boolean focus)
	{
		focused = focus;
	}
	
	/**
	 * Returns true if the graph node is selected, false otherwise.<br>
	 * The returned value is used to highlight the graph node in the View.
	 * @return true if selected, false otherwise
	 * 
	 */
	public boolean isSelected()
	{
		return selected;
	}
	
	/**
	 * Returns true if the graph node is focused, false otherwise.<br>
	 * The returned value is used to highlight the graph node in the View.
	 * @return true if focued, false otherwise
	 * 
	 */
	public boolean hasFocus()
	{
		return focused;
	}
	
	/**
	 * Returns true if the graph node contains the point given in parameter, return false otherwise.
	 * @param	x  the x coordinate of the point to test containment <br>
	 * 			y  the y coordinate of the point to test containment
	 * @return true if contained, false otherwise
	 */
	abstract public boolean contains(int x, int y);
	
	/**
	 * Returns the x coordinate of the graph node
	 * @return the x coordinate
	 */
	abstract public int getX();
	
	/**
	 * Returns the y coordinate of the graph node
	 * @return the y coordinate
	 */
	abstract public int getY();
	
	/**
	 * Returns the graph node height
	 * @return the graph node height
	 */
	abstract public int getHeight();
	
	/**
	 * Returns the graph node width
	 * @return the graph node width
	 */
	abstract public int getWidth();
	
	/**
	 * Draws the graph node in the given context
	 * @param context the graphical context to draw in
	 */
	abstract protected void draw(IGC context);
	
	
	/**
	 * Returns the GraphNode visibility for the given visible area.
	 * Wrong visibity calculation, may strongly impact drawing performance
	 * @param vx
	 * @param vy
	 * @param vwidth
	 * @param vheight
	 * @return true if visible false otherwise
	 */
	public boolean isVisible(int x, int y, int width, int height)
	{
		return true;
	}
	
	/**
	 * Return a comparator to sort the GraphNode of the same type
	 * This comparator is used to order the GraphNode array of the given node type.
	 * (see getArrayId).
	 * @return the comparator
	 */
	public Comparator getComparator()
	{
		return null;
	}
	
	/**
	 * If needed, return a different comparator to backward scan the GraphNode array
	 * @return the backward comparator or null if not needed
	 */
	public Comparator getBackComparator()
	{
		return null;
	}
	
	/**
	 * Compare two graphNodes
	 * @param node the node to compare to
	 * @return true if equal false otherwise
	 */
	public boolean isSameAs(GraphNode node)
	{
		return false;
	}
	
	/**
	 * Return the node type for all class instances.
	 * This id is used to store the same nodes kind in the same ordered array.
	 * @return the node type identifier
	 */
	abstract public String getArrayId();
	
	/**
	 * Return true if the distance from the GraphNode to the given point is positif
	 * @param x the point x coordinate
	 * @param y the point y coordinate
	 * @return true if positif false otherwise
	 */
	public boolean positiveDistanceToPoint(int x, int y)
	{
		return false;
	}
	
	/**
	 * Returns the graph node which contains the point given in parameter
	 * WARNING: Only graph nodes in the current visible area can be returned
	 * @param x the x coordinate of the point to test
	 * @param y the y coordinate of the point to test
	 * @return the graph node containing the point given in parameter, null otherwise
	 */
	public GraphNode getNodeAt(int x, int y)
	{
		GraphNode toReturn=null;
		
		if (!hasChilds)
			return null;
		
		Iterator it=nodes.keySet().iterator();
		GraphNode node=null;
		while (it.hasNext())
		{
			Object nodeType=it.next();
			List list=(List)nodes.get(nodeType);
			int index=((Integer)indexes.get(nodeType)).intValue();
			node = getNodeFromListAt(x, y, list, index);
			if (toReturn==null)
				toReturn=node;
			if (node != null)
			{
				GraphNode internalNode=node.getNodeAt(x,y);
				if (internalNode!=null)
					return internalNode;
//				else return node;
				else if (Math.abs(node.getWidth())<Math.abs(toReturn.getWidth())||
						Math.abs(node.getHeight())<Math.abs(toReturn.getHeight()))
					toReturn=node;
			}
		}
		return toReturn;
	}
	
	public ArrayList getNodeList(GraphNode from, GraphNode to)
	{	
		ArrayList result=new ArrayList();
		
		if (from!=null)
		{
			result.add(from);
		}
		else if (to!=null)
		{
			result.add(to);
		}
		
		if (from==null||to==null)
			return result;
		
		if (from==to)
			return result;
		
		int startX=Math.min(from.getX(),Math.min(to.getX(),Math.min(from.getX()+from.getWidth(),
				to.getX()+to.getWidth())));
		int endX=Math.max(from.getX(),Math.max(to.getX(),Math.max(from.getX()+from.getWidth(),
				to.getX()+to.getWidth())));
		int startY=Math.min(from.getY(),Math.min(to.getY(),Math.min(from.getY()+from.getHeight(),
				to.getY()+to.getHeight())));
		int endY=Math.max(from.getY(),Math.max(to.getY(),Math.max(from.getY()+from.getHeight(),
				to.getY()+to.getHeight())));
		
		if (!hasChilds)
			return result;
		
		Iterator it=nodes.keySet().iterator();
		while (it.hasNext())
		{
			Object nodeType=it.next();
			List nodesList=(List)nodes.get(nodeType);
			if (nodesList==null||nodesList.isEmpty())
				return null;
			for (int i=0;i<nodesList.size();i++)
			{
				GraphNode node=(GraphNode)nodesList.get(i);
				int nw=node.getWidth();
				int nh=node.getHeight();
				int nx=node.getX();
				int ny=node.getY();
				if (contains(startX,startY,endX-startX,endY-startY,nx+1,ny+1)&&
					contains(startX,startY,endX-startX,endY-startY,nx+nw-2,ny+nh-2))
					result.add(node);
				result.addAll(node.getNodeList(from,to));
			}
		}
		if ((to!=null)&&(!result.contains(to)))
			result.add(to);
		return result;
	}
	
	
	/**
	 * Returns the graph node which contains the point given in parameter
	 * for the given graph node list and starting the iteration at the given index<br>
	 * WARNING: Only graph nodes with smaller coordinates
	 * than the current visible area can be returned.<br>
	 * 
	 * @param x the x coordinate of the point to test
	 * @param y the y coordinate of the point to test
	 * @param list the list to search in
	 * @param fromIndex list browsing starting point
	 * @return the graph node containing the point given in parameter, null otherwise
	 */ 
	protected GraphNode getNodeFromListAt(int x, int y, List list, int fromIndex)
	{
		if (list==null)
			return null;
		for (int i=fromIndex; i<list.size();i++)
		{
			GraphNode node = (GraphNode)list.get(i);
			if (node.contains(x, y))
				return node;
		}
		return null;
	}
	
	/**
	 * Returns the start event occurrence attached to this graphNode.
	 * @return the start event occurrence attached to the graphNode
	 */
	public int getStartOccurrence()
	{
		return startEventOccurrence;
	}
	
	/**
	 * Returns the end event occurrence attached to this graphNode
	 * @return the start event occurrence attached to the graphNode
	 */
	public int getEndOccurrence()
	{
		return endEventOccurrence;
	}
	
	/**
	 * Computes the index of the first visible GraphNode for
	 * each ordered graph node lists depending on the visible 
	 * area given in parameter
	 *
	 * @param x visible area top left corner x coordinate
	 * @param y visible area top left corner y coordinate
	 * @param width visible area width
	 * @param height visible area height
	 */
	public void updateIndex(int x, int y, int width, int height)
	{
		if (!hasChilds)
			return;
		
		if (DebugUtil.debugIndex())
		{
			System.out.print("*****************************\n"); //$NON-NLS-1$
			System.out.print("Visible area position in virtual screen (x,y)= "+x+" "+y+"\n\n"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		}
		
		Iterator it=nodes.keySet().iterator();
		while (it.hasNext())
		{
			Object nodeType=it.next();
			int direction = 1;
			int drawIndex = ((Integer)indexes.get(nodeType)).intValue();
/*			if (x==0)
			{
				drawIndex = 0;
				indexes.put(nodeType,new Integer(drawIndex));
			}*/
			if ((nodes.get(nodeType) != null)&&(((List)nodes.get(nodeType)).size()>1))
			{
				if(((GraphNode)((List)nodes.get(nodeType)).get(drawIndex)).positiveDistanceToPoint(x,y))
					direction=-1;
				
				if (drawIndex==0)
					direction = 1;
			
				if ((direction == -1) && (bnodes.get(nodeType) != null))
				{
					GraphNode currentNode= (GraphNode)((List)nodes.get(nodeType)).get(drawIndex);
					drawIndex = Arrays.binarySearch(((List)bnodes.get(nodeType)).toArray(),
							((List)nodes.get(nodeType)).get(drawIndex),
							currentNode.getBackComparator());
					nodes.put(nodeType,bnodes.get(nodeType));
					if (drawIndex<0)
					{
						drawIndex=0;
						direction = 1;
					}
					else nodes.put(nodeType,bnodes.get(nodeType));
				}
				GraphNode prev=null;
				
				for (int i=drawIndex; i<((List)nodes.get(nodeType)).size()&&i>=0;i=i+direction)
				{
					drawIndex = i;
					indexes.put(nodeType,new Integer(i));
					
					GraphNode currentNode= (GraphNode)((List)nodes.get(nodeType)).get(i);
					
					if (prev==null)
						prev = currentNode;
					
					Comparator comp=currentNode.getComparator();
					HashMap sort=fSort;
					
					if (direction==-1)
					{
						if (currentNode.getBackComparator()!=null)
						{
							comp=currentNode.getBackComparator();
							sort=bSort;
						}
					}
					
					if (i<((List)nodes.get(nodeType)).size()-1)
					{
						GraphNode next= (GraphNode)((List)nodes.get(nodeType)).get(i+1);
						
						if ((comp!=null)&&(comp.compare(currentNode,next)==1))
							sort.put(nodeType,new Boolean(true));
					}
					if (direction ==1)
					{
						if(((GraphNode)((List)nodes.get(nodeType)).get(i)).positiveDistanceToPoint(x,y))
							break;
					}
					else 
					{
						if (currentNode.getBackComparator()==null)
						{
							if //(currentNode.isVisible(x,y,width,height)
									(!currentNode.positiveDistanceToPoint(x,y))
								break;
						}
						else 
						{
							if (currentNode.isVisible(x,y,width,height)
									&&!currentNode.positiveDistanceToPoint(x,y))
							{
								if ((comp!=null)&&(comp.compare(currentNode,prev)<=0))
									break;
							}
							else if ((comp!=null)&&(comp.compare(currentNode,prev)<=0))
								prev = currentNode;
						}
					}
				}
			
				nodes.put(nodeType,fnodes.get(nodeType));
				if ((bnodes.get(nodeType)!=null)&&(direction ==-1))
				{
//					nodes.put(nodeType,fnodes.get(nodeType));
					int index=((Integer)indexes.get(nodeType)).intValue();
					List list=(List)nodes.get(nodeType);
					List backList=(List)bnodes.get(nodeType);
					GraphNode currentNode = (GraphNode)(backList.get(index));
					if (index>0)
					{
						index = Arrays.binarySearch(list.toArray(),
								backList.get(index),
							currentNode.getComparator());
							if (index<0)
								index=0;
						indexes.put(nodeType,new Integer(index));
					}
				}
				
				for (int i=drawIndex; i<((List)nodes.get(nodeType)).size()&&i>=0;i++)
				{
					GraphNode toDraw = (GraphNode)((List)nodes.get(nodeType)).get(i);
					toDraw.updateIndex(x, y, width, height);
					if (!toDraw.isVisible(x,y,width,height))
						break;
				}
			}
			if (DebugUtil.debugIndex())
			{
				System.out.print("First drawn "+nodeType+" index = "+drawIndex+"\n"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$		
				System.out.print(nodeType + " found in "+0+" iterations\n"); //$NON-NLS-1$ //$NON-NLS-2$
			}
		}

		if (DebugUtil.debugIndex())
		{
			System.out.print("*****************************\n"); //$NON-NLS-1$
		}
	}
	
	/**
	 * Draws the childs nodes on the given context.<br>
	 * This method start width GraphNodes ordering if neeeded.<br>
	 * After, depending on the visible area, only visible GraphNodes are drawn.<br>
	 * @param context the context to draw to
	 * @param drawFrame indicate if the frame rectangle need to be redrawn
	 * @see org.eclipse.hyades.uml2sd.ui.core.GraphNode#draw(IGC)
	 */
	protected void drawChildsNodes (IGC context)
	{
		if (!hasChilds)
			return;
		//If the nodes have not been added ordered, the array is ordered
		Iterator it=fSort.keySet().iterator();
		while (it.hasNext())
		{
			Object nodeType=it.next();
			boolean sort = ((Boolean)fSort.get(nodeType)).booleanValue();
			if (sort)
			{
				Object[] temp=((List)fnodes.get(nodeType)).toArray();
				GraphNode node=(GraphNode)((List)nodes.get(nodeType)).get(0);
				Arrays.sort(temp,node.getComparator());
				fSort.put(nodeType,new Boolean(false));
				nodes.put(nodeType, Arrays.asList(temp));
				fnodes.put(nodeType, Arrays.asList(temp));
				if (DebugUtil.debugSorting())
					System.out.print(nodeType+" array sorted\n"); //$NON-NLS-1$
			}
		}
		
		Iterator it2=bSort.keySet().iterator();
		while (it2.hasNext())
		{
			Object nodeType=it2.next();
			boolean sort = ((Boolean)bSort.get(nodeType)).booleanValue();
			if (sort)
			{
				Object[] temp=((List)bnodes.get(nodeType)).toArray();
				GraphNode node=(GraphNode)((List)nodes.get(nodeType)).get(0);
				Arrays.sort(temp,node.getBackComparator());
				bSort.put(nodeType,new Boolean(false));
				bnodes.put(nodeType, Arrays.asList(temp));
				if (DebugUtil.debugSorting())
					System.out.print(nodeType+" back array sorted\n"); //$NON-NLS-1$
			}
		}
		
		if (DebugUtil.debugDisplay())
		{
			System.out.print("*****************************\n"); //$NON-NLS-1$
		}
		
		int arrayStep =1;
		if ((Metrics.getMessageFontHeigth()+Metrics.MESSAGES_NAME_SPACING*2)*context.getZoom()<Metrics.MESSAGE_SIGNIFICANT_VSPACING)
			arrayStep = Math.round(Metrics.MESSAGE_SIGNIFICANT_VSPACING/((Metrics.getMessageFontHeigth()
				+Metrics.MESSAGES_NAME_SPACING*2)*context.getZoom()));
		
		int count=0;
		Iterator it3=fSort.keySet().iterator();
		while (it3.hasNext())
		{
			count=0;
			Object nodeType=it3.next();
			GraphNode node=(GraphNode)((List)nodes.get(nodeType)).get(0);
			context.setFont(Frame.getUserPref().getFont(node.prefId));
			int index=((Integer)indexes.get(nodeType)).intValue();
			count=drawNodes(context,(List)nodes.get(nodeType),index,arrayStep);
			if (DebugUtil.debugDisplay())
				System.out.print(count +" "+nodeType+" drawn, starting from index "+index+"\r\n"); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		}
		if (DebugUtil.debugDisplay())
		{
			System.out.print("*****************************\n"); //$NON-NLS-1$
		}
	}
	
	/**
	 * Draw the GraphNode stored in the given list, starting at
	 * index startIndex with the given step
	 * @param context the context to draw to
	 * @param list the GraphNodes list
	 * @param startIndex the start index
	 * @param step the step to browse the list
	 * @return the number of GraphNodes drawn
	 */
	protected int drawNodes(IGC context, List list, int startIndex, int step)
	{
		if (!hasChilds)
			return 0;
		
		GraphNode last=null;
		int nodesCount=0;
		if (list.size()<0)
			return 0;
		
		GraphNode node=(GraphNode)list.get(0);
		context.setFont(Frame.getUserPref().getFont(node.prefId));
		Comparator comparator=node.getComparator();
		for (int i=startIndex; i<list.size(); i=i+step)
		{
			GraphNode toDraw= (GraphNode)list.get(i);
			if (i<list.size()-1)
			{
				GraphNode next= (GraphNode)list.get(i+1);
				if ((comparator!=null)&&(comparator.compare(toDraw,next)==1))
				{
					fSort.put(next.getArrayId(),new Boolean(true));
				}
			}
			int cx=context.getContentsX();
			int cy=context.getContentsY();
			int cw=context.getVisibleWidth();
			int ch=context.getVisibleHeight();
			//The arrays should be ordered, no needs to continue for this one
			if (!toDraw.isVisible(cx,cy,cw,ch)
				&&toDraw.positiveDistanceToPoint(cx+cw,cy+ch))
					break;
			//***Common*** nodes visibility
			if ((!toDraw.isSameAs(last)||toDraw.isSelected())&&
					(toDraw.isVisible(context.getContentsX(),context.getContentsY(),
					context.getVisibleWidth(),context.getVisibleHeight())))
			{
				nodesCount++;
				toDraw.draw(context);
				if (hasFocus())
					toDraw.drawFocus(context);
			}
			last=toDraw;
		}
		return nodesCount;
	}
	
	
	public void drawFocus(IGC context)
	{
		context.drawFocus(getX(),getY(),getWidth(),getHeight());
	}
	
	/**
	 * Determine if the given point (px,py) is contained in the rectangle
	 * (x,y,width,height)
	 * @param x the rectangle x coordinate
	 * @param y the rectangle y coordinate
	 * @param width the rectangle width
	 * @param height the rectangle height
	 * @param px the x coordinate of the point to test 
	 * @param py the y coordinate of the point to test 
	 * @return true if contained false otherwise
	 */
	public static boolean contains(int x,int y, int width, int height, int px, int py)
	{
		int locX = x;
		int locY = y;
		int locWidth = width;
		int locHeight = height;
		
		if (width<0)
		{
			locX = locX + width;
			locWidth = -locWidth;
		}
		
		if (height<0)
		{
			locY = locY + height;
			locHeight = -locHeight;
		}
		return (px >= locX) && (py >= locY) && ((px - locX) <= locWidth) && ((py - locY) <= locHeight);
	}
}
