/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: BackgroundLoader.java,v 1.3 2008/01/24 02:29:16 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.ui.load;

import java.lang.reflect.InvocationTargetException;
import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.uml2sd.util.SDUtil;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.part.ViewPart;

/**
 * An interface to the background thread for loading sequence
 * diagrams. Ensures that only one thread is running, and that
 * only one sequence diagram is being loaded at any time. A call
 * to newTask() overrides the current task, canceling it. 
 *
 * @author Curtis d'Entremont
 * @since 1.3
 */
public class BackgroundLoader {
	
	protected static ViewPart view;
	protected boolean debug = false;
	
	protected static BackgroundLoader instance;
	protected Thread thread;
	protected IProgressMonitor monitor;
	protected List taskList = new LinkedList();
	protected static Image image = SDUtil.getResourceImage("full/clcl16/sequencediagram_obj.gif").createImage(); //$NON-NLS-1$
	protected void addTask(IRunnableWithProgress task) {
		taskList.add(task);
	}
	
	protected IRunnableWithProgress getNextTask() {
		if (!taskList.isEmpty()) {
			return (IRunnableWithProgress)taskList.get(0);
		}
		return null;
	}
	
	protected IRunnableWithProgress consumeTask() {
		synchronized (taskList) {
			IRunnableWithProgress task = getNextTask();
			if (task != null) {
				taskList.remove(0);
			}
			return task;
		}
	}

	protected void clearTasks() {
		synchronized (taskList) {
			taskList.clear();
		}
	}

	/**
	 * This is a singleton class. If the instance does not yet
	 * exist, it is created.
	 * 
	 * @return the single instance.
	 */
	public synchronized static BackgroundLoader getInstance() {
		if (instance == null)
			instance = new BackgroundLoader();
		return instance;
	}
	
	/**
	 * Schedules a new task (load diagram, refresh, etc). If the
	 * thread is currently busy, the current task is canceled and
	 * the task is scheduled as the next task. If not, the task
	 * starts immediately.
	 * 
	 * Note: The task is responsible for polling the progress
	 * monitor for cancelation.
	 * 
	 * @param task the task to be performed.
	 */
	public synchronized void newTask(IRunnableWithProgress task) {
		addTask(task);
		if (debug) {
			System.err.println("newTask(): Adding "+task); //$NON-NLS-1$
		}
		if (thread == null || !thread.isAlive()) {
			thread = new Thread(new Runnable() {
				public void run() {
					IRunnableWithProgress task = getNextTask();
					if (debug) {
						System.err.println("newTask(): Testing next task #1 = "+task); //$NON-NLS-1$
					}
					while (task != null) {
						try {
							cancelIt(false);
							runIt(task);
							consumeTask();
							cancelIt(false);
						}
						catch (Exception e) {
							clearTasks();
							cancelIt(true);
							if (debug) {
								e.printStackTrace();
							}
						} finally {
						}
						task = getNextTask();
						if (debug) {
							System.err.println("newTask(): Testing next task #2 = "+task); //$NON-NLS-1$
						}
					}
				}

				/**
				 * @param task
				 * @throws InterruptedException
				 * @throws InvocationTargetException
				 */
				protected void runIt(final IRunnableWithProgress task)
				throws InvocationTargetException, InterruptedException {
					task.run(monitor);
				}
			});
			thread.setName("Sequence Diagram Background Loader"); //$NON-NLS-1$
			thread.start();
		}
		else {
			cancelIt(true);
		}
	}
	
	protected void cancelIt(final boolean b) {
		synchronized(monitor) {
			monitor.setCanceled(b);
		}
	}

	/**
	 * Request a new task to follow the eventually existing one and start immediately after that.
	 * @param task the task to be performed
	 * @param monitor_ the monitor to be used
	 */
	public void newSubsequentTask(final IRunnableWithProgress task) {
		addTask(task);
		if (getNextTask() == task) {
			if (debug) {
				System.err.println("newSubsequentTask(): Calling newTask because no current task is executed"); //$NON-NLS-1$
			}
			newTask(task);
		} else synchronized(BackgroundLoader.this) {
			if (debug) {
				System.err.println("newSubsequentTask(): Adding "+task); //$NON-NLS-1$
			}
		}
	}

	/**
	 * Cancels the current task. This will only take effect the
	 * next time the task checks for cancelation.
	 */
	public synchronized void cancelTask() {
		cancelIt(true);
		if (thread.isAlive()) {
			thread.destroy();
		}
		clearTasks();
	}
	
	public static void shutdown() {
		if (instance != null && instance.monitor != null && !instance.monitor.isCanceled()) {
			instance.monitor.setCanceled(true);
		}
		instance = null;
		image.dispose();
	}
	
	protected BackgroundLoader() {
		monitor = new CancelProgressMonitor();
	}
	
	public static void setView(ViewPart view_) {
		view = view_;
	}

	/**
	 * This monitor is for cancelation monitoring only. It is used to
	 * signal the running task that the user canceled. 
	 */
	public class CancelProgressMonitor implements IProgressMonitor {
		protected volatile boolean canceled = false;
		protected volatile String taskName;
		
		public boolean isCanceled() {
			return canceled;
		}
		public void setCanceled(boolean canceled_) {
			canceled = canceled_;
			if (canceled_ && taskName != null) {
				if (debug) {
					System.err.println("Canceled task \""+taskName+"\""); //$NON-NLS-1$ //$NON-NLS-2$
				}
			}
		}
		public void beginTask(String name, int totalWork) {
			setTaskName(name);
			if (debug) {
				System.err.println("Starting task \""+taskName+"\"..."); //$NON-NLS-1$ //$NON-NLS-2$
			}
		}
		public void done() {
			if (debug) {
				System.err.println((canceled?"Done normally ":"Done with canceled ")+"task \""+taskName+"\""); //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$ //$NON-NLS-4$
			}
			setTaskName(null);
		}
		public void internalWorked(double work) {}
		public void setTaskName(String name) {
			taskName = name;
			if (view == null) {
				return;
			}
			Display.getDefault().syncExec(new Runnable() {

				public void run() {
					try {
						IStatusLineManager lineManager =
							view.getViewSite().getActionBars().getStatusLineManager();;
						lineManager.setMessage(taskName != null ? image : null, taskName);
					}
					catch (Exception e) {
						if (debug) {
							e.printStackTrace();
						}
					}
				}
			});
		}
		public void subTask(String name) {}
		public void worked(int work) {}
	}
}
