/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LoadersManager.java,v 1.5 2008/01/24 02:29:16 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.uml2sd.ui.load;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.hyades.ui.util.ICommonUIHelper;
import org.eclipse.hyades.uml2sd.ui.view.SDView;
import org.eclipse.hyades.uml2sd.util.DebugUtil;
import org.eclipse.hyades.uml2sd.util.SDUtil;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;
import org.eclipse.ui.IViewReference;
import org.eclipse.ui.IWorkbenchPage;


/**
 * Only one loader provides the UML2 Sequence Diagram widget with a frame at one moment.
 * This class ensures this.
 */
public class LoadersManager {

	protected static LoadersManager loadersManager;
	
	protected static HashMap map;
		
	/**
	 * This should not be used by the clients
	 */
	public LoadersManager() {
		map = new HashMap();
	}
	
	/**
	 * A static method to get the manager instance
	 * @return the manager instance
	 */
	public static LoadersManager getLoadersManager() {
		if (loadersManager == null) {
			loadersManager = ICommonUIHelper.INSTANCE.createLoadersManager();
		}
		return loadersManager;
	}
	
	/**
	 * Create a loader instance and associate it to the view
	 * @param className the name of the class to create an instance from
	 * @param cl the class loader convenient for this class (same plugin class loader)
	 * @param view the UML2 Sequence Diagram view instance
	 * @return the created loader
	 */
	public IUml2SDLoader createLoader(String className, ClassLoader cl, SDView view) 
	{
		IUml2SDLoader loader = createLoaderInstance(className, cl);
		associateLoader(loader, view);
		return loader;
	}
	
	/**
	 * Create a loader instance
	 * @param className the name of the class to create an instance from
	 * @param cl the class loader convenient for this class (same plugin class loader)
	 * @return the created loader
	 */
	public IUml2SDLoader createLoaderInstance(String className, ClassLoader cl) {
		IUml2SDLoader loader = null;
		if (cl == null) {
			cl = getClass().getClassLoader();
		}
		try {
			Class clazz = cl.loadClass(className);
			loader = (IUml2SDLoader)clazz.newInstance();
		} catch(ClassNotFoundException e) {
			e.printStackTrace();
		} catch (IllegalAccessException e) {
			e.printStackTrace();
		} catch (InstantiationException e) {
			e.printStackTrace();
		}
		return loader;
	}

	/**
	 * Associate a loader to a view
	 * @param loader the loader to make the one for this view
	 * @param view the view
	 */
	public void associateLoader(IUml2SDLoader loader, SDView view) {
		//Make it current
		if (view != null) {
			setCurrentLoader(loader, view.getViewSite().getId());
		}
		
		//Assign a view instance to this loader
		if (loader != null) {
			loader.setViewer(view);
		}
	}

	/**
	 * Changes the current unique loader the given Seconfay viewId.
	 * Use null to reference the primary SD View
	 * @param loader the current loader
	 * @param id the view secondary id or null
	 */
	public void setCurrentLoader(IUml2SDLoader loader, String id) 
	{
		if (id==null)
			return;
		
		//Get the loader in use
		IUml2SDLoader currentLoader=null;
		if (map.get(id) instanceof IUml2SDLoader)
			currentLoader = (IUml2SDLoader)map.get(id);
		
		if ((currentLoader != null)&&(currentLoader!=loader))
		{
			if (loader != null) 
			{
				IWorkbenchPage persp =CommonUIPlugin.getDefault().
				   getWorkbench().getActiveWorkbenchWindow().getActivePage();
				try
				{
					//Search the view corresponfing to the viewId
					SDView sdview=null;
					IViewReference viewref = (IViewReference) persp.findViewReference(id);
					if (viewref!=null)
						sdview=(SDView)viewref.getView(false);
					
					//Make everthing clean for the new loader
					if (sdview != null) {
						sdview.resetProviders();
					}
				}
				catch (Exception e){
					e.printStackTrace();
				}
			}
			//The old loader is going to be kicked
			currentLoader.aboutToBeReplaced();
		}
		//Replace the current loader by the new one in the map
		map.put(id,loader);
		//Store this loader in the preferences to be able to restore it when the worbench will be relaunched
		if ((loader != null)&&(id!=null))
			SDUtil.getInstance().setLastLoader(loader.getClass().getName(),id);
	}

	/**
	 * Set the loader to null for this view, a kind of clean-up while disposing
	 * @param viewId the id of the view
	 */
	public void resetLoader(String viewId)
	{
		IUml2SDLoader loader = (IUml2SDLoader)map.get(viewId);
		if (loader != null) {
			loader.aboutToBeReplaced();
		}
		map.put(viewId, null);
		BackgroundLoader.setView(null);
	}
	
	/**
	 * Return the loader in use in this Sequence Diagram View 
	 * @param viewId the Sequence Diagram viewId
	 * Use null to reference the primary SD View
	 * @return the current loader if any - null otherwise
	 */
	public IUml2SDLoader getCurrentLoader(String viewId) {
		return getCurrentLoader(viewId, null);
	}

	/**
	 * Return the loader in use in this Sequence Diagram View 
	 * @param viewId the Sequence Diagram viewId
	 * @param view the Sequence Diagram view (if known)
	 * Use null to reference the primary SD View
	 * @return the current loader if any - null otherwise
	 */
	public IUml2SDLoader getCurrentLoader(String viewId, SDView view)
	{
		if (viewId == null)
			return null;
		
		IWorkbenchPage persp = CommonUIPlugin.getDefault().
		   getWorkbench().getActiveWorkbenchWindow().getActivePage();

		try 
		{
			//Search the view corresponding to the viewId
			if (view == null) {
				SDView v = null;
				IViewReference viewref = (IViewReference) persp.findViewReference(viewId);
				if (viewref!=null)
					v = (SDView)viewref.getView(false);
				if (v == null)
					return null;
			}

			//Return the loader corresponding to that view (if any)
			Object loader = map.get(viewId);
			if (loader == null)
			{
			 	createLastLoaderIfAny(viewId);
			    loader = map.get(viewId);
			}
			if (loader instanceof IUml2SDLoader) {
				BackgroundLoader.setView(view);
				return (IUml2SDLoader)loader;
			}
			else return null;
		}
	    catch (Exception e)
		{
	    	if (DebugUtil.debug()) {
	    		e.printStackTrace();
	    	}
		}
	    return null;
	}

	protected void createLastLoaderIfAny(String viewId) {
		String loader = SDUtil.getInstance().getSavedLoader(viewId);
		IConfigurationElement ce = null;
		List loaders = SDUtil.getLoaders(viewId);
		if (loader != null && loader.length() > 0) {
			for (Iterator i = loaders.iterator(); i.hasNext(); ) {
				ce = (IConfigurationElement)i.next();
				if (ce.getAttribute("class").equals(loader)) { //$NON-NLS-1$
					break;
				}
				ce = null;
			}
		} else {
			for (Iterator i = loaders.iterator(); i.hasNext(); ) {
				ce = (IConfigurationElement)i.next();
				if (Boolean.valueOf(ce.getAttribute("default")).booleanValue()) { //$NON-NLS-1$
					break;
				}
				ce = null;
			}
		}
		if (ce != null) {
			SDUtil.createLoaderForView(viewId, ce);
		}
	}

}
