 /**********************************************************************
 * Copyright (c) 2006, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CommonUIPlugin.java,v 1.11 2009/03/24 16:12:26 paules Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.tptp.platform.common.ui.internal;

import java.util.HashSet;
import java.util.Set;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.models.hierarchy.impl.HierarchyPackageImpl;
import org.eclipse.hyades.ui.HyadesUI;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialogWithToggle;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.internal.util.UIAgentControllerFactory;
import org.eclipse.tptp.platform.execution.util.internal.AgentControllerPool;
import org.eclipse.ui.IPerspectiveDescriptor;
import org.eclipse.ui.IPerspectiveRegistry;
import org.eclipse.ui.IPluginContribution;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.activities.IActivityManager;
import org.eclipse.ui.activities.IIdentifier;
import org.eclipse.ui.activities.IWorkbenchActivitySupport;
import org.eclipse.ui.activities.WorkbenchActivityHelper;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

import com.ibm.icu.text.MessageFormat;

/**
 * <p>Common UI plug-in class.</p>
 * 
 * <p><b>Note:</b> This class should not be used outside 
 * the context of Eclipse.</p>
 * 
 * 
 * @author  Paul Slauenwhite
 * @version March 24, 2009
 * @since   January 26, 2005
 */
public class CommonUIPlugin extends AbstractUIPlugin
{
	protected final static String KEYSTORE = "keystore.dat";
	
	/**
	 * Extension point id used to register navigator extensions.
	 */
	public static final String EP_NAVIGATOR_EXTENSIONS = "navigatorExtensions";
	
	/**
	 * The plug-in id
	 */
	public static final String PLUGIN_ID = "org.eclipse.tptp.platform.common.ui";
		
	/**
	 * Extension point id used to register the report extensions.
	 */
	public static final String EP_REPORT_EXTENSIONS = "reportExtensions";
	   	   
	/**
	 * The only instance of this singleton class
	 */
	protected static CommonUIPlugin plugin;

	/**
	 * The constructor
	 */
	public CommonUIPlugin()
	{
		plugin = this;
	}

	public static String getID()
	{
		return PLUGIN_ID;
	}
	
	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.plugin.AbstractUIPlugin#start(org.osgi.framework.BundleContext)
	 */
	public void start(BundleContext context) throws Exception
	{
		super.start(context);

		//Bug 188206 - initialize the hierarchy package so that the filter resource is loaded
		HierarchyPackageImpl.init();
				
		AgentControllerPool.setAgentControllerFactory(new UIAgentControllerFactory());
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.eclipse.ui.plugin.AbstractUIPlugin#stop(org.osgi.framework.BundleContext)
	 */
	public void stop(BundleContext context) throws Exception
	{
		plugin = null;
		super.stop(context);
	}

	/**
	 * Returns the shared instance
	 * 
	 * @return the shared instance
	 */
	public static CommonUIPlugin getDefault()
	{
		return plugin;
	}
	
	/**
	 * Asks the user whether they want to switch to the given
	 * perspective. Returns the user's response. If we are already
	 * in the given perspective, returns false. Must be called from
	 * the UI thread.
	 */
	public boolean openPerspectiveSwitchDialog(Shell shell, String perspectiveId, IPreferenceStore preferenceStore, String preferenceKey, String message) {
		if (isCurrentPerspective(PlatformUI.getWorkbench().getActiveWorkbenchWindow(), perspectiveId)) {
			return false;
		}
		String perspectiveName= getPerspectiveLabel(perspectiveId);
		if (perspectiveName == null) {
			return false;
		}
		String preferenceValue = preferenceStore.getString(preferenceKey);
		if (preferenceValue == null || preferenceValue.length() == 0) {
			preferenceValue = MessageDialogWithToggle.PROMPT;
			preferenceStore.setValue(preferenceKey, preferenceValue);
		}
		boolean saveAndChange = false;
		if (MessageDialogWithToggle.PROMPT.equals(preferenceValue)) {
			MessageDialogWithToggle dialog = MessageDialogWithToggle.openYesNoQuestion(shell, 
									CommonUIMessages._52, 
									MessageFormat.format(message, new String[] { perspectiveName }),
									null, false, preferenceStore, preferenceKey); //$NON-NLS-1$
			saveAndChange = (dialog.getReturnCode()== IDialogConstants.YES_ID);
		}

		if (!getContributions(perspectiveId)){
			return false;
		}
		preferenceValue = preferenceStore.getString(preferenceKey);
		if (MessageDialogWithToggle.ALWAYS.equals(preferenceValue)) {
			return true;
		}
		else if (!saveAndChange || MessageDialogWithToggle.NEVER.equals(preferenceValue)){
			return false;
		}
		return true;
	}
	
	/**
	 * Returns whether the given perspective identifier matches the
	 * identifier of the current perspective.
	 * 
	 * @param perspectiveId the identifier
	 * @return whether the given perspective identifier matches the
	 *  identifier of the current perspective
	 */
	protected boolean isCurrentPerspective(IWorkbenchWindow window, String perspectiveId) {
		boolean isCurrent= false;
		if (window != null) {
			IWorkbenchPage page = window.getActivePage();
			if (page != null) {
				IPerspectiveDescriptor perspectiveDescriptor = page.getPerspective();
				if (perspectiveDescriptor != null) {
					isCurrent= perspectiveId.equals(perspectiveDescriptor.getId());
				}
			}
		}
		return isCurrent;
	}	
	
	
	/**
	 * Check if required activities are enabled.
	 */
	protected boolean getContributions(String perspectiveId){
		// Map perspective id to descriptor.
        IPerspectiveRegistry reg = PlatformUI.getWorkbench().getPerspectiveRegistry();

        // leave this code in - the perspective of a given project may map to
        // activities other than those that the wizard itself maps to.
        IPerspectiveDescriptor finalPersp = reg.findPerspectiveWithId(perspectiveId);
        if (finalPersp != null && finalPersp instanceof IPluginContribution) {
            IPluginContribution contribution = (IPluginContribution) finalPersp;
            
            if (contribution.getPluginId() != null) {
                IWorkbenchActivitySupport workbenchActivitySupport = PlatformUI.getWorkbench().getActivitySupport();
                IActivityManager activityManager = workbenchActivitySupport.getActivityManager();
                IIdentifier identifier = activityManager.getIdentifier(WorkbenchActivityHelper.createUnifiedId(contribution));
                Set idActivities = identifier.getActivityIds();

                if (!idActivities.isEmpty()) {
                    Set enabledIds = new HashSet(activityManager.getEnabledActivityIds());

                    if (enabledIds.addAll(idActivities))
                        workbenchActivitySupport.setEnabledActivityIds(enabledIds);
                }
            }
        } else {
            CommonPlugin.logError("Unable to find perspective " + perspectiveId);//$NON-NLS-1$
            return false;
        }
        return true;
	}
	
	
	/**
	 * Returns the label of the perspective with the given identifier or
	 * <code>null</code> if no such perspective exists.
	 * 
	 * @param perspectiveId the identifier
	 * @return the label of the perspective with the given identifier or
	 *  <code>null</code> if no such perspective exists 
	 */
	protected String getPerspectiveLabel(String perspectiveId) {
		IPerspectiveDescriptor newPerspective = PlatformUI.getWorkbench().getPerspectiveRegistry().findPerspectiveWithId(perspectiveId);
		if (newPerspective == null) {
			return null;
		}
		return newPerspective.getLabel();
	}
	
	public String getWorkspaceName(){
		IPath workspace = Platform.getLocation();
		int nr = workspace.segmentCount();
		return workspace.segment(nr - 1);
	}
	
	public static String getKeyStoreLocation() {
		
		return Platform.getPluginStateLocation(plugin).append(KEYSTORE).toOSString();
	}

	protected void initializeDefaultPreferences(IPreferenceStore store) {
	 	store.setDefault(HyadesUI.SWITCH_TO_PERSPECTIVE_KEY, MessageDialogWithToggle.PROMPT);		
		store.setDefault(CommonUIConstants.LOCALHOST_PORT, "10002");
	}
	
	public static void log(Throwable e) {
		String msg = e.getMessage();
		getDefault().getLog().log(new Status(IStatus.ERROR, PLUGIN_ID, IStatus.ERROR, msg == null ? "" : msg, e));
	}	

}
