/**********************************************************************
  * Copyright (c)2007, 2008 IBM Corporation and others.
  * All rights reserved.   This program and the accompanying materials
  * are made available under the terms of the Eclipse Public License v1.0
  * which accompanies this distribution, and is available at
  * http://www.eclipse.org/legal/epl-v10.html
  * $Id:
  *
  * Contributors:
  * IBM - Initial API and implementation
  **********************************************************************/

package org.eclipse.tptp.platform.common.ui.internal.util;

import org.eclipse.core.runtime.Assert;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.jface.resource.JFaceResources;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.graphics.FontMetrics;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIMessages;
import org.eclipse.ui.PlatformUI;
/**
 * Helper class that displays a description and text box for inputting viewer page sizes. It's currently used by the log viewer, log interactions view and symptom results view page size preference pages.
 * @author apnan
 *
 */
public class CommonPageSizeUI implements ModifyListener{

	protected PreferencePage parent = null;
	protected String labelText = null;
	protected String info = null;
	protected String contextHelpID = null;
	protected Text txtPageSize = null;
	protected String initValue = "";
	
	/**
	 * 
	 * @param parent parent preference page
	 * @param info labels displaying a description related to what the function does
	 * @param labelText label for the page size text box
	 * @param contextHelpID context help ID
	 * @param initValue
	 */
	public CommonPageSizeUI(PreferencePage parent, String info, String labelText, String contextHelpID, String initValue){
		Assert.isNotNull(parent);
		Assert.isNotNull(labelText);
		this.parent = parent;
		this.labelText = labelText;
		this.info = info;
		this.contextHelpID = contextHelpID;
		this.initValue = initValue; 
	}
	public Control createContents(Composite parentComposite) {
		Composite composite = new Composite(parentComposite, SWT.NONE);

		if(contextHelpID != null){
			PlatformUI.getWorkbench().getHelpSystem().setHelp(
				parent.getControl(),
				contextHelpID);
		}

		GridLayout layout = new GridLayout();
		layout.marginWidth = 10;
		layout.marginHeight = 5;
		layout.numColumns = 2;
		composite.setLayout(layout);
		GridData data = new GridData();
		if(info != null){
			Label note = new Label(composite, SWT.WRAP);
			note.setText(info);
			data = new GridData();
			data.horizontalSpan=2;
			data.widthHint = 375;
			note.setLayoutData(data);

		}
		
		Label dummy = new Label(composite, SWT.NULL);
		data = new GridData();
		data.horizontalSpan=2;
		data.grabExcessHorizontalSpace=true;
		dummy.setLayoutData(data);
		
		Label label = new Label(composite, SWT.NULL);
		label.setText(labelText);
		txtPageSize = new Text(composite, SWT.BORDER);
		txtPageSize.setTextLimit(5);
		data = new GridData();
		data.widthHint = convertWidthInCharsToPixels(7, parentComposite);
		txtPageSize.setLayoutData(data);
		txtPageSize.setText(initValue!=null ? initValue : "");
		txtPageSize.addModifyListener(this);
					    		
		return composite;
	}
	
    protected int convertWidthInCharsToPixels(int chars, Control parent) {
        // Compute the required number of pixels for a number of characters based on a font metric 
        GC gc = new GC(parent);
        gc.setFont(JFaceResources.getDialogFont());
        FontMetrics fontMetrics = gc.getFontMetrics();
        gc.dispose();

        if (fontMetrics == null) {
			return 0;
		}
        return Dialog.convertWidthInCharsToPixels(fontMetrics, chars);
    }
	
	public String getPageSize(){
		return txtPageSize.getText();
	}
	
	public void setPageSize(String newValue){
		if(newValue == null){
			return;
		}
		txtPageSize.setText(newValue);
	}
	
	
	/* (non-Javadoc)
	 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
	 */
	public void modifyText(ModifyEvent e) {
		IStatus status = validateInput();
		parent.setValid(status.getSeverity()==IStatus.OK);
		if(status.getSeverity()==IStatus.ERROR)	
			parent.setErrorMessage(status.getMessage());
		else
			parent.setErrorMessage(null);
	}
	
	protected IStatus validateInput(){
		
		String nrOfPages = txtPageSize.getText();
		
		PreferenceStatus status = new PreferenceStatus();
		
		if (nrOfPages.length() == 0) {			
			status.setMessage(CommonUIMessages.emptyPageSizeErr);
			status.setSeverity(IStatus.ERROR);
		} else {
			try {
				int value= Integer.parseInt(nrOfPages);
				if (value <= 0) {
					status.setMessage(NLS.bind(CommonUIMessages.invalidPageSizeErr,  new String[] {nrOfPages}));
					status.setSeverity(IStatus.ERROR);
				}
			} catch (NumberFormatException e) {
				status.setMessage(NLS.bind(CommonUIMessages.invalidPageSizeErr,  new String[] {nrOfPages}));
				status.setSeverity(IStatus.ERROR);
			}
		}
		
		return status;

	}
	
	class PreferenceStatus extends Status{
		
		public PreferenceStatus(){
			super(OK, "tptp.platform.common.ui", 0, "", null);
		}
		
		
		/* (non-Javadoc)
		 * @see org.eclipse.core.runtime.Status#setMessage(java.lang.String)
		 */
		protected void setMessage(String message) {
			// TODO Auto-generated method stub
			super.setMessage(message);
		}

		/* (non-Javadoc)
		 * @see org.eclipse.core.runtime.Status#setSeverity(int)
		 */
		protected void setSeverity(int severity) {
			// TODO Auto-generated method stub
			super.setSeverity(severity);
		}

	}	
}
