/*******************************************************************************
 * Copyright (c) 2006, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: UIUtil.java,v 1.10 2010/02/24 15:39:19 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.tptp.platform.common.ui.internal.util;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIMessages;
import org.eclipse.ui.IEditorDescriptor;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IElementFactory;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.ide.IDE;
import org.eclipse.ui.part.FileEditorInput;
import org.osgi.framework.Bundle;

/**
 * <p>UI utility methods.</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Valentina Popescu
 * @author  Paul E. Slauenwhite
 * @version February 24, 2010
 * @since   August 16, 2006
 */
public class UIUtil{
	
	/**
	 * <p>Resolves the {@link ImageDescriptor} for an image located at the 
	 * plug-in-relative path.</p>
	 * 
	 * <p>The plug-in-relative path <b>must</b> not contain any leading path 
	 * separator characters.  For example:</p>
	 * 
	 * <p><code>icons/myIcon.gif</code></p>
	 * 
	 * <p>If the {@link ImageDescriptor} cannot be resolved, an error message 
	 * is logged and <code>null</code> is returned.</p>
	 *
	 * <p>Note: This method is intended to be used in a workbench run-time 
	 * since it explicitly uses the workbench's registry for caching and retrieving 
	 * images.</p>
	 * 
	 * @param plugin The plug-in containing the image.
	 * @param path The plug-in-relative path of the image
	 * @return The {@link ImageDescriptor} for an image located at the plug-in-relative path, otherwise <code>null</code>.
	 */
	public static ImageDescriptor getImageDescriptorFromPlugin(Bundle plugin, String path){
				
		try{
			return (ImageDescriptor.createFromURL(new URL(plugin.getEntry("/"), path))); //$NON-NLS-1$
		}
		catch (MalformedURLException m){
			CommonPlugin.logError(NLS.bind(CommonUIMessages.UIUtil_ERROR_RESOLVING_IMAGE, path, plugin.getSymbolicName()));
		}
		
		return null;
	}
	
	
	/**
	 * Returns whether the two selections have the same elements.
	 * @param selection1
	 * @param selection2
	 * @return boolean
	 */
	public static boolean areEquals(ISelection selection1, ISelection selection2)
	{
		if(selection1 == selection2)
			return true;
			
		if((selection1 instanceof IStructuredSelection) && (selection2 instanceof IStructuredSelection))
		{
			List list1 = new ArrayList(((IStructuredSelection)selection1).toList());
			List list2 = new ArrayList(((IStructuredSelection)selection2).toList());
			
			if(list1.size() == list2.size())
			{
				list1.removeAll(list2);
				return list1.isEmpty();
			}
		}
		
		return false;	
	}
	
	/**
	 * Returns the {@link IElementFactory} registered with the specified id.  This
	 * methods looks for the factory in the plugin registry returning <code>null</code> 
	 * if no factory was found.
	 * @param factoryId
	 * @return IElementFactory
	 */
	public static IElementFactory getElementFactory(String factoryId)
	{
		return PlatformUI.getWorkbench().getElementFactory(factoryId);
	}
	
	/**
	 * Calculates the size of one line of text in a composite. 
	 * @param parent
	 * @return int
	 */
	public static int availableRows(Composite parent)
	{
		int fontHeight = (parent.getFont().getFontData())[0].getHeight();
		int displayHeight = parent.getDisplay().getClientArea().height;

		return displayHeight / fontHeight;
	}
	
	/**
	 * Returns all the nodes that are currently visible in the given tree
	 * viewer. An element is visible if its parent and all ancestors are
	 * expanded. This is useful for maintaining the expanded state of a tree
	 * after performing an operation that collapses it.
	 * 
	 * @param viewer The viewer containing the tree of elements.
	 * @return All the visible elements showing in the tree.
	 */
	public static Set getVisibleViewerNodes(TreeViewer viewer) {
		Set set = new HashSet();
		ITreeContentProvider content = (ITreeContentProvider)viewer.getContentProvider();
		
		// Recurse down all the top-level elements
		Object[] top = content.getElements(viewer.getInput());
		for (int i=0;i<top.length;++i) {
			collectVisibleElements(viewer, content, top[i], set);
		}
		return set;
	}
	
	/**
	 * Recursively finds and collects all the visible elements in the viewer.
	 * An element is visible if its parent and all ancestors are expanded.
	 * 
	 * @param viewer The tree viewer containing the tree.
	 * @param content The content provider for the viewer.
	 * @param current The current element in the tree being traversed.
	 * @param set The set of visible elements to collect the elements into.
	 */
	protected static void collectVisibleElements(TreeViewer viewer, ITreeContentProvider content, Object current, Set set) {
		set.add(current);
		if (viewer.getExpandedState(current)) {
			Object[] children = content.getChildren(current);
			for (int i=0;i<children.length;++i) {
				collectVisibleElements(viewer, content, children[i], set);
			}
		}
	}
	
	/**
	 * Expands the tree contained in the given viewer in such a way that all
	 * the given elements are visible. This is useful for maintaining the
	 * expanded state of a tree after performing an operation that collapses
	 * it.
	 * 
	 * @param viewer The viewer containing the tree of elements.
	 * @param elements The elements to be made visible.
	 */
	public static void setVisibleViewerNodes(TreeViewer viewer, Set set) {
		/*
		 * This is necessary to force the creation of all the tree nodes. You
		 * cannot expand a node that hasn't been seen yet; this is a limitation
		 * of the JFace TreeViewer.
		 */
		viewer.expandAll();
		viewer.collapseAll();
		
		/*
		 * For each previously visible node, traverse through each top level
		 * node in the tree to search for it, while expanding its ancestors
		 * along the way, in order to make it visible.
		 */
		Iterator iter = set.iterator();
		while (iter.hasNext()) {
			Object obj = iter.next();
			ITreeContentProvider content = (ITreeContentProvider)viewer.getContentProvider();
			Object[] top = content.getElements(viewer.getInput());
			for (int i=0;i<top.length;++i) {
				hasElement(viewer, content, top[i], obj);
			}
		}
	}
	
	/**
	 * Returns whether or not the target node is a descendant of the current node,
	 * and if so, expands the current node. This is used to expand the tree in such
	 * a way that the target node is visible.
	 * 
	 * @param viewer The tree viewer containing the tree to traverse.
	 * @param content The viewer's content provider.
	 * @param current The current node being traversed.
	 * @param target The node being searched for and made visible.
	 * @return Whether or not the target node is a descendant of the current node.
	 */
	protected static boolean hasElement(TreeViewer viewer, ITreeContentProvider content, Object current, Object target) {
		if (current == target) {
			return true;
		}
		else {
			// It has the target if one of its children has it
			Object[] children = content.getChildren(current);
			boolean found = false;
			for (int i=0;i<children.length;++i) {
				if (hasElement(viewer, content, children[i], target)) {
					found = true;
				}
			}
			// Expand the current node if it contains the target
			if (found) {
				viewer.setExpandedState(current, true);
			}
			return found;
		}
	}

	/**
	 * Returns an active workbench page.
	 * @return IWorkbenchPage
	 */
	public static IWorkbenchPage getActiveWorkbenchPage()
	{
		IWorkbench workbench = PlatformUI.getWorkbench();
		IWorkbenchPage activePage = null;
		if(workbench.getActiveWorkbenchWindow() != null)
		{
			activePage = workbench.getActiveWorkbenchWindow().getActivePage();
		}
		else if(workbench.getWorkbenchWindows().length > 0)
		{
			activePage = workbench.getWorkbenchWindows()[0].getActivePage();
		}
		
		return activePage;
	}
	
	
	/**
	 * Shows an error dialog to present to the user why a file saving attempt
	 * has failed.  
	 * @param filePath
	 * @param throwable
	 */
	public static void openSaveFileErrorDialog(Shell shell, String filePath, Throwable throwable)
	{
		if(filePath == null)
			filePath = ""; //$NON-NLS-1$
		
		
		String error = ""; //$NON-NLS-1$
		if(throwable != null)
		{
			if(throwable.getLocalizedMessage() != null)
				error = throwable.getLocalizedMessage();
			else if(throwable.getMessage() != null)
				error = throwable.getMessage();
		}
		
		String message = NLS.bind(CommonUIMessages._48, new String[]{filePath, error});
		MessageDialog.openError(shell, CommonUIMessages._8, message);
	}
	
	/**
	 * <p>Opens and returns the editor for the specified {@link IFile} 
	 * based on the specified editor identifier.</p>
	 * 
	 * <p>If editor identifier is <code>null</code> or an empty string, the identifier for the 
	 * default editor is used.</p>
	 *  
	 * <p>If substring is true then the editor identifier is the first segment of the editor identifier.</p>
	 * 
	 * @param file The {@link IFile} to be opened in the editor.
	 * @param editorId The editor identifier.
	 * @param substring <code>true</code> if the editor identifier is the first segment of the editor identifier, otherwise <code>false</code>.
	 * @return The opened {@link IEditorPart}, otherwise <code>null</code>.
	 * @see   IDE#openEditor(IWorkbenchPage, IFile, boolean)
	 * @see   IDE#openEditor(IWorkbenchPage, IEditorInput, String)
	 */
	public static IEditorPart openEditor(IFile file, String editorId, boolean substring)
	{
		if(file != null){
				
			IWorkbenchPage activeWorkbenchPage = getActiveWorkbenchPage();
			
			if(activeWorkbenchPage != null){
			
				try{
					
			    	//Refresh the file:
			    	file.refreshLocal(IResource.DEPTH_ZERO, null);
		
					if((editorId == null) || (editorId.trim().length() == 0)){
						return (IDE.openEditor(activeWorkbenchPage, file, true));
					}
					else{
						
						IEditorDescriptor[] editors = PlatformUI.getWorkbench().getEditorRegistry().getEditors(file.getName());
						
						if(editors != null){
							
							boolean editorFound = false;
							
							for(int counter = 0; counter < editors.length; counter++){
								
								if(substring){
									editorFound = editors[counter].getId().startsWith(editorId);
								}
								else{
									editorFound = editorId.equals(editors[counter].getId());
								}
									
								if(editorFound){
									return (IDE.openEditor(activeWorkbenchPage, new FileEditorInput(file), editors[counter].getId()));
								}
							}
						}				
					}
				}
				catch(Exception e){
					CommonPlugin.logError(e);
				}
			}
		}
		
		return null;
	}
	
	/**
	 * <p>Opens and returns the editor for the specified editor input based on the 
	 * specified editor identifier.</p>
	 *   
	 * <p>The editor is resolved by the specified editor identifier.  If the editor 
	 * identifier is <code>null</code>, empty, or invalid, <code>null</code> is 
	 * returned.</p>
	 *  
	 * <p>If the active workbench page already has an editor open on the specified 
	 * editor input, it is brought to front and returned.  Otherwise, a new editor is 
	 * opened and returned.</p>
	 * 
	 * <p>If the editor cannot be opened, this method logs the exception in the Eclipse 
	 * <code>.log</code> file and <code>null</code> is returned.</p>  
	 * 
	 * @param editorInput The editor input to be opened in the editor.
	 * @param editorId The non-<code>null</code> editor identifier.
	 * @return The opened editor, otherwise <code>null</code>.
	 * @see IDE#openEditor(IWorkbenchPage, IEditorInput, String)
	 */
	public static IEditorPart openEditor(IEditorInput editorInput, String editorId){
		
		if((editorInput != null) && (editorId != null) && (editorId.trim().length() > 0)){

			IWorkbenchPage activeWorkbenchPage = getActiveWorkbenchPage();

			if(activeWorkbenchPage != null){
				
				try {
					return (IDE.openEditor(activeWorkbenchPage, editorInput, editorId.trim()));
				}
				catch(Exception e){
					CommonPlugin.logError(e);
				}
			}
		}

		return null;
	}
}