/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TraceUIManager.java,v 1.2 2008/01/24 02:29:14 apnan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.tptp.platform.common.ui.trace.internal;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.ListenerList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.hyades.models.hierarchy.HierarchyFactory;
import org.eclipse.hyades.models.hierarchy.impl.HierarchyFactoryImpl;
import org.eclipse.hyades.models.hierarchy.impl.HierarchyPackageImpl;
import org.eclipse.hyades.models.hierarchy.util.HierarchyResourceSetImpl;
import org.eclipse.hyades.trace.internal.ui.TraceSelection;
import org.eclipse.hyades.trace.ui.IDeleteListener;
import org.eclipse.hyades.trace.ui.IProfileEventListener;
import org.eclipse.hyades.trace.ui.ITraceSelection;
import org.eclipse.hyades.trace.ui.IViewSelectionChangedListener;
import org.eclipse.hyades.trace.ui.ProfileEvent;
import org.eclipse.hyades.trace.ui.ViewSelectionChangedEvent;
import org.eclipse.hyades.ui.extension.INavigatorItem;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;
import org.eclipse.ui.ISelectionListener;
import org.eclipse.ui.IWorkbenchPart;

/**
 * @author popescu
 */
public class TraceUIManager implements ISelectionChangedListener
{
	protected ResourceSet resourceSet;

	protected HierarchyFactory factory;

	protected static TraceUIManager traceUIManager;

	protected ProfileEvent fProfileEvent = new ProfileEvent();

	/**
	 * The collection of profile event listeners.
	 */
	protected ListenerList fProfileEventListeners = new ListenerList();

	/**
	 * list of delete listeners
	 */
	protected ArrayList fDeleteListener = new ArrayList(1);

	protected HashMap fSelectionModel = new HashMap();

	/**
	 * The selection providers for the trace UI
	 */
	protected List fSelectionProviders = new ArrayList(2);
	
	/**
	 * The desktop parts that contain the selections
	 */
	protected List fSelectionParts = new ArrayList(2);
	
	
	/**
	 * The list of selection listeners for the trace UI
	 */
	protected ListenerList fListeners = new ListenerList();
	
	protected ProfileEvent refreshViewsEvent = new ProfileEvent(ProfileEvent.REFRESH_VIEWS);
	
	protected ProfileEvent refreshOptionEvent = new ProfileEvent(ProfileEvent.REFRESH_OPTION);
	
	protected ViewSelectionChangedEvent pdViewEvent = new ViewSelectionChangedEvent();
	
	/**
	 * The collection of problem determination view event listeners.
	 */
	protected ListenerList fPDViewEventListeners = new ListenerList();
	
	
	protected ProfileEvent updateModelEvent = new ProfileEvent(
			ProfileEvent.UPDATE_MODEL);
	
	
	protected ProfileEvent contextHandlerSelectionChangedEvent = new ProfileEvent(
			ProfileEvent.CONTEXT_HANDLER_SELECTION_CHANGED);
	
	
	
	protected TraceUIManager()
	{
	}

	
	/**
	 * Adds an <code>ISelectionListener</code> to the trace selection manager.
	 */
	public void addSelectionListener(ISelectionListener l) {
		fListeners.add(l);
	}
	
	/**
	 * Remove an <code>ISelectionListener</code> to the trace selection
	 * manager.
	 */
	public void removeSelectionListener(ISelectionListener l) {
		fListeners.remove(l);
	}
	
	
	/**
	 * 
	 * @return traceUIManager singleton
	 */
	public static TraceUIManager getTraceUIManager()
	{
		if (traceUIManager == null)
			traceUIManager = new TraceUIManager();

		return traceUIManager;
	}

	public ResourceSet getResourceSet()
	{

		if (resourceSet == null)
		{
			HierarchyPackageImpl.init();
			factory = new HierarchyFactoryImpl();

			// init resourceSet
			resourceSet = HierarchyResourceSetImpl.getInstance();
		}

		return resourceSet;
	}

	public ProfileEvent getProfileEvent()
	{
		return fProfileEvent;
	}

	public void notifyProfileEventListener(final ProfileEvent event)
	{
		CommonUIPlugin.getDefault().getWorkbench().getDisplay().syncExec(new Runnable()
		{
			public void run()
			{
				Object[] listeners = fProfileEventListeners.getListeners();
				for (int idx = 0; idx < listeners.length; idx++)
				{
					((IProfileEventListener) listeners[idx]).handleProfileEvent(event);
				}				
			}
		});
	}

	/**
	 * Insert the method's description here. Creation date: (5/25/2001 3:25:42
	 * PM)
	 * 
	 * @return java.util.ArrayList
	 */
	public ArrayList getDeleteListeners()
	{
		return fDeleteListener;
	}

	public void addDeleteListener(IDeleteListener listener)
	{
		fDeleteListener.add(listener);
	}

	public void removeDeleteListener(IDeleteListener listener)
	{
		fDeleteListener.remove(listener);
	}

	/**
	 * 
	 * @param mofObject
	 * @return SelectionModel
	 */
	public ITraceSelection getSelectionModel(EObject mofObject)
	{

		Object model = fSelectionModel.get(mofObject);
		if (model == null)
		{
			model = new TraceSelection();
			fSelectionModel.put(mofObject, model);
		}

		return (ITraceSelection) model;
	}

	/**
	 * 
	 * @param mofObject
	 * @param model
	 */
	public void addSelectionModel(EObject mofObject, ITraceSelection model)
	{
		fSelectionModel.put(mofObject, model);
	}

	/**
	 * 
	 * @param mofObject
	 */
	public void removeSelectionModel(EObject mofObject)
	{
		fSelectionModel.remove(mofObject);
	}
	
	/**
	 * Adds the given listener to the collection of registered profile event
	 * listeners. Has no effect if an identical listener is already registered.
	 * 
	 * @param listener
	 *            the listener to add
	 */
	public void addProfileEventListener(IProfileEventListener listener) {
		fProfileEventListeners.add(listener);
	}

	/**
	 * Removes the given listener from the collection of registered profile
	 * event listeners. Has no effect if an identical listener is not already
	 * registered.
	 * 
	 * @param listener
	 *            the listener to remove
	 */
	public void removeProfileEventListener(IProfileEventListener listener) {
		fProfileEventListeners.remove(listener);
	}
	
	/**
	 * Dispose all resources on this plugin shut down
	 */
	public void dispose() 
	{
		fListeners.clear();
		fDeleteListener.clear();
		resourceSet = null;
		factory = null;
		fSelectionParts.clear();
		fDeleteListener.clear();
		fProfileEventListeners.clear();
		fSelectionModel.clear();
		fProfileEvent = null;
		refreshViewsEvent = null;
		traceUIManager = null;
		pdViewEvent = null;
		updateModelEvent = null;
		contextHandlerSelectionChangedEvent = null;
		fPDViewEventListeners.clear();
		
		fSelectionProviders.clear();
	}
	
	/**
	 * Adds the selection provider for the debug UI.
	 */
	public void addSelectionProvider(ISelectionProvider provider,
			IWorkbenchPart part) {
		fSelectionProviders.add(provider);
		fSelectionParts.add(part);
		provider.addSelectionChangedListener(this);
	}
	

	/**
	 * Removes the selection provider from the debug UI.
	 */
	public void removeSelectionProvider(ISelectionProvider provider,
			IWorkbenchPart part) {
		fSelectionProviders.remove(provider);
		fSelectionParts.remove(part);
		provider.removeSelectionChangedListener(this);
		selectionChanged(null);
	}
	
	/*
	 * Viewers don't understand INavigatorItems; replacing them with their
	 * corresponding data (getData()).
	 */
	protected static ISelection convertNavigatorItems(ISelection sel) {
		if (sel != null && sel instanceof IStructuredSelection) {
			IStructuredSelection s = (IStructuredSelection) sel;
			List list = new ArrayList();
			Iterator iter = s.iterator();
			while (iter.hasNext()) {
				Object obj = iter.next();
				if (obj instanceof INavigatorItem) {
					list.add(((INavigatorItem) obj).getData());
				} else {
					list.add(obj);
				}
			}
			return new StructuredSelection(list);
		}
		return sel;
	}
	
	
	/**
	 * Selection has changed in the debug selection provider. Notify the
	 * listeners.
	 */
	public void selectionChanged(SelectionChangedEvent event) {
		Object[] copiedListeners = fListeners.getListeners();
		ISelection selection = null;
		ISelectionProvider provider = null;
		IWorkbenchPart part = null;
		if (event != null) {
			selection = convertNavigatorItems(event.getSelection());
			provider = (ISelectionProvider) event.getSource();
			int index = fSelectionProviders.indexOf(provider);
			if (index == -1) {
				return;
			}
			part = (IWorkbenchPart) fSelectionParts.get(index);
			selectionChanged(part, selection);
		} else {

			return;
		}
		for (int i = 0; i < copiedListeners.length; i++) {
			ISelectionListener l = ((ISelectionListener) copiedListeners[i]);
			l.selectionChanged(part, selection);
		}

	}
	
	public void selectionChanged(IWorkbenchPart part, ISelection sel) 
	{
	}
	
	/**
	 * @return ProfileEvent
	 */
	public ProfileEvent getRefreshViewEvent(Object source) {
		refreshViewsEvent.setSource(source);
		return refreshViewsEvent;
	}
	
	/**
	 * @author Navid Mehregani - bugzilla 153453
	 * @return ProfileEvent
	 */
	public ProfileEvent getRefreshOptionEvent(Object source) {
		refreshOptionEvent.setSource(source);
		return refreshOptionEvent;
	}
	
	public HierarchyFactory getPerftraceFactory() {
		return factory;
	}
	
	public ViewSelectionChangedEvent getViewSelectionChangedEvent() {
		return pdViewEvent;
	}
	/**
	 * 
	 * @param event
	 */
	public void notifyViewSelectionChangedListener(
			ViewSelectionChangedEvent event) {
		Object[] listeners = fPDViewEventListeners.getListeners();
		for (int idx = 0; idx < listeners.length; idx++) {
			((IViewSelectionChangedListener) listeners[idx])
					.handleViewSelectionChangedEvent(event);
		}
	}
	
	/**
	 * Adds the given listener to the collection of registered pd view event
	 * listeners. Has no effect if an identical listener is already registered.
	 * 
	 * @param listener
	 *            the listener to add
	 */
	public void addViewSelectionChangedListener(
			IViewSelectionChangedListener listener) {
		fPDViewEventListeners.add(listener);
	}
	
	/**
	 * Removes the given listener from the collection of registered pd view
	 * event listeners. Has no effect if an identical listener is not already
	 * registered.
	 * 
	 * @param listener
	 *            the listener to remove
	 */
	public void removeViewSelectionChangedListener(
			IViewSelectionChangedListener listener) {
		fPDViewEventListeners.remove(listener);
	}
	
	/**
	 * @return ProfileEvent
	 */
	public ProfileEvent getUpdateModelEvent(Object source) {
		updateModelEvent.setSource(source);
		return updateModelEvent;
	}
	
	/**
	 * @return ProfileEvent
	 */
	public ProfileEvent getContextHandlerSelectionChangedEvent(Object source) {
		contextHandlerSelectionChangedEvent.setSource(source);
		return contextHandlerSelectionChangedEvent;
	}
}
