/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: LocationPage.java,v 1.6 2010/05/28 16:05:32 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.tptp.platform.common.ui.wizard;

import java.net.URL;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.ui.util.LocationValidator;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IMessageProvider;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.DecoratingLabelProvider;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.IFilter;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.WizardDialog;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.osgi.util.NLS;
import org.eclipse.osgi.util.TextProcessor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.swt.widgets.MenuItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.ToolBar;
import org.eclipse.swt.widgets.ToolItem;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIImages;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIMessages;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.NewProjectAction;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.dialogs.WizardNewFileCreationPage;
import org.eclipse.ui.help.IWorkbenchHelpSystem;
import org.eclipse.ui.ide.undo.CreateFileOperation;
import org.eclipse.ui.ide.undo.WorkspaceUndoUtil;
import org.eclipse.ui.model.WorkbenchLabelProvider;
import org.eclipse.ui.part.DrillDownComposite;
import org.eclipse.ui.views.navigator.ResourceComparator;
import org.eclipse.ui.wizards.newresource.BasicNewFolderResourceWizard;

/**
 * <p>Wizard location page for specifying/selecting a file with a specific file extension 
 * in a new/existing container (project or folder) within the workspace.</p>
 * 
 * <p>The location page provides the following common features:</p>
 * 
 * <ul>
 * <li>Common UI (based on {@link WizardNewFileCreationPage}) for a common user experience.</li>
 * <li>New project/folder drop-drop with menus for creating a category-specific (see {@link #setNewProjectWizardCategory(String)}) projects 
 * and folders directly from the location page.</li>
 * <li>Page validation (see {@link #validatePage(boolean)}).</li>
 * <li>Filtering (see {@link #addFilter(IFilter)})</li>
 * <li>Displaying existing files (see {@link #setDisplayFiles(boolean)}) with the same file extension (see {@link #getFileExtension()}) with a customized image (see {@link #setFileImage(Image)}).</li>
 * <li>Overwriting existing files (see {@link #setAllowOverwrite(boolean)}) with the same file extension (see {@link #getFileExtension()}) .</li>
 * </ul>
 *
 * <p>Extensions include:</p>
 * 
 * <ul>
 * <li>Customized Context Sensitive Help (CSH) (see {@link IWorkbenchHelpSystem#setHelp(org.eclipse.swt.widgets.Control, String)}) on the page's control (see {@link #createControl(Composite)}, {@link #setControl(Control)}, and {@link #getControl()}).</li>
 * <li>Custom page validation (see {@link #validatePage(boolean)}) and custom file name validation (see {@link #setNameValidator(ICustomLocationNameValidation)}).</li>
 * <li>Adding advanced controls to the top (requires calling {@link Control#moveAbove()} with <code>null</code> on the {@link Composite} containing the advance controls) or bottom of the location page (see {@link #createAdvancedControls(Composite)}).</li>
 * <li>Pre-selecting an existing container (project or folder) (see {@link #setContainerFullPath(IPath)}) and/or file name (see {@link #setFileName(String)}).</li>
 * </ul>
 * 
 * <p><b>Note:</b>This class no longer subclasses {@link WizardNewFileCreationPage}.</p>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @author  Marcelo Paternostro
 * @author  Jerome Bozier
 * @version May 28, 2010
 * @since   August 16, 2006
 * @see     WizardPage
 */
public class LocationPage extends WizardPage implements Listener {

	//Note: This class was derived from org.eclipse.ui.dialogs.WizardNewFileCreationPage.

	private Text containerFullPathText = null;

	private TreeViewer treeViewer = null;

	private Text fileNameText = null;

	private List filters = null;

	private ICustomLocationNameValidation nameValidator = null;

	private IStructuredSelection initialSelection = null;

	private IPath initialContainerFullPath = null;

	private String initialFileName = null;

	private String fileExtension = null;

	private Image fileImage = null;

	private boolean allowOverwrite = false;

	private boolean displayFiles = false;

	private String newProjectWizardCategory = null;

	private IWorkspaceRoot workspaceRoot = null;

	private static final int TREE_HEIGHT_HINT = 250;

	private static final int TEXT_WIDTH_HINT = 200;

	/**
	 * <p>Constructor.</p>
	 * 
	 * @param pageName The name of the page.
	 * @param selection The initial selection, otherwise <code>null</code>/
	 */
	public LocationPage(String pageName, IStructuredSelection initialSelection) {

		super(pageName);

		this.initialSelection = initialSelection;
		this.workspaceRoot = ResourcesPlugin.getWorkspace().getRoot();
	}

	/**
	 * <p>Resolves the absolute (see {@link IPath#makeAbsolute()})/de-processed (see {@link TextProcessor#deprocess(String)}) 
	 * {@link IPath} of the selected container (project or folder).</p>
	 * 
	 * <p>If called before the location page is created (see {@link #createControl(Composite)}), the 
	 * {@link IPath} of the initial container (project or folder) is returned.</p>
	 * 
	 * @return The {@link IPath} of the initial or selected container (project or folder), otherwise <code>null</code>.
	 * @see #setContainerFullPath(IPath)
	 */
	public IPath getContainerFullPath() {

		if(containerFullPathText != null){

			String containerFullPath = containerFullPathText.getText();

			if (containerFullPath.length() > 0) {
				return (new Path(TextProcessor.deprocess(containerFullPath))).makeAbsolute();			
			}
		}

		return initialContainerFullPath;
	}

	/**
	 * @deprecated As of TPTP 4.7.0, use {@link #getFileName()}. 
	 */
	public String getItemName() {
		return (getFileName());
	}

	/**
	 * <p>Resolves the specified/selected file name with the file extension (see {@link #getFileExtension()}).</p>
	 * 
	 * <p>If called before the location page is created (see {@link #createControl(Composite)}), the 
	 * initial file name is returned.</p>
	 * 
	 * @return The specified/selected or initial file name with the file extension, otherwise <code>null</code>.
	 * @see #setFileName(String)
	 */
	public String getFileName(){

		if(fileNameText != null){
			return (addFileExtension(fileNameText.getText()));
		}

		return initialFileName;
	}

	/**
	 * <p>Resolves the specified/selected file name without the file extension (see {@link #getFileExtension()}).</p>
	 * 
	 * <p>If called before the location page is created (see {@link #createControl(Composite)}), the 
	 * initial file name is returned.</p>
	 * 
	 * @return The specified/selected or initial file name without the file extension, otherwise <code>null</code>.
	 * @see #setFileName(String)
	 */
	public String getFileNameWithoutExtension(){

		if(fileNameText != null){
			return (removeFileExtension(fileNameText.getText()));
		}

		return initialFileName;
	}

	/**
	 * <p>Resolves the specified/selected absolute (see {@link IPath#makeAbsolute()})/de-processed (see {@link TextProcessor#deprocess(String)}) 
	 * workspace path of the specified/selected file name with the file extension (see {@link #getFileExtension()}).</p>
	 * 
	 * <p>If called before the location page is created (see {@link #createControl(Composite)}) or the container (project or folder) 
	 * (see {@link #getContainerFullPath()}) or file name (see {@link #getFileName()}) is not specified/selected, <code>null</code> 
	 * is returned.</p>
	 * 
	 * @return The workspace path of the specified/selected container (project or folder) and file name with the file extension, otherwise <code>null</code>.
	 */
	public String getFilePath(){

		IPath containerFullPath = getContainerFullPath();
		String fileName = getFileName();

		if((containerFullPath != null) && (fileName != null) && (fileName.length() > 0)){
			return (containerFullPath.append(fileName).toString());
		}

		return null;
	}

	/**
	 * <p>Resolves the file extension.</p>
	 * 
	 * @return The file extension, otherwise <code>null</code>.
	 * @see #setFileExtension(String)}
	 */
	public String getFileExtension() {
		return fileExtension;
	}

	/**
	 * <p>Resolves if files with the same file extension (see {@link #getFileExtension()}) are displayed.</p>
	 * 
	 * @return <code>true</code> if files with the same file extension are displayed, otherwise <code>false</code>.
	 * @see #setDisplayFiles(boolean)
	 */
	public boolean getDisplayFiles() {
		return displayFiles;
	}

	/**
	 * <p>Resolves if files can be overwritten.</p>
	 * 
	 * @return <code>true</code> if files can be overwritten, otherwise <code>false</code>.
	 * @see #setAllowOverwrite(boolean)
	 */
	public boolean allowOverwrite() {
		return allowOverwrite;
	}

	/**
	 * <p>Sets the initial {@link IPath} of the container (project or folder).</p>
	 * 
	 * <p>If called before the location page is created (see {@link #createControl(Composite)}), the 
	 * {@link IPath} is used when creating the location page.</p>
	 * 
	 * @param containerFullPath The initial {@link IPath} of the container (project or folder).
	 * @see #getContainerFullPath()
	 */
	public void setContainerFullPath(IPath containerFullPath) {

		if(containerFullPathText != null){

			IContainer container = getContainer(containerFullPath);

			if(container != null){
				updateSelection(new StructuredSelection(container));	
			}
			else{
				updateContainerFullPath(containerFullPath);
			}
		}
		else{
			this.initialContainerFullPath = containerFullPath;
		}
	}
	
	/**
	 * <p>Sets the initial file name without the file extension (see {@link #getFileExtension()}).</p>
	 * 
	 * <p>If called before the location page is created (see {@link #createControl(Composite)}), the 
	 * file name is used when creating the location page.</p>
	 * 
	 * @param fileName The initial file name without the file extension.
	 * @see #getFileName()
	 */
	public void setFileName(String fileName) {

		if(fileNameText != null){

			IFile file = getFile(fileName);

			if(file != null){
				updateSelection(new StructuredSelection(file));	
			}
			else{
				updateFileName(fileName);
			}
		}
		else{
			this.initialFileName = fileName;
		}
	}

	/**
	 * <p>Sets the file extension.</p>
	 * 
	 * @param fileExtension The file extension.
	 * @see #getFileExtension()
	 */
	public void setFileExtension(String fileExtension) {
		this.fileExtension = fileExtension;
	}

	/**
	 * <p>Sets the file image for displayed files (see {@link #getDisplayFiles()}) with the same file extension 
	 * (see {@link #getFileExtension()}).</p>
	 * 
	 * @param fileImage The file image for displayed files with the same file extension, otherwise <code>null</code>.
	 */
	public void setFileImage(Image fileImage) {
		this.fileImage = fileImage;
	}

	/**
	 * <p>Sets if files with the same file extension (see {@link #getFileExtension()}) are displayed.</p>
	 * 
	 * <p>In addition, {@link #setAllowOverwrite(boolean)} is called with {@link #displayFiles} for validation to succeed.</p>
	 * 
	 * @param displayFiles <code>true</code>If files with the same file extension are displayed, otherwise <code>false</code>.
	 * @see #getDisplayFiles() 
	 */
	public void setDisplayFiles(boolean displayFiles) {

		this.displayFiles = displayFiles;

		//Required for validation to succeed:
		setAllowOverwrite(displayFiles);
	}

	/**
	 * @deprecated As of TPTP 4.7.0, use {@link #setAllowOverwrite(boolean)}. 
	 */
	public void setAllowExistingResources(boolean allowExistingResources) {
		setAllowOverwrite(allowExistingResources);
	}

	/**
	 * <p>Sets if files can be overwritten.</p>
	 * 
	 * @param allowOverwrite <code>true</code> if files can be overwritten, otherwise <code>false</code>.
	 * @see #allowOverwrite()
	 */
	public void setAllowOverwrite(boolean allowOverwrite) {
		this.allowOverwrite = allowOverwrite;
	}

	/**
	 * <p>Sets the category identifier of new project menus added to the new project/folder drop-drop.</p>
	 * 
	 * <p>The category identifier is defined in the <code>org.eclipse.ui.newWizards</code> extension point for new wizards 
	 * with the <code>project</code> attribute set to <code>true</code>.</p>
	 * 
	 * @param newProjectWizardCategory The category identifier of new project menus to be added to the new project/folder drop-drop.
	 */
	public void setNewProjectWizardCategory(String newProjectWizardCategory) {
		this.newProjectWizardCategory = newProjectWizardCategory;
	}

	/**
	 * <p>Sets the custom name validator for validating the file name (see {@link #validatePage(boolean)}).</p>
	 * 
	 * @param nameValidator The custom file name validator for validating the file name.
	 */
	public void setNameValidator(ICustomLocationNameValidation nameValidator) {
		this.nameValidator = nameValidator;
	}

	/**
	 * <p>Adds a filter to remove containers (projects and folders) and/or files.</p>
	 * 
	 * <p>If called before the location page is created (see {@link #createControl(Composite)}), the 
	 * filter is used when creating the location page.  Otherwise, the location page is refreshed.</p>
	 * 
	 * @param filter The filter to remove containers (projects and folders) and/or files.
	 */
	public void addFilter(IFilter filter){

		if(filters == null){
			filters = new ArrayList();
		}

		filters.add(filter);
		
		if(treeViewer != null){
			treeViewer.refresh();
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent){

		initializeDialogUnits(parent);

		//Reusable layouts:
		GridLayout twoColumnGridLayout = new GridLayout(2, false);
		twoColumnGridLayout.marginWidth = 0;
		twoColumnGridLayout.marginHeight = 0;

		GridData textGridData = new GridData(SWT.FILL, SWT.CENTER, true, false);
		textGridData.widthHint = TEXT_WIDTH_HINT;

		GridData treeGridData = new GridData(SWT.FILL, SWT.FILL, true, true);
		treeGridData.heightHint = TREE_HEIGHT_HINT;

		//Top level composite:
		Composite topComposite = new Composite(parent, SWT.NONE);    
		topComposite.setLayout(new GridLayout());			
		topComposite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
		topComposite.setFont(parent.getFont());

		//Default to the help context ID for org.eclipse.ui.dialogs.WizardNewFileCreationPage:
		PlatformUI.getWorkbench().getHelpSystem().setHelp(topComposite, "org.eclipse.ui.ide.new_file_wizard_page_context"); //$NON-NLS-1$

		//Container label:
		Label containerLabel = new Label(topComposite, SWT.WRAP);
		containerLabel.setText(CommonUIMessages.LocationPage_CONTAINER_LABEL_TEXT);

		Composite containerComposite = new Composite(topComposite, SWT.NONE);
		containerComposite.setLayout(twoColumnGridLayout);
		containerComposite.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

		//Container full path text:
		containerFullPathText = new Text(containerComposite, (SWT.SINGLE | SWT.BORDER));
		containerFullPathText.setLayoutData(textGridData);
		containerFullPathText.addListener(SWT.Modify, this);

		ToolBar newContainerToolBar = new ToolBar(containerComposite, SWT.FLAT);

		ToolItem newContainerToolItem = new ToolItem(newContainerToolBar, (SWT.FLAT | SWT.DROP_DOWN));
		newContainerToolItem.setImage(CommonUIImages.INSTANCE.getImage("e", CommonUIImages.NEW_WIZARD)); //$NON-NLS-1$
		newContainerToolItem.setDisabledImage(CommonUIImages.INSTANCE.getImage("d", CommonUIImages.NEW_WIZARD)); //$NON-NLS-1$
		newContainerToolItem.setToolTipText(CommonUIMessages.LocationPage_NEW_CONTAINER_TOOL_ITEM_TOOL_TIP);
		newContainerToolItem.addSelectionListener(new SelectionAdapter(){

			/**
			 * <p>The menu containing menu items for creating a new container.</p>
			 */
			private Menu menu = null;

			/**
			 * <p>The new {@link IProject} created by the new project wizard(s).</p> 
			 */
			private IProject newProject = null;

			/**
			 * <p>The new {@link IFolder} created by the new folder wizard.</p> 
			 */
			private IFolder newFolder = null;

			/* (non-Javadoc)
			 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
			 */
			public void widgetSelected(SelectionEvent selectionEvent) {

				//Lazy create the menu:
				if (menu == null) {

					menu = new Menu(getShell());

					//Create a workspace resource change listener to capture the new project:
					//Assumption: The new project wizard will create only one project, otherwise the last project created is used.
					final IResourceChangeListener newProjectListener = new IResourceChangeListener(){

						/* (non-Javadoc)
						 * @see org.eclipse.core.resources.IResourceChangeListener#resourceChanged(org.eclipse.core.resources.IResourceChangeEvent)
						 */
						public void resourceChanged(IResourceChangeEvent resourceChangeEvent) {

							IResourceDelta delta = resourceChangeEvent.getDelta();

							if(delta != null){

								try {

									delta.accept(new IResourceDeltaVisitor(){

										/* (non-Javadoc)
										 * @see org.eclipse.core.resources.IResourceDeltaVisitor#visit(org.eclipse.core.resources.IResourceDelta)
										 */
										public boolean visit(IResourceDelta delta) throws CoreException{

											IResource newResource = delta.getResource();
											int resourceType = newResource.getType();

											if((resourceType == IResource.PROJECT) && (delta.getKind() == IResourceDelta.ADDED)){
												newProject = ((IProject)(newResource));
											}

											return (resourceType != IResource.FILE);
										}
									});
								} 
								catch (CoreException e) {
									CommonUIPlugin.log(e);
								}
							}
						}
					};

					//Create a workspace resource change listener to capture the new folder:
					//Assumption: The BasicNewFolderResourceWizard will create only one folder, otherwise the last folder created is used.
					final IResourceChangeListener newFolderListener = new IResourceChangeListener(){

						/* (non-Javadoc)
						 * @see org.eclipse.core.resources.IResourceChangeListener#resourceChanged(org.eclipse.core.resources.IResourceChangeEvent)
						 */
						public void resourceChanged(IResourceChangeEvent resourceChangeEvent) {

							IResourceDelta delta = resourceChangeEvent.getDelta();

							if(delta != null){

								try {

									delta.accept(new IResourceDeltaVisitor(){

										/* (non-Javadoc)
										 * @see org.eclipse.core.resources.IResourceDeltaVisitor#visit(org.eclipse.core.resources.IResourceDelta)
										 */
										public boolean visit(IResourceDelta delta) throws CoreException{

											IResource newResource = delta.getResource();
											int resourceType = newResource.getType();

											if((resourceType == IResource.FOLDER) && (delta.getKind() == IResourceDelta.ADDED)){
												newFolder = ((IFolder)(newResource));
											}

											return (resourceType != IResource.FILE);
										}
									});
								} 
								catch (CoreException e) {
									CommonUIPlugin.log(e);
								}
							}
						}
					};

					//Create the new project menu(s):
					int newProjectWizardMenuCount = 0;

					//Resolve the list of wizard configuration elements for new wizards (implementations of the 
					//org.eclipse.ui.newWizards extension point) for projects with the category:
					if(newProjectWizardCategory != null){

						//Resolve the org.eclipse.ui.newWizards extension point:
						IExtensionPoint extensionPoint = Platform.getExtensionRegistry().getExtensionPoint("org.eclipse.ui.newWizards"); //$NON-NLS-1$

						if (extensionPoint != null) {

							IConfigurationElement[] configurationElements = extensionPoint.getConfigurationElements();

							for (int counter = 0; counter < configurationElements.length; counter++) {

								final IConfigurationElement configurationElement = configurationElements[counter];

								if (("wizard".equals(configurationElement.getName())) && (Boolean.parseBoolean(configurationElement.getAttribute("project")) && (newProjectWizardCategory.equals(configurationElement.getAttribute("category"))))) { //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$

									//Create the new project menu item:
									MenuItem newProjectMenuItem = new MenuItem(menu, SWT.NONE);
									newProjectMenuItem.setText(configurationElement.getAttribute("name")); //$NON-NLS-1$

									Image newProjectMenuItemImage = null;

									try{							
										newProjectMenuItemImage = ImageDescriptor.createFromURL(new URL(Platform.getBundle(configurationElement.getDeclaringExtension().getNamespaceIdentifier()).getEntry("/"), configurationElement.getAttribute("icon"))).createImage(false); //$NON-NLS-1$ //$NON-NLS-2$
									}
									catch (Exception e){
										CommonUIPlugin.log(e);
									}

									//If the wizard image cannot be loaded, default to the new project wizard image:
									if(newProjectMenuItemImage != null){
										newProjectMenuItem.setImage(newProjectMenuItemImage);
									}
									else{
										newProjectMenuItem.setImage(CommonUIImages.INSTANCE.getImage("e", CommonUIImages.NEW_WIZARD)); //$NON-NLS-1$							
									}

									newProjectMenuItem.addSelectionListener(new SelectionAdapter() {

										/* (non-Javadoc)
										 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
										 */
										public void widgetSelected(SelectionEvent selectionEvent) {

											//Close the menu:
											menu.setVisible(false);

											try {

												//Create the new project wizard:
												INewWizard newProjectWizard = ((INewWizard)(configurationElement.createExecutableExtension("class"))); //$NON-NLS-1$
												newProjectWizard.init(PlatformUI.getWorkbench(), ((IStructuredSelection)(treeViewer.getSelection())));

												//Create the new project wizard dialog:
												WizardDialog newProjectWizardDialog = new WizardDialog(Display.getCurrent().getActiveShell(), newProjectWizard);
												newProjectWizardDialog.create();

												newProject = null;

												//Add the workspace resource change listener to capture the new project:
												ResourcesPlugin.getWorkspace().addResourceChangeListener(newProjectListener, IResourceChangeEvent.POST_CHANGE);

												//Open the new folder wizard dialog and select the new folder in the tree:
												if ((newProjectWizardDialog.open() == Window.OK) && (newProject != null)) {

													treeViewer.refresh(true);

													updateSelection(new StructuredSelection(newProject));
												}
											} 
											catch (Exception e) {
												CommonUIPlugin.log(e);
											}
											finally{

												//Remove the workspace resource change listener:
												ResourcesPlugin.getWorkspace().removeResourceChangeListener(newProjectListener);
											}								
										}
									});

									newProjectWizardMenuCount++;
								} 
							}
						}
					}

					//Create the new project menu item:
					MenuItem newProjectMenuItem = new MenuItem(menu, SWT.NONE);
					newProjectMenuItem.setText(CommonUIMessages.LocationPage_NEW_PROJECT_MENU_ITEM_TEXT);
					newProjectMenuItem.setImage(CommonUIImages.INSTANCE.getImage("e", CommonUIImages.NEW_WIZARD)); //$NON-NLS-1$
					newProjectMenuItem.addSelectionListener(new SelectionAdapter() {

						private NewProjectAction newProjectAction = null;

						/* (non-Javadoc)
						 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
						 */
						public void widgetSelected(SelectionEvent selectionEvent) {

							//Close the menu:
							menu.setVisible(false);

							try {

								newProject = null;

								//Add the workspace resource change listener to capture the new project:
								ResourcesPlugin.getWorkspace().addResourceChangeListener(newProjectListener, IResourceChangeEvent.POST_CHANGE);

								//Open the new project wizard:
								if(newProjectAction == null){
									newProjectAction = new NewProjectAction();						
								}

								newProjectAction.run();

								//Select the new project in the tree:
								if (newProject != null){

									treeViewer.refresh(true);

									updateSelection(new StructuredSelection(newProject));
								}
							} 
							finally{

								//Remove the workspace resource change listener:
								ResourcesPlugin.getWorkspace().removeResourceChangeListener(newProjectListener);
							}								
						}
					});

					new MenuItem(menu, SWT.SEPARATOR);

					//Create the new folder menu item:
					MenuItem newFolderMenuItem = new MenuItem(menu, SWT.NONE);
					newFolderMenuItem.setText(CommonUIMessages.LocationPage_NEW_FOLDER_MENU_ITEM_TEXT);
					newFolderMenuItem.setImage(CommonUIImages.INSTANCE.getImage("e", CommonUIImages.NEW_FOLDER_WIZARD)); //$NON-NLS-1$
					newFolderMenuItem.addSelectionListener(new SelectionAdapter() {

						/* (non-Javadoc)
						 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
						 */
						public void widgetSelected(SelectionEvent selectionEvent) {

							//Close the menu:
							menu.setVisible(false);

							//Create the new folder wizard:
							BasicNewFolderResourceWizard newFolderWizard = new BasicNewFolderResourceWizard();
							newFolderWizard.init(PlatformUI.getWorkbench(), ((IStructuredSelection)(treeViewer.getSelection())));

							//Create the new folder wizard dialog:
							WizardDialog newFolderWizardDialog = new WizardDialog(Display.getCurrent().getActiveShell(), newFolderWizard);
							newFolderWizardDialog.create();

							try {

								newFolder = null;

								//Add the workspace resource change listener to capture the new folder:
								ResourcesPlugin.getWorkspace().addResourceChangeListener(newFolderListener, IResourceChangeEvent.POST_CHANGE);

								//Open the new folder wizard dialog and select the new folder in the tree:
								if ((newFolderWizardDialog.open() == Window.OK) && (newFolder != null)) {

									treeViewer.refresh(true);

									updateSelection(new StructuredSelection(newFolder));
								}
							} 
							finally{

								//Remove the workspace resource change listener:
								ResourcesPlugin.getWorkspace().removeResourceChangeListener(newFolderListener);
							}								
						}
					});
				}

				//Open/close the menu when the arrow drop down is selected:
				if (selectionEvent.detail == SWT.ARROW) {

					boolean isMenuVisible = menu.getVisible();

					if (!isMenuVisible) {

						//Position the menu below and vertically aligned with the the drop down tool button:
						ToolItem toolItem = ((ToolItem)(selectionEvent.widget));
						Rectangle toolItemBounds = toolItem.getBounds();
						Point position = toolItem.getParent().toDisplay(new Point(toolItemBounds.x, toolItemBounds.y));

						menu.setLocation(position.x, (position.y + toolItemBounds.height));
					}

					//Toggle the menu to give the arrow the appearance of a toggle button:
					menu.setVisible(!isMenuVisible);
				} 

				//Select the first selectable menu item when the button is selected:
				else {

					MenuItem[] menuItems = menu.getItems();

					for (int counter = 0; counter < menuItems.length; counter++) {

						if((menuItems[counter] != null) && ((menuItems[counter].getStyle() & SWT.SEPARATOR) == 0)){

							//Fire a menu selection event:
							Event menuSelectionEvent = new Event();
							menuSelectionEvent.type = SWT.Selection;
							menuSelectionEvent.widget = menuItems[counter];

							menuItems[counter].notifyListeners(SWT.Selection, menuSelectionEvent);

							break;
						}
					}
				}
			}
		});

		//Create drill down:
		DrillDownComposite drillDown = new DrillDownComposite(topComposite, SWT.BORDER);
		drillDown.setLayoutData(treeGridData);

		//Create tree viewer inside the drill down:
		treeViewer = new TreeViewer(drillDown, SWT.NONE);
		
		treeViewer.setContentProvider(new ITreeContentProvider() {

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
			 */
			public Object[] getChildren(Object element) {

				if (element instanceof IWorkspace) {

					ArrayList openProjects = new ArrayList();
					IProject[] projects = ((IWorkspace)(element)).getRoot().getProjects();

					for (int counter = 0; counter < projects.length; counter++) {

						if ((!isFiltered(projects[counter])) && (projects[counter].isOpen())) {
							openProjects.add(projects[counter]);
						}
					}

					return (openProjects.toArray());
				} 
				else if (element instanceof IContainer) {

					IContainer container = ((IContainer)(element));

					if (container.isAccessible()) {

						try {

							List children = new ArrayList();
							IResource[] members = container.members();

							for (int counter = 0; counter < members.length; counter++) {

								if((!isFiltered(members[counter])) && ((members[counter].getType() != IResource.FILE) || (isDisplayed(((IFile)(members[counter])))))){
									children.add(members[counter]);		
								}
							}

							return (children.toArray());
						} 
						catch (CoreException e) {
							CommonUIPlugin.log(e);
						}
					}
				}

				return (new Object[0]);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
			 */
			public Object[] getElements(Object element) {
				return (getChildren(element));
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
			 */
			public Object getParent(Object element) {

				if (element instanceof IResource) {
					return (((IResource)(element)).getParent());
				}

				return null;
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
			 */
			public boolean hasChildren(Object element) {
				return (getChildren(element).length > 0);
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
			 */
			public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
				//No-operation.
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
			 */
			public void dispose() {
				//No-operation.
			}
		});

		treeViewer.setLabelProvider(new DecoratingLabelProvider(new WorkbenchLabelProvider(), PlatformUI.getWorkbench().getDecoratorManager().getLabelDecorator()){

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.DecoratingLabelProvider#getImage(java.lang.Object)
			 */
			public Image getImage(Object element) {

				if((fileImage != null) && (element instanceof IFile) && (containsFileExtension(((IFile)(element)).getName()))){
					return fileImage;					
				}

				return (super.getImage(element));
			}

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.DecoratingLabelProvider#getText(java.lang.Object)
			 */
			public String getText(Object element) {
				return (removeFileExtension(super.getText(element)));
			}
		});

		treeViewer.setComparator(new ResourceComparator(ResourceComparator.NAME));
		treeViewer.setUseHashlookup(true);

		treeViewer.addSelectionChangedListener(new ISelectionChangedListener(){

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
			 */
			public void selectionChanged(SelectionChangedEvent event){

				ISelection selection = event.getSelection();

				if(selection instanceof IStructuredSelection){

					IResource resource = getResource(((IStructuredSelection)(selection)));

					if(resource instanceof IContainer){			
						updateContainerFullPath(((IContainer)(resource)).getFullPath());
					}
					else if(resource instanceof IFile){

						IFile file = ((IFile)(resource));

						updateContainerFullPath(file.getParent().getFullPath());
						updateFileName(file.getName());
					}	
				}
			}
		});

		treeViewer.addDoubleClickListener(new IDoubleClickListener() {

			/* (non-Javadoc)
			 * @see org.eclipse.jface.viewers.IDoubleClickListener#doubleClick(org.eclipse.jface.viewers.DoubleClickEvent)
			 */
			public void doubleClick(DoubleClickEvent event) {

				ISelection selection = event.getSelection();

				if(selection instanceof IStructuredSelection) {

					Object selectedElement = ((IStructuredSelection) selection).getFirstElement();

					if (selectedElement != null) {

						if (treeViewer.getExpandedState(selectedElement)) {
							treeViewer.collapseToLevel(selectedElement, 1);
						} 
						else {
							treeViewer.expandToLevel(selectedElement, 1);
						}
					}
				}
			}
		});

		//Note: The input can only be set once the tree viewer has been configured.
		treeViewer.setInput(workspaceRoot);

		drillDown.setChildTree(treeViewer);

		Composite fileComposite = new Composite(topComposite, SWT.NONE);
		fileComposite.setLayout(twoColumnGridLayout);
		fileComposite.setLayoutData(new GridData(SWT.FILL, SWT.CENTER, true, false));

		Label fileNameLabel = new Label(fileComposite, SWT.WRAP);
		fileNameLabel.setText(CommonUIMessages.LocationPage_NAME_LABEL_TEXT);

		fileNameText = new Text(fileComposite, SWT.BORDER);
		fileNameText.setLayoutData(textGridData);
		fileNameText.addListener(SWT.Modify, this);

		createAdvancedControls(topComposite);

		//Update the UI with the initial selection:
		if(initialSelection != null){
			updateSelection(initialSelection);
		}

		//Update the UI with the initial container:
		if(initialContainerFullPath != null){    	

			IContainer container = getContainer(initialContainerFullPath);

			if(container != null){
				updateSelection(new StructuredSelection(container));	
			}
			else{
				updateContainerFullPath(initialContainerFullPath);
			}
		}		

		//Default to the first open project if no container:
		if(containerFullPathText.getText().length() == 0) {

			IProject[] projects = workspaceRoot.getProjects();

			for (int counter = 0; counter < projects.length; counter++) {

				if(projects[counter].isAccessible()){

					updateSelection(new StructuredSelection(projects[counter]));

					break;
				}
			}
		}

		//Update the UI with the initial file name:
		if(initialFileName != null){

			IFile file = getFile(initialFileName);

			if(file != null){
				updateSelection(new StructuredSelection(file));	
			}
			else{
				updateFileName(initialFileName);
			}
		}

		setPageComplete(validatePage(false));

		//Clear the wizard error/message since the user has 
		//not yet selected/specified any input for validation
		//resulting in an error/message:
		setMessage(null);
		setErrorMessage(null);

		Dialog.applyDialogFont(topComposite);

		setControl(topComposite);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.dialogs.DialogPage#setVisible(boolean)
	 */
	public void setVisible(boolean visible) {

		super.setVisible(visible);

		if(visible){
		
			fileNameText.selectAll();
			fileNameText.setFocus();
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.widgets.Listener#handleEvent(org.eclipse.swt.widgets.Event)
	 */
	public void handleEvent(Event event) {
		setPageComplete(validatePage(true));
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.wizard.IWizardPage#isPageComplete()
	 */
	public boolean isPageComplete(){		
		return (validatePage(false));
	}

	/**
	 * <p>Creates an empty file for the workspace path of the specified/selected file 
	 * name with the file extension, if the file does not exist.</p>
	 * 
	 * <p>Any missing folders in the workspace path of the specified/selected file 
	 * name with the file extension are created.</p>
	 * 
	 * <p>This method should be called within a {@link WorkspaceModifyOperation} since it
	 * creates a file.</p>
	 * 
	 * @return The newly created file resource, otherwise <code>null</code>.
	 */
	public IFile createNewFile() {

		String filePath = getFilePath();

		if(filePath != null){

			final IFile file = workspaceRoot.getFile(new Path(filePath));

			if(!file.isAccessible()){

				try {

					getContainer().run(true, true, new IRunnableWithProgress() {

						/* (non-Javadoc)
						 * @see org.eclipse.jface.operation.IRunnableWithProgress#run(org.eclipse.core.runtime.IProgressMonitor)
						 */
						public void run(IProgressMonitor monitor) {

							try {
								new CreateFileOperation(file, null, null, CommonUIMessages.LocationPage_CREATE_FILE_OPERATION_LABEL).execute(monitor, WorkspaceUndoUtil.getUIInfoAdapter(getShell()));
							} 
							catch (Exception e) {
								CommonUIPlugin.log(e);
							}
						}
					});
				} 
				catch (Exception e) {
					CommonUIPlugin.log(e);
				}
			}

			if(file.isAccessible()){
				return file;
			}
		}

		return null;
	}

	/**
	 * <p>Adds advanced controls to the top (requires calling {@link Control#moveAbove()} with 
	 * <code>null</code> on the {@link Composite} containing the advance controls) or bottom 
	 * of the parent composite.</li>
	 * 
	 * <p>This method is invoked on the location page's control at the end of 
	 * {@link #createControl(Composite)} but before the location page is initialized.</p> 
	 * 
	 * <p>By default, this method does nothing.</p>
	 * 
	 * @param parent The parent composite.
	 */
	 protected void createAdvancedControls(Composite parent){
		//No-operation by default.
	}

	/**
	 * @deprecated As of TPTP 4.7.0, use {@link #validatePage(boolean)}. 
	 */
	protected boolean validatePage(){
		return validatePage(true);
	}
	
	/**
	 * @deprecated As of TPTP V4.7.0, this method is considered internal.
	 * @noreference This method is not intended to be referenced by clients.
	 */
	//Required for backward compatibility.
	protected TreeViewer internal_getTreeViewer(){
		return treeViewer;
	}

	/**
	 * <p>Determines if the page's controls contain valid values.</p>
	 * 
	 * <p>Valid values require all of the following conditions (in order):</p>
	 * 
	 * <ol>
	 * <li>The container is specified.</li>
	 * <li>The container is valid:
	 * <ol>
	 * <li>{@link LocationValidator#validatePath(IPath)}:
	 * <ol>
	 * <li>{@link IWorkspace#validatePath(String, int)}</li>
	 * <li>{@link LocationValidator#validateEMFSegment(String)}</li>
	 * </ol>
	 * </li>
	 * </ol>
	 * </li>
	 * <li>The file name is specified.</li>
	 * <li>The file name is valid:
	 * <ol>
	 * <li>Does not start with a period (.).</li>
	 * <li>{@link LocationValidator#validateOSFileName(String)}:
	 * <ol>
	 * <li>Does not start with a dash (-).</li>
	 * <li>{@link Path.ROOT#isValidPath(String)}</li>
	 * <li>{@link IWorkspace#validateName(String, int)}</li>
	 * <li>{@link LocationValidator#validateEMFSegment(String)}</li>
	 * </ol>
	 * </li>
	 * </ol>
	 * </li>
	 * <li>The file path is valid:
	 * <ol>
	 * <li>{@link ICustomLocationNameValidation#isNameValid(String, IContainer)}</li>
	 * </ol>
	 * </li>
	 * <li>The project exists.</li>
	 * <li>If overwrite is not allowed, the file does not exist.</li>
	 * <li>The file is not filtered (see {@link IWorkspace#validateFiltered(IResource)}).</li>
	 * </ol>
	 * 
	 * <p>If the values are not valid, an error message is displayed if the page's message is to be 
	 * updated and the method returns false.</p>
	 * 
	 * @param updateMessage <code>true</code> if the page's message is to be updated, otherwise <code>false</code>.
	 * @return <code>true</code> if the page's controls contain valid values, otherwise <code>false</code>.
	 */
	protected boolean validatePage(boolean updateMessage){

		IPath containerFullPath = getContainerFullPath();

		//Case 1: Container full path is missing:
		if(containerFullPath == null){

			if(updateMessage){

				setMessage(null);
				setErrorMessage(CommonUIMessages.LocationPage_MISSING_CONTAINER_ERROR_);
			}

			return false;
		}

		String invalidCharacter = LocationValidator.validatePath(containerFullPath);

		//Case 2: Container is invalid:
		if(invalidCharacter != null){

			if(updateMessage){

				setMessage(null);

				if (LocationValidator.INVALID_PATH.equals(invalidCharacter)) {
					setErrorMessage(CommonUIMessages.LocationPage_INVALID_CONTAINER_ERROR_);
				} 
				else {
					setErrorMessage(NLS.bind(CommonUIMessages.LocationPage_INVALID_CONTAINER_CHARS_ERROR_, invalidCharacter)); 
				}
			}

			return false;
		}

		String fileNameWithoutExtension = getFileNameWithoutExtension();

		//Case 3: File name is missing:
		if((fileNameWithoutExtension == null) || (fileNameWithoutExtension.trim().length() == 0)){

			if(updateMessage){

				setMessage(null);
				setErrorMessage(CommonUIMessages.LocationPage_MISSING_NAME_ERROR_);
			}

			return false;
		}

		//Case 4: File name is invalid:
		if (fileNameWithoutExtension.trim().startsWith(".")) { //$NON-NLS-1$

			if(updateMessage){

				setMessage(null);
				setErrorMessage(CommonUIMessages.LocationPage_INVALID_NAME_PERIOD_ERROR_);
			}

			return false;
		}

		invalidCharacter = LocationValidator.validateOSFileName(getFileName());

		if(invalidCharacter != null){

			if(updateMessage){

				setMessage(null);

				if (invalidCharacter.trim().equals("-")) { //$NON-NLS-1$
					setErrorMessage(CommonUIMessages.LocationPage_INVALID_NAME_DASH_ERROR_);
				} 
				else if (LocationValidator.INVALID_PATH.equals(invalidCharacter)) {
					setErrorMessage(CommonUIMessages.LocationPage_INVALID_NAME_ERROR_);
				} 
				else {
					setErrorMessage(NLS.bind(CommonUIMessages.LocationPage_INVALID_NAME_CHARS_ERROR_, invalidCharacter)); 
				}			
			}

			return false;
		}

		//Case 5: File path is invalid:
		IFile file = workspaceRoot.getFile(new Path(getFilePath()));

		if ((nameValidator != null) && (!nameValidator.isNameValid(file.getName(), file.getParent()))) {

			if(updateMessage){

				setMessage(null);
				setErrorMessage(CommonUIMessages.LocationPage_INVALID_NAME_ERROR_);
			}						

			return false;
		}

		IProject project = file.getProject();

		//Case 6: Project does not exist:
		if((project == null) || (!project.exists())){

			if(updateMessage){

				setMessage(null);
				setErrorMessage(CommonUIMessages.LocationPage_NO_PROJECT_ERROR_);
			}

			return false;
		}

		//Case 7: If file overwrite is not permitted, the file exists:
		if ((!allowOverwrite) && (file.exists())) {

			if(updateMessage){

				setMessage(null);
				setErrorMessage(CommonUIMessages.LocationPage_EXISTING_NAME_ERROR_);
			}						

			return false;
		}

		//Case 8: File is filtered:
		if (!workspaceRoot.getWorkspace().validateFiltered(file).isOK()) {

			if(updateMessage){

				setMessage(CommonUIMessages.LocationPage_FILTERED_NAME_WARNING_, IMessageProvider.WARNING);
				setErrorMessage(null);
			}						

			return false;
		}

		//Case 9 (default): Valid container and file name:
		if(updateMessage){

			setMessage(null);
			setErrorMessage(null);
		}

		return true;
	}

	private void updateContainerFullPath(IPath containerFullPath){

		if (containerFullPath != null) {

			String containerFullPathString = TextProcessor.process(containerFullPath.makeRelative().toString());

			//Set the container:
			containerFullPathText.setText(containerFullPathString);
			containerFullPathText.setToolTipText(containerFullPathString);
		}

		validatePage(true);
	}

	private void updateFileName(String fileName){

		String fileNameWithExtension = removeFileExtension(fileName);

		if(fileNameWithExtension != null){

			fileNameText.setText(fileNameWithExtension);
			fileNameText.setToolTipText(fileNameWithExtension);
		}

		validatePage(true);
	}

	private void updateSelection(IStructuredSelection selection){

		IResource resource = getResource(selection);

		if(resource != null){

			//Reveal, expand, and select the container in the tree:
			IContainer parentContainer = resource.getParent();

			if(parentContainer != null){
				treeViewer.expandToLevel(parentContainer, 1);		
			}

			treeViewer.setSelection(new StructuredSelection(resource), true);
		}

		//Fire a selection event:
		Event selectionEvent = new Event();
		selectionEvent.type = SWT.Selection;
		selectionEvent.widget = treeViewer.getTree();

		handleEvent(selectionEvent);	
	}

	private IFile getFile(String fileName){

		if((fileName != null) && (fileName.length() > 0)){

			IPath containerFullPath = getContainerFullPath();

			if(containerFullPath != null){

				IResource resource = workspaceRoot.findMember(containerFullPath.append(addFileExtension(fileName)));

				if(resource instanceof IFile) {

					IFile file = ((IFile)(resource));

					if(isDisplayed(file)){
						return file;
					}
				}					
			}
		}

		return null;
	}

	private IContainer getContainer(IPath path){

		if(path != null){

			IResource resource = workspaceRoot.findMember(path);

			if (resource != null) {

				if (resource instanceof IContainer) {
					return ((IContainer)(resource));
				}
				else{
					return (resource.getParent());
				}
			}
		}

		return null;
	}

	private IResource getResource(IStructuredSelection selection){

		if(selection != null){

			Object selectedObject = selection.getFirstElement();

			if(selectedObject instanceof IResource){
				return ((IResource)(selectedObject));
			}
			else if(selectedObject instanceof IAdaptable){
				return ((IResource)(((IAdaptable)(selectedObject)).getAdapter(IResource.class)));
			}
		}

		return null;
	}

	private String addFileExtension(String fileName){

		if((fileName != null) && (fileName.length() > 0) && (fileExtensionExists()) && (!containsFileExtension(fileName))){
			return (fileName + '.' + fileExtension);
		}

		return fileName;
	}

	private String removeFileExtension(String fileName){

		if(containsFileExtension(fileName)){
			return (fileName.substring(0, (fileName.length() - fileExtension.length() - 1)));
		}

		return fileName;
	}

	private boolean isDisplayed(IFile file){
		return ((file != null) && (displayFiles) && (containsFileExtension(file.getName())));
	}

	private boolean containsFileExtension(String fileName){		
		return((fileName != null) && (fileName.length() > 0) && (fileExtensionExists()) && (fileName.trim().endsWith('.' + fileExtension))); 
	}

	private boolean fileExtensionExists(){		
		return((fileExtension != null) && (fileExtension.length() > 0)); 
	}
	
	private boolean isFiltered(Object object){
		
		if((filters != null) && (filters.size() > 0)){

			Iterator filtersIterator = filters.iterator();

			while (filtersIterator.hasNext()) {

				IFilter filter = ((IFilter)(filtersIterator.next()));

				if((filter != null) && (!filter.select(object))){
					return true;					
				}
			}
		}
		
		return false;
	}
}