/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: InvokedTestSelectionDialog.java,v 1.8 2010/05/21 21:24:51 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.ui.common.internal.dialog;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.hyades.models.common.facades.behavioral.ITest;
import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.adapter.TestWorkbenchAdapter;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.IFilter;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.TestOnlyFilter;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.TestProvider;
import org.eclipse.hyades.test.ui.navigator.IProxy;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.ITestCaseProxyNode;
import org.eclipse.hyades.test.ui.navigator.ITestSuiteProxyNode;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.ITypeValidator;
import org.eclipse.hyades.ui.internal.provider.WorkbenchAdapterLabelProvider;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.DecoratingLabelProvider;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ILabelDecorator;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.PlatformUI;

/**
 * <p>InvokedTestSelectionDialog.java</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Paul Slauenwhite
 * @version May 21, 2010
 * @since   February 1, 2005
 */
public class InvokedTestSelectionDialog extends Dialog implements IDoubleClickListener, ISelectionChangedListener {
	
	protected static final int HELP_DLG_CONTROL = 1;
	protected static final int HELP_LOCAL_TEST_CASES_CONTROL = 2;
	protected static final int HELP_TESTS_CONTROL = 3;
	
	private ITestSuite testSuite;
	private Collection tests;
	
	private TableViewer localTestCasesTableViewer;
	private TreeViewer testsTreeViewer;
	
	private String title;
	private String tableLabel;
	private String treeLabel;
	private boolean singleSelection;
	
	private ITypeValidator typeValidator;
	
	/**
	 * Constructor for InvokedTestSelectionDialog
	 * @param parentShell
	 */
	public InvokedTestSelectionDialog(Shell parentShell, ITestSuite testSuite, String title, String tableLabel, String treeLabel, boolean singleSelection) {
		super(parentShell);
		setShellStyle(getShellStyle() | SWT.RESIZE | SWT.MAX);
		this.testSuite = testSuite;
		this.title = title;
		this.tableLabel = tableLabel;
		this.treeLabel = treeLabel;
		this.singleSelection = singleSelection;
		tests = new ArrayList();
		
		String type = testSuite.getType();
		if(type != null) {
			IAssociationDescriptor associationDescriptor = TestUIExtension.getTestSuiteMappingRegistry().getAssociationMapping(HyadesUIPlugin.EP_TYPE_VALIDATORS).getDefaultAssociationDescriptor(type);
			if(associationDescriptor != null) {
				typeValidator = (ITypeValidator)associationDescriptor.createImplementationClassInstance();
			}
		}
	}

	public void setTest(ITest test) {
		tests.clear();
		if (test instanceof ITestCase && ((ITestCase)test).getOwner().equals(getTestSuite())) {
			tests.add(test);
		} else if (test instanceof IAdaptable) {
			tests.add(((IAdaptable)test).getAdapter(IProxy.class));
		}
		selectTest(tests);
	}
	
	public Collection getTests() {
		Collection ret = new ArrayList(tests.size());
		for (Iterator it = tests.iterator(); it.hasNext();) {
			Object obj = it.next();
			if (obj instanceof ITestCase) {
				ret.add(obj);
			} else if (obj instanceof ITestSuiteProxyNode) {
				ITestSuiteProxyNode proxy = (ITestSuiteProxyNode)obj;
				ret.add(proxy.getTestSuite());
			} else if (obj instanceof ITestCaseProxyNode) {
				ITestCaseProxyNode proxy = (ITestCaseProxyNode)obj;
				ret.add(proxy.getTestCase());
			}
		} 
		return ret;
	}
	
	public ITestSuite getTestSuite() {
		return testSuite;
	}
	
	protected ITypeValidator getTypeValidator() {
		return typeValidator;
	}

	/**
	 * @see org.eclipse.jface.window.Window#createContents(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createContents(Composite parent) {
		Control control = super.createContents(parent);
		
		getButton(IDialogConstants.OK_ID).setEnabled(false);
		localTestCasesTableViewer.setInput(testSuite);
		if(testsTreeViewer != null) {
			testsTreeViewer.setInput(ResourcesPlugin.getWorkspace().getRoot());
		}
		selectTest(tests);
						
		registerHelp(HELP_DLG_CONTROL, control);	
		registerHelp(HELP_LOCAL_TEST_CASES_CONTROL, localTestCasesTableViewer.getControl());
		if(testsTreeViewer != null)				
			registerHelp(HELP_TESTS_CONTROL, testsTreeViewer.getControl());				

		return control;
	}
	
	protected void registerHelp(int flag, Object object) {
	}

	/**
	 * @see org.eclipse.jface.dialogs.Dialog#createDialogArea(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createDialogArea(Composite parent) {
		Composite superComposite = (Composite)super.createDialogArea(parent);
		GridData gridData = GridDataUtil.createFill();
		superComposite.setLayoutData(gridData);
		
		createLocalTestCasesControl(superComposite);
		if(getTypeValidator() != null) {
			createTestControl(superComposite);
		}
			
		if(title != null) {
			getShell().setText(title);
		}
		return superComposite;
	}
	
	protected void selectTest(Collection tests) {
		if(localTestCasesTableViewer == null) return;
		
		if(tests != null) {
			for (Iterator i = tests.iterator(); i.hasNext();) {
				Object obj = i.next();
				if(obj instanceof ITestCase) {
					if(((ITestCase)obj).getOwner() == testSuite) {
						localTestCasesTableViewer.setSelection(new StructuredSelection(obj), true);
						continue;
					}
				}
				
				if(obj instanceof IProxy && testsTreeViewer != null) {
					TestProvider testProvider = (TestProvider)testsTreeViewer.getContentProvider();
					IProxyNode proxy = testProvider.getFileProxyManager().getCorrespondingProxy(obj);
					IProject project = proxy.getUnderlyingResource().getProject();
					testProvider.getChildren(project);
					testsTreeViewer.setSelection(new StructuredSelection(proxy), true);					
				}
			}
		} else {
			localTestCasesTableViewer.setSelection(StructuredSelection.EMPTY);
			if(testsTreeViewer != null) {
				testsTreeViewer.setSelection(StructuredSelection.EMPTY);
			}
		}
	}

	protected void createLocalTestCasesControl(Composite parent) {
		if(tableLabel != null) {
			Label label = new Label(parent, SWT.NONE);
			label.setText(tableLabel);
		}
		
		int selection = singleSelection ? SWT.SINGLE : SWT.MULTI;
		localTestCasesTableViewer = new TableViewer(parent,  selection | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
		localTestCasesTableViewer.addDoubleClickListener(this);
		localTestCasesTableViewer.addSelectionChangedListener(this);
		
		if(getTypeValidator() != null) {
			GridData gridData = GridDataUtil.createHorizontalFill();
			gridData.grabExcessVerticalSpace = false;
			gridData.heightHint = 100;
			localTestCasesTableViewer.getControl().setLayoutData(gridData);
		} else {
			localTestCasesTableViewer.getControl().setLayoutData(GridDataUtil.createFill());
		}
		
		IStructuredContentProvider tableViewContentProvider = new IStructuredContentProvider() {
			public void dispose() {
			}
			public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
			}
			public Object[] getElements(Object inputElement) {
				if(getTestSuite() != null)
					return getTestSuite().getITestCases().toArray();
				return new Object[0];
			}
		};
		localTestCasesTableViewer.setContentProvider(tableViewContentProvider);
		localTestCasesTableViewer.setLabelProvider(new WorkbenchAdapterLabelProvider(TestWorkbenchAdapter.class));
	}
	
	protected void createTestControl(Composite parent) {
		if(treeLabel != null) {
			Label label = new Label(parent, SWT.NONE);
			label.setText(treeLabel);
		}
		
		int selection = singleSelection ? SWT.SINGLE : SWT.MULTI;
		testsTreeViewer =	new TreeViewer(parent, selection | SWT.H_SCROLL | SWT.V_SCROLL | SWT.BORDER);
		GridData gridData = GridDataUtil.createFill();
		gridData.heightHint = 220;
		testsTreeViewer.getControl().setLayoutData(gridData);
		testsTreeViewer.addDoubleClickListener(this);
		testsTreeViewer.addSelectionChangedListener(this);
		
		final IProxy testSuiteProxy = (IProxy) ((IAdaptable)getTestSuite()).getAdapter(IProxy.class);
		IFilter filter = new TestOnlyFilter() {
			public boolean satisfies(IProxyNode node) {
				if (node instanceof ITestSuiteProxyNode) {
					if (testSuiteProxy.equals(node)) return false;
					String type = ((ITestSuiteProxyNode)node).getType();
					if (getTypeValidator().isValidType(type)) {
						// TODO Check that the underlying test suite has a model behavior
						return true;
					}
					return false;
				}
				return super.satisfies(node);
			}
			public boolean satisfies(String type) {
				return getTypeValidator().isValidType(type);
			}
			public boolean satisfiesChildren(IProxyNode node) {
				return true;
			}
			
		};
		final TestProvider testProvider = new TestProvider(filter, false);
		testsTreeViewer.setContentProvider(testProvider);				
		ILabelDecorator decorator = PlatformUI.getWorkbench().getDecoratorManager().getLabelDecorator();
		testsTreeViewer.setLabelProvider(new DecoratingLabelProvider(testProvider, decorator));
		testsTreeViewer.getControl().addDisposeListener(new DisposeListener() {
			public void widgetDisposed(DisposeEvent e) {
				testProvider.dispose();
			}
		});
	}

	/**
	 * @see org.eclipse.jface.viewers.IDoubleClickListener#doubleClick(org.eclipse.jface.viewers.DoubleClickEvent)
	 */
	public void doubleClick(DoubleClickEvent event) {
		
		if(processSelection()) {
			okPressed();
		}
	}

	/**
	 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
	 */
	public void selectionChanged(SelectionChangedEvent event){
		processSelection();
	}
	
	protected boolean processSelection() {		

		tests.clear();
		
		processSelection(localTestCasesTableViewer);
		processSelection(testsTreeViewer);
		
		boolean isEnabled = (tests.size() > 0);
		
		getButton(IDialogConstants.OK_ID).setEnabled(isEnabled);
		
		return isEnabled;		
	}
	
	private void processSelection(StructuredViewer structuredViewer){
		
		IStructuredSelection structuredSelection = ((IStructuredSelection)(structuredViewer.getSelection()));

		if(!structuredSelection.isEmpty()) {

			Iterator structuredSelectioniIterator = structuredSelection.iterator();

			while(structuredSelectioniIterator.hasNext()){

				Object object = structuredSelectioniIterator.next();

				if(object instanceof ITestSuiteProxyNode || object instanceof ITestCaseProxyNode || object instanceof ITestCase) {
					tests.add(object);
				}
			}
		}
	}
}
