/**********************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: BehaviorEclipseForm.java,v 1.9 2007/05/03 20:50:01 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.ui.common.internal.editor;

import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.test.tools.ui.common.internal.editor.BehaviorAccessories.ExtendedObjectTree;
import org.eclipse.hyades.test.tools.ui.common.internal.util.FormUtil;
import org.eclipse.hyades.test.tools.ui.common.internal.util.IITestSuiteProvider;
import org.eclipse.hyades.test.tools.ui.common.internal.util.FormUtil.CommonSection;
import org.eclipse.hyades.test.ui.internal.editor.form.base.StructuredViewerPart;
import org.eclipse.hyades.test.ui.internal.editor.form.util.DetailPropertiesSection;
import org.eclipse.hyades.test.ui.internal.editor.form.util.DetailsPageProvider;
import org.eclipse.hyades.test.ui.internal.editor.form.util.EditorEclipseForm;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.ui.editor.IEditorExtension;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.forms.IManagedForm;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.events.ExpansionEvent;
import org.eclipse.ui.forms.events.IExpansionListener;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;


/**
 * This class represents the behavior form that appears in the multi-page test
 * suite editor
 * <p>
 * 
 * @author Ali Mehregani & marcelop
 */
public class BehaviorEclipseForm extends EditorEclipseForm 
implements IITestSuiteProvider, ISelectionChangedListener
{
	/* Help flags */
	protected static final int HELP_FORM_CONTROL = 1;
	protected static final int HELP_SECTION_CONTROL = 2;
	protected static final int HELP_NAMED_ELEMENT_CONTROL = 3;
	
	
	/* The form editor that this form is apart of */
	private IEditorExtension formEditor;
	
	/* The form */
	private ScrolledForm form;
	
	/* Common Section */
	private Section sectionc;
	private CommonSection commonSection;
	
	/* Detail Section */
	private Section sectiond;
	private DetailPropertiesSection propertySection;
	private ExtendedObjectTree behaviourObjectTree;
	
	
	/**
	 * The constructor
	 * 
	 * @param hyadesEditorPart
	 *            The form editor that this form is apart of
	 * @param id
	 *            The id of the form
	 * @param title
	 *            The title of the form
	 */
	public BehaviorEclipseForm(IEditorExtension editor, String editorTitle, String id, String title) 
	{
		super(editor, editorTitle, id, title);
		formEditor = editor;
	}

	/**
	 * A chance to clean up after we're done
	 */
	public void dispose()
	{
		if (form != null)
		{
			form.dispose();
			form = null;
		}
		
		if (sectionc != null)
		{
			sectionc.dispose();
			sectionc = null;
		}
		
		if (commonSection != null)
		{
			commonSection.dispose();
			commonSection = null;
		}
		
		if (sectiond != null)
		{
			sectiond.dispose();
			sectiond = null;
		}
		
		if (propertySection != null)
		{
			propertySection.dispose();
			propertySection = null;
		}
		
		super.dispose();
	}
	
	
	/**
	 * Initial method called when creating the form's content.
	 */
	protected void createFormContent(IManagedForm managedForm)
	{
		form = managedForm.getForm();
		super.createFormContent(managedForm);		
	}
	
	
    protected void createRightColumnControls(Composite parent) 
    {
    	FormToolkit toolkit = ((FormEditor)formEditor.getHyadesEditorPart()).getToolkit();
		Control commonSection = createCommonPropertySection (toolkit, parent);
		propertySection = new DetailPropertiesSection(formEditor, this, toolkit);
		Control detailSection = createDetailPropertySection(toolkit, parent);
		if (detailSection != null)
			detailSection.setLayoutData(new GridData(SWT.FILL, SWT.BEGINNING, true, false));
		commonSection.setLayoutData(new GridData(SWT.FILL, SWT.BEGINNING, true, false));
	}
     
    
    /**
     * Create thje common property section on the right hand side of the 
     * behavior form
     * 
     * @param toolkit The forms toolkit
     * @param parent The parent composite
     */
    protected Control createCommonPropertySection (FormToolkit toolkit, Composite parent)
    {
    	sectionc = FormUtil.createSection(toolkit, parent, form);

		/* The name and the description text boxes */
    	commonSection = new CommonSection(CommonSection.BEHAVIOR);
		Composite sectionClient = commonSection.createControl(formEditor, sectionc, toolkit, "", "");

		FormUtil.setSectionProperty (toolkit, sectionc, sectionClient, UiPluginResourceBundle.TTL_CMN_PROPS);
		sectionc.addExpansionListener(new IExpansionListener(){

			public void expansionStateChanged(ExpansionEvent e)
			{
				
			}

			public void expansionStateChanging(ExpansionEvent e)
			{
				BehaviorEclipseForm.this.getForm().reflow(false);
				
			}});
		sectionc.setExpanded(getCommonExpansionState());
		return sectionc;    	
    }
    
    
    /**
     * Create the detail property section on the right hand side of the 
     * behavior form
     * 
     * @param parent The parent composite
     * @param header The header of the section
     * @param desc The description of the section
     */
    protected Control createDetailPropertySection (FormToolkit toolkit, Composite parent)
    {
		DetailsPageProvider detailPageProvider = new DetailsPageProvider();
		addDetails(detailPageProvider);	
		
		sectiond = FormUtil.createSection(toolkit, parent, form);	
		Control sectionClient = propertySection.createDefaultControl(sectiond);
		propertySection.setDetailPageProvider(detailPageProvider);
		
		FormUtil.setSectionProperty (toolkit, sectiond, sectionClient, UiPluginResourceBundle.TTL_DTL_PROPS);
		sectiond.addExpansionListener(new IExpansionListener(){

			public void expansionStateChanged(ExpansionEvent e)
			{
				
			}

			public void expansionStateChanging(ExpansionEvent e)
			{
				BehaviorEclipseForm.this.getForm().reflow(false);
				
			}});
		sectiond.setExpanded(getDetailsExpansionState());
		return sectiond;
    }
    
    
    /**
     * Returns the original expansion state of the common properties 
     * section.  This method returns true by default.
     * 
     * @return true if the common properties section is suppose to be 
     * expanded; false otherwise
     */
    protected boolean getCommonExpansionState()
    {
    	return true;
    }
    
    
    /**
     * Returns the original expansion state of the detail property 
     * section.  This method returns true by default.
     * 
     * @return true if the detail property section is suppose to be 
     * expanded; false otherwise
     */
    protected boolean getDetailsExpansionState()
    {
    	return true;
    }
    
    
    /**
     * Returns the detailed sections.  May be null if it hasn't yet been created.
     * 
     * @return The detail section
     */
    protected Section getDetailSection()
    {
    	return sectiond;
    }
    
    /**
     * Returns the common sections.  May be null if it hasn't yet been created.
     * 
     * @return The common section
     */
    protected Section getCommonSection()
    {
    	return sectionc;
    }

	/**
	 * Create the left column controls for this form
	 * 
	 * @param parent
	 *            The parent composite
	 */
	protected void createLeftColumnControls(Composite parent)
	{
		Control control = createBehaviorSection (parent, UiPluginResourceBundle.W_BEHAVIOR, UiPluginResourceBundle.EDT_TC_BEH_DESCRIPTION);
		control.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));
	}
	
	protected Control createBehaviorSection (Composite parent, String header, String desc)
	{
		FormToolkit toolkit = ((FormEditor)formEditor.getHyadesEditorPart()).getToolkit();

		Section section = FormUtil.createSection(toolkit, parent, form);
		
		BehaviorAccessories ba = new BehaviorAccessories();
		behaviourObjectTree = ba.new ExtendedObjectTree(this);
		Control control = behaviourObjectTree.createControls(section, toolkit);		
		control.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.VERTICAL_ALIGN_BEGINNING));		
		
		/* Set the input of the tree created */
		behaviourObjectTree.getViewerPart().getViewer().setInput(getTestSuite().getImplementor().getBlock());
		behaviourObjectTree.getTreeViewer().addSelectionChangedListener(this);
		
		/* The button are all suppose to initially be disabled, except for add child */		
		toggleEnableStatus(0, 0, true);
		toggleEnableStatus(1, 4, false);
		
		/* Set the properties of the section */
		FormUtil.setSectionProperty (toolkit, section, control, header);

		registerHelp(HELP_FORM_CONTROL, parent);
		registerHelp(HELP_SECTION_CONTROL, control);
		
		return section;
		
	}

	
	/**
	 * Return the test suite that this form is associated with
	 * 
	 * @return The test suite
	 */
    public ITestSuite getTestSuite() 
    {
		return super.getCommonTestSuite();
	}

    
    /**
     * Needs to be over-written by clients 
     */
	protected void registerHelp(int flag, Control object)
	{
	}

	
	/**
	 * This method is invoked when there is a selection made in the behavior tree 
	 */
	public void selectionChanged(SelectionChangedEvent event)
	{
		if (!(event.getSource() instanceof TreeViewer))
			return;
						
		IStructuredSelection structuredSelection = (IStructuredSelection)((TreeViewer)event.getSource()).getSelection();
		Object selection = structuredSelection.getFirstElement();
		commonSection.setInput(selection);
		
		if(propertySection != null)
			propertySection.setInput(structuredSelection);
		
	}
	
	
	/**
	 * @see org.eclipse.ui.part.ISetSelectionTarget#selectReveal(org.eclipse.jface.viewers.ISelection)
	 */
	public void selectReveal(ISelection selection)
	{
		behaviourObjectTree.selectReveal(selection);
	}
	
	
	protected void toggleEnableStatus(int start, int end, boolean isEnabled) 
	{
		StructuredViewerPart viewerPart = behaviourObjectTree.getViewerPart();
		
		for (int i = start; i <= end; i++)
			viewerPart.setButtonEnabled(i, isEnabled);
	}

	public ExtendedObjectTree getBehaviourObjectTree() {
		return behaviourObjectTree;
	}

}
