/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestCasesForm.java,v 1.21 2010/05/20 12:47:31 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.ui.common.internal.editor;

import java.util.Iterator;
import java.util.List;

import org.eclipse.emf.common.util.EList;
import org.eclipse.hyades.models.common.facades.behavioral.IAction;
import org.eclipse.hyades.models.common.facades.behavioral.IBlock;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.fragments.BVRCombinedFragment;
import org.eclipse.hyades.models.common.fragments.BVRInteraction;
import org.eclipse.hyades.models.common.fragments.BVRInteractionOperand;
import org.eclipse.hyades.models.common.interactions.BVRExecutionOccurrence;
import org.eclipse.hyades.models.common.interactions.BVRInteractionFragment;
import org.eclipse.hyades.models.common.testprofile.Common_TestprofilePackage;
import org.eclipse.hyades.test.tools.ui.common.internal.util.IITestSuiteProvider;
import org.eclipse.hyades.test.ui.editor.extension.BaseEditorExtension;
import org.eclipse.hyades.test.ui.editor.form.util.EditorForm;
import org.eclipse.hyades.test.ui.editor.form.util.NamedElementSection;
import org.eclipse.hyades.test.ui.editor.form.util.WidgetFactory;
import org.eclipse.hyades.test.ui.internal.editor.form.base.FormSection;
import org.eclipse.hyades.test.ui.internal.editor.form.base.FormWidgetFactory;
import org.eclipse.hyades.test.ui.internal.editor.form.util.DetailSection;
import org.eclipse.hyades.test.ui.internal.editor.form.util.EObjectTreeSection;
import org.eclipse.hyades.test.ui.internal.editor.form.util.StaticDetailPageProvider;
import org.eclipse.hyades.test.ui.internal.editor.form.util.TreeSection;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Tree;

/**
 * Form for the <code>Test Cases</code> tab.
 * <p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Paul E. Slauenwhite
 * @version May 20, 2010
 * @since   February 1, 2005
 */
public class TestCasesForm extends EditorForm implements ISelectionChangedListener, IITestSuiteProvider {
	
	protected static final int HELP_NAMED_ELEMENT_CONTROL = 1;
	protected static final int HELP_TEST_CASES_SECTION_CONTROL = 2;
	protected static final int HELP_TEST_CASES_SECTION_TREE_CONTROL = 3;
	protected static final int HELP_DETAIL_SECTION_CONTROL = 4;

	private TreeSection testCasesSection;
	private NamedElementSection namedElementSection;
	private DetailSection detailSection;
	private boolean firstActivation = true;
    private TestInvocationCleaner testInvocationCleaner;

	
	/**
	 * Constructor for CommonTestCasesForm
	 * @param testSuiteEditorExtension
	 * @param widgetFactory
	 */
	public TestCasesForm(BaseEditorExtension testSuiteEditorExtension, WidgetFactory widgetFactory)
	{
		super(testSuiteEditorExtension, widgetFactory);
        ITestSuite ts = getTestSuite();
        if (ts!= null) {
            IBlock block = ts.getImplementor().getBlock();
            if (block instanceof BVRInteraction) {
                BVRInteraction interaction = (BVRInteraction) block;
                //- install the cleaner if it is not yet present in the adapters list of the block.
                //- this is to catch add/remove of new invocation
                if (testInvocationCleaner == null || !interaction.eAdapters().contains(testInvocationCleaner)) {
                    testInvocationCleaner = new TestInvocationCleaner();
                    interaction.eAdapters().add(testInvocationCleaner);
                }
                //- add adapter in existing test invocations
                List actions = block.getActions();
                for (Iterator it = actions.iterator(); it.hasNext();) {
                    IAction action = (IAction) it.next();
                    //- case of direct invocation
                    if (action instanceof BVRExecutionOccurrence) {
                        addInvocationCleanerAdapter((BVRExecutionOccurrence) action);
                    } else if (action instanceof BVRCombinedFragment) {
                        //- case of loops or conditions
                        BVRCombinedFragment combi = (BVRCombinedFragment) action;
                        EList operands = combi.getInteractionOperands();
                        for (Iterator iter = operands.iterator(); iter.hasNext();) {
                            BVRInteractionOperand operand = (BVRInteractionOperand) iter.next();
                            EList fragments = operand.getInteractionFragments();
                            for (Iterator iterator = fragments.iterator(); iterator.hasNext();) {
                                BVRInteractionFragment interact = (BVRInteractionFragment) iterator.next();
                                if (interact instanceof BVRExecutionOccurrence) {
                                    addInvocationCleanerAdapter((BVRExecutionOccurrence) interact);
                                }
                            }
                        }
                    }
                }
            }
        }
	}

    /**
     * @param occur
     */
    private void addInvocationCleanerAdapter(BVRExecutionOccurrence occurrence) {
        if (occurrence != null && !occurrence.eAdapters().contains(testInvocationCleaner)) {
            occurrence.eAdapters().add(testInvocationCleaner);
        }
    }

	/**
	 * @see org.eclipse.update.ui.forms.internal.IForm#dispose()
	 */
	public void dispose()
	{
		testCasesSection.getTreeViewer().removeSelectionChangedListener(this);
		testCasesSection.dispose();
		namedElementSection.dispose();
		if(detailSection != null)
			detailSection.dispose();
		
		super.dispose();
	}
	
	/**
	 * @see org.eclipse.hyades.test.common.internal.util.IITestSuiteProvider#getTestSuite()
	 */
	public ITestSuite getTestSuite()
	{
		return ((IITestSuiteProvider)getBaseEditorExtension()).getTestSuite();		
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.common.EditorForm#createEditorFormContents(org.eclipse.swt.widgets.Composite)
	 */
	protected void createEditorFormContents(Composite parent)
	{
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		gridLayout.makeColumnsEqualWidth = true;
		gridLayout.marginWidth = 10;
		gridLayout.horizontalSpacing = 15;
		parent.setLayout(gridLayout);

		Composite leftColumn = createColumn(parent);
		Composite rightColumn = createColumn(parent);
		
		createLeftColumn(leftColumn);
		createRightColumn(rightColumn);
	}
	
	protected void createLeftColumn(Composite parent)
	{
		createTestCaseSection(parent);
	}

	protected void createRightColumn(Composite parent)
	{
		createNamedElementSection(parent);
		createDetailSection(parent);
	}
	
	protected void createTestCaseSection(Composite parent)
	{
		testCasesSection = createTestCasesSection();
		registerSection(testCasesSection);
		testCasesSection.setHeaderText(getTestCasesSectionHeaderText());
		testCasesSection.setCollapsable(true);
		Control control = testCasesSection.createControl(parent, getWidgetFactory());
		control.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.VERTICAL_ALIGN_BEGINNING));
		registerHelp(HELP_TEST_CASES_SECTION_CONTROL, control);
		registerHelp(HELP_TEST_CASES_SECTION_TREE_CONTROL, testCasesSection.getTreeViewer().getControl());
	}
	
	protected TreeSection createTestCasesSection()
	{
		return new EObjectTreeSection(this, Common_TestprofilePackage.eINSTANCE.getTPFTestSuite_TestCases(), getAddTestCaseAction());
	}

	protected void createNamedElementSection(Composite parent)
	{
	    class MyNamedElementSection extends NamedElementSection {
	        public MyNamedElementSection(TestCasesForm form) {
	            super(form);
	        }
	        public Composite createClient(Composite parent, FormWidgetFactory formWidgetFactory)
	        {
	            Composite comp = super.createClient(parent, formWidgetFactory);
	            StyledText text = getDescriptionTextField();
	            text.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.VERTICAL_ALIGN_BEGINNING | GridData.GRAB_VERTICAL));
	            return comp;
	        }
	    }
        namedElementSection = new MyNamedElementSection(this);

		namedElementSection.setHeaderText(UiPluginResourceBundle.EDT_GENERAL_INFO); 
		registerSection(namedElementSection);		
		Control control = namedElementSection.createControl(parent, getWidgetFactory());
		GridData gridData = new GridData(GridData.FILL_BOTH | GridData.VERTICAL_ALIGN_BEGINNING);
		control.setLayoutData(gridData);
		registerHelp(HELP_NAMED_ELEMENT_CONTROL, control);	
	}
	
	protected void createDetailSection(Composite parent)
	{
		StaticDetailPageProvider detailPageProvider = new StaticDetailPageProvider();
		addDetails(detailPageProvider);	
		if(!detailPageProvider.isEmpty())
		{
			detailSection = new DetailSection(this);
			detailSection.setDetailPageProvider(detailPageProvider);
			detailSection.setHeaderText(UiPluginResourceBundle.TTL_DTL_PROPS); 
			registerSection(detailSection);
			Control control = detailSection.createControl(parent, getWidgetFactory());
			control.setLayoutData(new GridData(GridData.FILL_BOTH | GridData.VERTICAL_ALIGN_BEGINNING));
			registerHelp(HELP_DETAIL_SECTION_CONTROL, control);
		}
	}
	
	protected org.eclipse.jface.action.IAction getAddTestCaseAction()
	{
		return null;
	}
	
	protected String getTestCasesSectionHeaderText()
	{
		return UiPluginResourceBundle.W_TST_CASES; 
	}
		
	protected void registerHelp(int flag, Object object)
	{
	}

	protected void addDetails(StaticDetailPageProvider detailPageProvider)
	{
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.common.EditorForm#load()
	 */
	public void load()
	{
		testCasesSection.getTreeViewer().removeSelectionChangedListener(this);
		testCasesSection.setInput(getTestSuite());		
		testCasesSection.getTreeViewer().addSelectionChangedListener(this);
		
		//By default, select the first item in the tree:
		Tree tree = testCasesSection.getTreeViewer().getTree();
		
		if(tree.getItemCount() > 0){
			selectReveal(new StructuredSelection(tree.getItem(0).getData()));
		}
	}
	
	/**
	 * @see org.eclipse.ui.part.ISetSelectionTarget#selectReveal(org.eclipse.jface.viewers.ISelection)
	 */
	public void selectReveal(ISelection selection)
	{
		testCasesSection.selectReveal(selection);
	}
	
	/**
	 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
	 */
	public void selectionChanged(SelectionChangedEvent event)
	{
		if(event.getSource() == testCasesSection.getTreeViewer())
		{
			IStructuredSelection structuredSelection = (IStructuredSelection)getSelection();
			namedElementSection.setInput(structuredSelection);
			if(detailSection != null)
				detailSection.setInput(structuredSelection);
		}
	}
	
	public ISelection getSelection()
	{
		return testCasesSection.getTreeViewer().getSelection();
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.common.EditorForm#activated()
	 */
	public boolean activated()
	{
		if(firstActivation)
		{
			firstActivation = false;
			testCasesSection.setFocus();
		}
		
		return true;
	}
	
	protected DetailSection getDetailSection()
	{
		return this.detailSection;
	}
	
	/**
	 * <p>Updates the test cases section.</p>
	 * 
	 * @see FormSection#update()
	 */
	public void updateTestCasesSection() {
		testCasesSection.update();
	}
}
