/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestInvocationDetailPage.java,v 1.24 2009/04/17 00:47:33 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.ui.common.internal.editor;

import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.hyades.models.common.facades.behavioral.ITest;
import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.models.common.facades.behavioral.ITestInvocation;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.testprofile.TPFTestCase;
import org.eclipse.hyades.models.common.testprofile.TPFTestSuite;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.tools.ui.common.internal.dialog.InvokedTestSelectionDialog;
import org.eclipse.hyades.test.tools.ui.internal.resources.ToolsUiPluginResourceBundle;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.TestUIImages;
import org.eclipse.hyades.test.ui.editor.form.base.IHyperlinkListener;
import org.eclipse.hyades.test.ui.editor.form.util.AbstractDetailPage;
import org.eclipse.hyades.test.ui.editor.form.util.IDetailPageContext;
import org.eclipse.hyades.test.ui.editor.form.util.WidgetFactory;
import org.eclipse.hyades.test.ui.internal.editor.form.base.FormWidgetFactory;
import org.eclipse.hyades.test.ui.internal.editor.form.base.SelectableFormLabel;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.util.TestUIUtil;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.editor.IHyadesEditorPart;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.model.IWorkbenchAdapter;

/**
 * <p>Test invocation pane in the behavior tabbed pane used in various test editors.</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Paul E. Slauenwhite
 * @version April 16, 2009
 * @since   February 1, 2005
 * @see     AbstractDetailPage
 * @see     SelectionListener
 * @see     IHyperlinkListener
 */
abstract public class TestInvocationDetailPage extends AbstractDetailPage implements SelectionListener, IHyperlinkListener
{
	protected static final int HELP_PAGE_CONTROL = 1;
	protected static final int HELP_SYNCHRONOUS_CONTROL = 2;
	protected static final int HELP_FILE_CONTROL = 3;
	protected static final int HELP_EDIT_CONTROL = 4;
	protected static final int HELP_TYPE_CONTROL = 5;

	private Button synchronousButton;
	private SelectableFormLabel testLink;
	private Label imageLabel;
	
	private StyledText typeText;
	private StyledText fileText;
	private Button editButton;
	
	private ITestInvocation testInvocation;
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		if((testLink != null) && (!testLink.isDisposed()))
			testLink.setData(null);
		testInvocation = null;
	}

	/**
	 * @see org.eclipse.hyades.test.ui.editor.form.util.AbstractDetailPage#createControl(org.eclipse.swt.widgets.Composite, org.eclipse.hyades.test.ui.internal.editor.form.base.FormWidgetFactory)
	 */
	public Control createControl(Composite parent, FormWidgetFactory factory, IDetailPageContext context)
	{
		this.context = context;
		WidgetFactory widgetFactory = getEditorForm().getWidgetFactory();
		Composite composite = widgetFactory.createComposite(parent);
		composite.setLayout(new GridLayout());
		composite.setLayoutData(GridDataUtil.createFill());

		synchronousButton = widgetFactory.createButton(composite, getSynchronousLabel(), SWT.CHECK);
		synchronousButton.setLayoutData(GridDataUtil.createHorizontalFill());

		createTestLink(composite);

        //widgetFactory.paintBordersFor(composite);

		registerHelp(HELP_PAGE_CONTROL, composite);				
		registerHelp(HELP_PAGE_CONTROL, synchronousButton);				
		registerHelp(HELP_PAGE_CONTROL, fileText);				
		registerHelp(HELP_PAGE_CONTROL, editButton);				
		registerHelp(HELP_PAGE_CONTROL, typeText);				

		return composite;
	}
	
	/**
	 * Resolves the label for the synchronous check-box.
	 * <p>
	 * 
	 * @return The label for the synchronous check-box.
	 */
	abstract protected String getSynchronousLabel();

	protected void registerHelp(int flag, Object object)
	{
	}
	
	protected void createTestLink(Composite parent)
	{
		WidgetFactory widgetFactory = getEditorForm().getWidgetFactory();
		
		Composite composite = widgetFactory.createComposite(parent);
		GridLayout gridLayout = new GridLayout();
		gridLayout.marginHeight = 0;
		gridLayout.marginWidth = 0;
		composite.setLayout(gridLayout);
		composite.setLayoutData(new GridData(GridData.FILL_BOTH)/*GridDataUtil.createFill()*/);
		
		Composite linkComposite = widgetFactory.createComposite(composite);
		gridLayout = new GridLayout(3, false);
		gridLayout.marginHeight = 0;
		gridLayout.marginWidth = 0;
		linkComposite.setLayout(gridLayout);
		linkComposite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

        imageLabel = widgetFactory.createLabel(linkComposite, ""); //$NON-NLS-1$
		imageLabel.setImage(TestUIImages.INSTANCE.getImage(TestUIImages.IMG_ERROR));
        
		Composite testLinkComposite = widgetFactory.createComposite(linkComposite);
		testLinkComposite.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));

        testLink = widgetFactory.createSelectableLabel(testLinkComposite, null);
        testLink.setText(getDefaultLinkText());
		testLink.setToolTipText(getDefaultLinkToolTip()); 
        testLink.setLayoutData(new GridData(SWT.LEFT, SWT.CENTER, true, false));
		widgetFactory.turnIntoHyperlink(testLink, this);
        
        editButton = widgetFactory.createButton(linkComposite, UiPluginResourceBundle.label_Update, SWT.PUSH); 
        editButton.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, false, false));

        Composite detailComposite = widgetFactory.createComposite(composite);
		gridLayout = new GridLayout();
		gridLayout.marginHeight = 2;
		gridLayout.marginWidth = 2;
		detailComposite.setLayout(gridLayout);
		detailComposite.setLayoutData(GridDataUtil.createFill());

		widgetFactory.createLabel(detailComposite, UiPluginResourceBundle.LBL_TYPE); 
		typeText = widgetFactory.createStyledText(detailComposite, SWT.FULL_SELECTION | SWT.SINGLE);
        typeText.setEditable(false);
		typeText.setLayoutData(GridDataUtil.createHorizontalFill());

		widgetFactory.createLabel(detailComposite, UiPluginResourceBundle.LBL_FILE); 
		fileText = widgetFactory.createStyledText(detailComposite, SWT.FULL_SELECTION | SWT.SINGLE);
        fileText.setEditable(false);
		fileText.setLayoutData(GridDataUtil.createHorizontalFill());
        widgetFactory.paintBordersFor(detailComposite);
	}
	
	protected String getDefaultLinkText()
	{
		return "x"; //$NON-NLS-1$
	}

	protected String getDefaultLinkToolTip()
	{
		return ""; //$NON-NLS-1$
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EditorSection#setInput(java.lang.Object)
	 */
	public void setInput(Object object)
	{
		testInvocation = null;
		synchronousButton.removeSelectionListener(this);
		editButton.removeSelectionListener(this);

		if(object instanceof IStructuredSelection)
		{
			IStructuredSelection structuredSelection = (IStructuredSelection)object;
			if(structuredSelection.size() == 1)
				object = structuredSelection.getFirstElement();
		}

		boolean doPack = false;
		boolean testIsSet = false;
		if(object instanceof ITestInvocation)
		{
			testInvocation = (ITestInvocation)object;
			ITest test = testInvocation.getInvokedTest();
			doPack = true;
			
			editButton.setEnabled(true);
			synchronousButton.setSelection(testInvocation.isSynchronous());
			synchronousButton.setEnabled(true);

			if(test != null)
			{		
				testIsSet = true;
				
				ImageDescriptor imageDescriptor = null;
				if(test instanceof IAdaptable)
				{
					IWorkbenchAdapter workbenchAdapter = (IWorkbenchAdapter)((IAdaptable)test).getAdapter(IWorkbenchAdapter.class);
					if((workbenchAdapter != null))
						imageDescriptor = workbenchAdapter.getImageDescriptor(test);
				}
				if(imageDescriptor == null)
					imageDescriptor = TestUIImages.INSTANCE.getImageDescriptor(TestUIImages.IMG_DEFAULT);

				imageLabel.setImage(imageDescriptor.createImage());
				
				if(test.getName() != null)
					testLink.setText(test.getName());
				else
					testLink.setText(ToolsUiPluginResourceBundle.W_TEST); 
				testLink.setData(test);
				
				String value = test.getType();
				if(value != null)
				{
					IAssociationDescriptor descriptor = null;
					if(test instanceof ITestSuite)
						descriptor = TestUIExtension.getTestSuiteMappingRegistry().getAssociationMapping(HyadesUIPlugin.EP_TYPE_DESCRIPTIONS).getDefaultAssociationDescriptor(value);
					else if(test instanceof ITestCase)
						descriptor = TestUIExtension.getTestCaseMappingRegistry().getAssociationMapping(HyadesUIPlugin.EP_TYPE_DESCRIPTIONS).getDefaultAssociationDescriptor(value);
						
					if((descriptor != null) && (descriptor.getName() != null))
						value = descriptor.getName();
					typeText.setText(value);
				}
					
				fileText.setText(EMFUtil.getFilePath((EObject)test));
			}						
		}
		else
		{
			editButton.setEnabled(false);
		}
		
		if(!testIsSet)
		{
			doPack = true;
			imageLabel.setImage(TestUIImages.INSTANCE.getImage(TestUIImages.IMG_ERROR));
			testLink.setText(getDefaultLinkText());
			testLink.setData(null);
			fileText.setText(""); //$NON-NLS-1$
			typeText.setText(""); //$NON-NLS-1$
			synchronousButton.setEnabled(false);
		}
		
		if(doPack)
		{
			imageLabel.pack(true);
			imageLabel.redraw();
			testLink.pack(true);
			testLink.redraw();			
		}
		
		editButton.addSelectionListener(this);
		synchronousButton.addSelectionListener(this);
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.util.EditorSection#getInput()
	 */
	public Object getInput()
	{
		return testInvocation;
	}

	/**
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e)
	{
		e.doit = !isReadOnly();
	}

	/**
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e)
	{
		if(isReadOnly())
		{
			e.doit = false;
		}
		else
		{
			if(e.widget == editButton)
			{
				ITestSuite testSuite = (ITestSuite)EcoreUtil.getRootContainer((EObject)testInvocation);
				InvokedTestSelectionDialog selectionDialog = createInvokedTestSelectionDialog(testSuite);
				selectionDialog.setTest((ITest)testLink.getData());
				if(selectionDialog.open() == IDialogConstants.OK_ID)
				{
					ITest test = (ITest)selectionDialog.getTests().iterator().next();
					ResourceSet resourceSet = ((EObject)testSuite).eResource().getResourceSet();
					test = (ITest)EMFUtil.getEObject(resourceSet, (EObject)test, true);
					
					if(testInvocation.getInvokedTest() != test)
					{
						testInvocation.setInvokedTest(test);			
						setInput(testInvocation);
					}
				}
			}
			
			else if(e.widget == synchronousButton)
			{
				testInvocation.setSynchronous(synchronousButton.getSelection());
				getEditorForm().getBaseEditorExtension().markDirty();
			}
		}			
	}
	
	abstract protected InvokedTestSelectionDialog createInvokedTestSelectionDialog(ITestSuite testSuite);
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.IHyperlinkListener#linkActivated(org.eclipse.swt.widgets.Control)
	 */
	public void linkActivated(Control linkLabel)
	{
		Object data = linkLabel.getData();
		TestSuiteEditorExtension ext = (TestSuiteEditorExtension) getEditorForm().getBaseEditorExtension();
		if(data != null)
		{
			EObject eObject = (EObject)data;
			if (eObject instanceof TPFTestSuite) {
				IEditorPart editorPart = TestUIUtil.openEditor(eObject.eResource(), null, false);
				if (editorPart instanceof ISelectionProvider) {
					((ISelectionProvider) editorPart).setSelection(new StructuredSelection(eObject));
				}
				return;
			} else if(eObject instanceof TPFTestCase) {
				TPFTestCase tc = (TPFTestCase)eObject;
				Object editorObject = ext.getHyadesEditorPart().getEditorObject();
				if(editorObject != tc.getTestSuite()) {
					IEditorPart editorPart = TestUIUtil.openEditor(eObject.eResource(), null, false);
					if (editorPart instanceof IHyadesEditorPart) {
						IHyadesEditorPart hep = (IHyadesEditorPart) editorPart;
						hep.setActivePage(TestSuiteEditorExtension.PAGE_TEST_CASES);
					}
				} else {
					IHyadesEditorPart editorPart = ext.getHyadesEditorPart();
					editorPart.setActivePage(TestSuiteEditorExtension.PAGE_TEST_CASES);
				}
				ext.getTestCasesForm().selectReveal(new StructuredSelection(eObject));
				return;
			}
		}
		
		IActionBars actionBars = ext.getHyadesEditorPart().getEditorPart().getEditorSite().getActionBars();
		IStatusLineManager manager = actionBars.getStatusLineManager();
		if (manager != null)
			manager.setErrorMessage(UiPluginResourceBundle._ERROR_MSG_UN_OPEN_OBJ);	 
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.IHyperlinkListener#linkEntered(org.eclipse.swt.widgets.Control)
	 */
	public void linkEntered(Control linkLabel)
	{
		IActionBars actionBars = getEditorForm().getBaseEditorExtension().getHyadesEditorPart().getEditorPart().getEditorSite().getActionBars();
		IStatusLineManager manager = actionBars.getStatusLineManager();
		if (manager != null)
			manager.setMessage(((SelectableFormLabel)linkLabel).getText());
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.IHyperlinkListener#linkExited(org.eclipse.swt.widgets.Control)
	 */
	public void linkExited(Control linkLabel)
	{
		IActionBars actionBars = getEditorForm().getBaseEditorExtension().getHyadesEditorPart().getEditorPart().getEditorSite().getActionBars();
		IStatusLineManager manager = actionBars.getStatusLineManager();
		if (manager != null)
		{
			manager.setMessage(""); //$NON-NLS-1$
			manager.setErrorMessage(""); //$NON-NLS-1$
		}
	}
}
