/*******************************************************************************
 * Copyright (c) 2006, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FormUtil.java,v 1.11 2009/04/17 00:47:33 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.ui.common.internal.util;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.interactions.BVRInteractionFragment;
import org.eclipse.hyades.test.core.util.EMFUtil;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.editor.extension.BaseEditorExtension;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.ui.editor.IEditorExtension;
import org.eclipse.hyades.ui.extension.IAssociationConstants;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.accessibility.AccessibleAdapter;
import org.eclipse.swt.accessibility.AccessibleEvent;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.ScrolledForm;
import org.eclipse.ui.forms.widgets.Section;


/**
 * <p>A utility class used to create the forms in the multi-page test suite editor.</p>
 * 
 * 
 * @author  Ali Mehregani
 * @author  Paul E. Slauenwhite
 * @version April 16, 2009
 * @since   August 19, 2005
 */
public class FormUtil 
{
	/* Constants for the size of the controls relative to their parents */
	public static final float TEXT_AREA_PARENT_PROPORTION = (float)0.90;
	public static final float TEXT_PARENT_PROPORTION = (float)0.931;
	
	private static FormUtil instance;
	
	private FormUtil()
	{
		
	}
	
	public static FormUtil getInstance()
	{
		if (instance == null)
			instance = new FormUtil();
		return instance;
	}
	
	/**
	 * Creates the type and path controls.
	 * 
	 * 
	 * @param testSuite
	 * 			  The test suite.
	 * @param parent
	 *            The parent composite.
	 * @param sectionClient
	 *            The section client that is being build on.
	 * @param toolkit
	 *            The form toolkit
	 */
	public static Composite addSouthControls(ITestSuite testSuite, Section parent, Composite sectionClient, FormToolkit toolkit)
	{
		createTypeFileControls (testSuite, parent, sectionClient, toolkit);
		return sectionClient;
	}
	
	
	public static Text[] createTypeFileControls(ITestSuite testSuite, Section parent, Composite sectionClient, FormToolkit toolkit)
	{
		final Composite composite = toolkit.createComposite(sectionClient);
		GridLayout gridLayout = new GridLayout(2, false);
		
		GridData compositeGridData = new GridData(SWT.FILL, SWT.DEFAULT, false, false);
		compositeGridData.verticalAlignment = SWT.BOTTOM;
		composite.setLayout(gridLayout);
		composite.setLayoutData(compositeGridData);

		
		GridData td = new GridData(SWT.FILL, SWT.DEFAULT, false, false);
		td.widthHint = 300;
		toolkit.createLabel(composite, UiPluginResourceBundle.LBL_TYPE);
		final Text typeText = toolkit.createText(composite, "", SWT.FULL_SELECTION | SWT.SINGLE);
		typeText.setEditable(false);
		typeText.setLayoutData(td);

		toolkit.createLabel(composite, UiPluginResourceBundle.LBL_FILE);
		final Text fileText = toolkit.createText(composite, "", SWT.FULL_SELECTION | SWT.SINGLE);
		fileText.setEditable(false);
		fileText.setLayoutData(td);

		/* Find the value of the type and the path of the test suite */
		String value = (testSuite == null ? null : testSuite.getType());
		if (value != null)
		{
			IAssociationDescriptor descriptor = TestUIExtension.getTestSuiteMappingRegistry().getAssociationMapping(IAssociationConstants.EP_TYPE_DESCRIPTIONS).getDefaultAssociationDescriptor(value);
			if ((descriptor != null) && (descriptor.getName() != null))
				value = descriptor.getName();
			typeText.setText(value);
		}
		String fileTextValue = (testSuite == null ? null : EMFUtil.getFilePath((EObject) testSuite));
		if (fileTextValue != null)
			fileText.setText(fileTextValue);
		
		toolkit.paintBordersFor(composite);
		return new Text[] {typeText, fileText};
	}
	
	/**
	 * Used to set common properties for sections of forms
	 * 
	 * @param toolkit The form tool kit
	 * @param section The section
	 * @param sectionClient The section client
	 * @param header The header of the section
	 * @param desc The description of the section
	 */
	public static void setSectionProperty (FormToolkit toolkit, Section section, Control sectionClient, String header)
	{
		section.setClient(sectionClient); 
		section.setText(header);
		toolkit.createCompositeSeparator(section);
				
		toolkit.paintBordersFor(section);
		section.setExpanded(false);
		section.setExpanded(true);
	}
	
	
	public static Section createSection (FormToolkit toolkit, Composite parent, final ScrolledForm form)
	{
		Section section = toolkit.createSection(parent, Section.TITLE_BAR | Section.TWISTIE);
		GridData td = new GridData(SWT.FILL, SWT.FILL, true, true);
		section.setLayoutData(td);		
		return section;
	}
		
	/**
	 * <p>
	 * A modify listener that will often mark an editor as dirty when its fields are modified.
	 * </p>
	 * 
	 * @author Ali Mehregani & marcelop
	 */
	public class FormModifyListener implements ModifyListener
	{
		private IEditorExtension editor;
		private Text nameText, descText;
		private int mode;
		private Object input;
		
		public FormModifyListener(IEditorExtension editor, Text nameText, Text descText, int mode)
		{
			this.editor = editor;
			this.nameText = nameText;
			this.descText = descText;
			this.mode = mode;
		}
		
		
		/**
		 * Invoked when a registered field has been changed
		 * 
		 * Pre-condition: setInput is invoked prior to the invokation of this method.
		 */
		public void modifyText(ModifyEvent e) 
		{
			/* We need input to be set */
			if (input == null)
				return;
			
			boolean isDirty = false;
			String oldfield = null, newfield = null;
			newfield = (e.widget == nameText ? nameText.getText() : descText.getText());
						
			switch (mode)
			{
				case CommonSection.OVERVIEW:
					ITestSuite testSuite = (ITestSuite) input;
					oldfield = (e.widget == nameText ? testSuite.getName() : testSuite.getDescription());
					if (oldfield == null) oldfield = "";
					
					if (input instanceof ITestSuite && !oldfield.equals(newfield))
					{
						if (e.widget == nameText)
							testSuite.setName(newfield);
						else
							testSuite.setDescription(newfield);
						isDirty = true;
					}
					break;
				case CommonSection.TEST_CASES:
					ITestCase testCase = (ITestCase) input;
					oldfield = (e.widget == nameText ? testCase.getName() : testCase.getDescription());					
					if (oldfield == null) oldfield = "";
					
					if (input instanceof ITestCase && !oldfield.equals(newfield))
					{
						if (e.widget == nameText)
							testCase.setName(newfield);
						else
							testCase.setDescription(newfield);
						isDirty = true;
					}
					break;

				case CommonSection.BEHAVIOR:
					BVRInteractionFragment testBehavior = (BVRInteractionFragment) input;
					oldfield = (e.widget == nameText ? testBehavior.getName() : testBehavior.getDescription());					
					if (oldfield == null) oldfield = "";
					
					if (input instanceof BVRInteractionFragment && !oldfield.equals(newfield))
					{
						if (e.widget == nameText)
							testBehavior.setName(newfield);
						else
							testBehavior.setDescription(newfield);
						isDirty = true;
					}
					break;
			}
					
			if (isDirty && editor instanceof BaseEditorExtension)
					((BaseEditorExtension)editor).markDirty();
		}

		public void setInput (Object input)
		{
			this.input = input;
		}
	}
	
	
	/**
	 * This is the section that displays the name and description
	 * 
	 * @author amehrega
	 *
	 */
	public static class CommonSection
	{
		public static final byte OVERVIEW = 0x00;
		public static final byte TEST_CASES = 0x01;
		public static final byte BEHAVIOR = 0x02;
		
		private Composite composite;
		private Text nameText, descriptionText;
		private Label namelbl, desclbl;
		private byte mode;
		private FormModifyListener modifyListeners;
		private Object input;
		
		/**
		 * You can construct a common section using 3 modes indicating whether the
		 * section belongs to the overview, test cases, or behavior form (see OVERVIEW,
		 * TEST_CASES, BEHAVIOR)
		 * 
		 * @param mode The mode of this section
		 */
		public CommonSection (byte mode)
		{
			this.mode = mode;
		}
		
		public Composite createControl (IEditorExtension editor, Section parent, FormToolkit toolkit, String name, String desc)
		{
			composite = toolkit.createComposite(parent);	
			GridLayout gridLayout = new GridLayout();
	
			gridLayout.marginHeight = 10;
			gridLayout.marginWidth = 0;
			composite.setLayout(gridLayout);
			composite.setLayoutData(new GridData());
	
			namelbl = toolkit.createLabel(composite, UiPluginResourceBundle.LBL_NAME); 
			nameText = toolkit.createText(composite, name, SWT.FULL_SELECTION | SWT.BORDER | SWT.FLAT);
			GridData nameTextGD = new GridData (SWT.FILL, SWT.DEFAULT, true, false);	
			nameText.setLayoutData(nameTextGD);			
			nameText.setEnabled(false);
			nameText.getAccessible().addAccessibleListener(new AccessibleAdapter()
			{
				public void getName(AccessibleEvent e)
				{
					e.result = UiPluginResourceBundle.W_NAME; 
				}
			});
					
			desclbl = toolkit.createLabel(composite, UiPluginResourceBundle.LBL_DESC); 
			descriptionText = toolkit.createText(composite, desc, SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL);
			GridData descriptionTextGD = new GridData (SWT.FILL, SWT.DEFAULT, true, false);		
			descriptionTextGD.heightHint = 15 * descriptionText.getLineHeight();
			descriptionText.setLayoutData(descriptionTextGD);	
			descriptionText.setEnabled(false);
			descriptionText.getAccessible().addAccessibleListener(new AccessibleAdapter()
			{
				public void getName(AccessibleEvent e)
				{
					e.result = UiPluginResourceBundle.W_DESCRIPTION; 
				}
			});
			
			/* Add the listeners */
			modifyListeners = new FormUtil().new FormModifyListener(editor, nameText, descriptionText, mode);
			modifyListeners.setInput(input);
			nameText.addModifyListener(modifyListeners);
			descriptionText.addModifyListener(modifyListeners);
								
			return composite;
		}
		
		
		/**
		 * A chance for us to clean up
		 */
		public void dispose()
		{
			composite.dispose();
			namelbl.dispose();
			desclbl.dispose();
			nameText.dispose();
			descriptionText.dispose();
			
			composite = null;
			namelbl = null;
			desclbl = null;
			nameText = null;
			descriptionText = null;
			modifyListeners = null;
			input = null;
		}
		
		/**
		 * Setter methods for setting the name and description values
		 */
		public void setInput (Object input)
		{
			String name = null, desc = null;
			
			/* Disable the controls if the input is invalid */
			boolean controlsAreAvailable = nameText != null && descriptionText != null;
			if (controlsAreAvailable && input == null)
			{
				nameText.setEnabled(false);
				descriptionText.setEnabled(false);
			}
			/* Otherwise enable them */
			else if (controlsAreAvailable)
			{
				nameText.setEnabled(true);
				descriptionText.setEnabled(true);
			}
						
			if (modifyListeners != null)
				modifyListeners.setInput (input);
			
			switch (mode)
			{
				case OVERVIEW:
					if (input instanceof ITestSuite)
					{
						name = ((ITestSuite)input).getName();
						desc = ((ITestSuite)input).getDescription();
						this.input = input;
					}
					break;
				case TEST_CASES:
					if (input instanceof ITestCase)
					{
						name = ((ITestCase)input).getName();
						desc = ((ITestCase)input).getDescription();
						this.input = input;
					}
					break;
				case BEHAVIOR:
					if (input instanceof BVRInteractionFragment)
					{
						name = ((BVRInteractionFragment)input).getName();
						desc = ((BVRInteractionFragment)input).getDescription();
						this.input = input;
					}
					break;					
			}
			
			name = (name == null ? "" : name);
			desc = (desc == null ? "" : desc);
			
			if (nameText != null && name != null)
				nameText.setText(name);
			if (descriptionText != null && desc != null)
				descriptionText.setText(desc);
		}
		
	}
}
