/*******************************************************************************
 * Copyright (c) 2005, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TestCaseAdapter.java,v 1.5 2007/05/02 19:36:08 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.ui.common.internal.util;

import java.util.Iterator;

import org.eclipse.emf.common.notify.Notification;
import org.eclipse.emf.common.notify.impl.AdapterImpl;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.hyades.models.common.common.CMNNamedElement;
import org.eclipse.hyades.test.ui.datapool.internal.util.GridDataUtil;
import org.eclipse.hyades.ui.editor.IEditorExtension;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jface.action.IStatusLineManager;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.forms.editor.FormEditor;
import org.eclipse.ui.forms.events.HyperlinkAdapter;
import org.eclipse.ui.forms.events.HyperlinkEvent;
import org.eclipse.ui.forms.widgets.FormToolkit;
import org.eclipse.ui.forms.widgets.Hyperlink;


/**
 * An adapter that is used to notify the overview form of any changes in 
 * the test case form.
 * 
 * @author Ali Mehregani & marcelop
 */
public class TestCaseAdapter 
extends AdapterImpl implements IDisposable
{
	private EStructuralFeature feature;
	private String statusLinePrefix;
	private Composite hyperlinksParent;
	private IEditorExtension editor;
	
	public TestCaseAdapter(Composite hyperlinkParent, EStructuralFeature feature, String statusLinePrefix,
			IEditorExtension formEditor)
	{
		this.feature = feature;
		this.statusLinePrefix = statusLinePrefix;
		this.editor = formEditor;
		setHyperLinkComposite(hyperlinkParent);
		
	}
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		feature = null;
		setTarget(null);
	}

	/**
	 * @see org.eclipse.emf.common.notify.Adapter#notifyChanged(org.eclipse.emf.common.notify.Notification)
	 */
	public void notifyChanged(Notification msg)
	{
		if(msg.getFeature() != feature)
			return;
			
		switch(msg.getEventType())
		{
			case Notification.ADD:
			case Notification.ADD_MANY:
			case Notification.REMOVE:
			case Notification.REMOVE_MANY:
			case Notification.MOVE:
				refreshComposite((EObject)msg.getNotifier());
				break;
		}
	}
	
	public void refreshComposite(EObject notifier)
	{				
		if (hyperlinksParent == null)
			return;
		
		FormToolkit toolkit = ((FormEditor)editor.getHyadesEditorPart()).getToolkit();
		disposeChildren(hyperlinksParent);
		EList namedElements = (EList)notifier.eGet(feature);
		
		for (Iterator i = namedElements.iterator(); i.hasNext();)
		{
			CMNNamedElement element = (CMNNamedElement)i.next();
			if(element.eIsProxy())
				continue;
			

			final Hyperlink link = toolkit.createHyperlink(hyperlinksParent, "", SWT.WRAP);
			link.setData(element);
			link.addHyperlinkListener(new HyperlinkAdapter() {
				public void linkActivated(HyperlinkEvent e) {
					editor.setSelection(new StructuredSelection(e.widget.getData()));
				}
			});
			link.setToolTipText(element.getName());
			link.setText(element.getName());
			GridData linkGD = GridDataUtil.createHorizontalFill();
			link.setLayoutData(linkGD);
			
			/* We need to register our own adapter here so that the label of links are updated when the test case
			 * name is changed */
			AdapterImpl testCaseAdapter = new AdapterImpl() 
			{
				public void notifyChanged(Notification msg)
				{
					switch(msg.getEventType())
					{
						case Notification.SET:
						case Notification.UNSET:	
							link.setText(msg.getNewStringValue());										
							break;
					}								
				}
			};
			element.eAdapters().add(testCaseAdapter);
			link.setData("testcase.element", element);
			link.setData("testcase.adapter", testCaseAdapter);
					
			/* Set the status line message of the editor */
			IActionBars actionBars = editor.getHyadesEditorPart().getEditorPart().getEditorSite().getActionBars();
			IStatusLineManager manager = actionBars.getStatusLineManager();
			if (manager != null)
			{
				String status = (statusLinePrefix != null ? statusLinePrefix + link.getText() : link.getText());
				manager.setMessage (status);
			}			
		}		
		hyperlinksParent.redraw();
		
	}		
	
	protected void disposeChildren(Composite parent)
	{
		if (parent == null)
			return;
		
		Control[] children = parent.getChildren();
		for (int i = 0, maxi = children.length; i < maxi; i++)
		{
			if (children[i] instanceof Hyperlink)
			{
				Hyperlink link = (Hyperlink)children[i];
				Object elementObject = link.getData("testcase.element");
				if (elementObject != null)
				{
					CMNNamedElement element = (CMNNamedElement)elementObject;
					element.eAdapters().remove(link.getData("testcase.adapter"));
				}
			}
	
			children[i].dispose();
		}
			
	}
	
	
	
	/**
	 * A setter method for initializing the hyper link composite.  This is needed because
	 * often the hyper link composite is not ready when this class is instantiated.
	 * 
	 * @param hyperLinkComposite The hyper link composite
	 */
	public void setHyperLinkComposite (Composite hyperLinkComposite)
	{
		this.hyperlinksParent = hyperLinkComposite;
	}
	
	
	/**
	 * Returns the hyper linke composite
	 * 
	 * @return hyper linke composite
	 */
	public Composite getHyperLinkComposite ()
	{
		return hyperlinksParent;
	}
}

