/*******************************************************************************
 * Copyright (c) 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CheckTestCaseAction.java,v 1.1 2010/05/14 16:44:34 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.ui.http.internal.junit.editor;

import org.eclipse.hyades.models.common.facades.behavioral.ITestCase;
import org.eclipse.hyades.test.http.runner.HttpExecutor;
import org.eclipse.hyades.test.http.runner.HttpResponse;
import org.eclipse.hyades.test.tools.core.http.util.RequestHelper;
import org.eclipse.hyades.test.tools.ui.ToolsUiPlugin;
import org.eclipse.hyades.test.tools.ui.internal.resources.ToolsUiPluginResourceBundle;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;
import org.eclipse.ui.actions.BaseSelectionListenerAction;

/**
 * <p>Action to test a HTTP request.</p>
 *   
 * 
 * @author  Paul E. Slauenwhite
 * @version May 14, 2010
 * @since   May 14, 2010
 */
public class CheckTestCaseAction extends BaseSelectionListenerAction implements IDisposable{

	private RequestHelper requestHelper = null;
	private static final String LINE_SEPARATOR = System.getProperty("line.separator", "\n"); //$NON-NLS-1$ //$NON-NLS-2$

	public CheckTestCaseAction() {

		super(ToolsUiPluginResourceBundle.HTTP_TEST_ACTION_TEXT);

		setToolTipText(ToolsUiPluginResourceBundle.HTTP_TEST_ACTION_TOOL_TIP); 
	}

	/* (non-Javadoc)
	 * @see org.eclipse.jface.action.Action#run()
	 */
	public void run() {

		if(requestHelper != null){

			if(requestHelper.getAttribute(RequestHelper.ATT_HOST) == null){
				MessageDialog.openError(Display.getDefault().getActiveShell(), UiPluginResourceBundle.W_ERROR, ToolsUiPluginResourceBundle.HTTP_TEST_MISSING_HOST_ERROR); 
			}
			else{

				String attribute = requestHelper.getAttribute(RequestHelper.ATT_ABS_PATH);

				if((attribute == null) || (!attribute.startsWith("/"))){ //$NON-NLS-1$
					MessageDialog.openError(Display.getDefault().getActiveShell(), UiPluginResourceBundle.W_ERROR, ToolsUiPluginResourceBundle.HTTP_TEST_INVALID_PATH_ERROR); 
				}
				else {

					try{

						HttpResponse response = new HttpExecutor().execute(requestHelper.createHttpRequest());

						StringBuffer testResults = new StringBuffer();

						int code = response.getCode();

						if(code != -1){
							testResults.append(NLS.bind(ToolsUiPluginResourceBundle.HTTP_TEST_RESULTS_RESPONSE_CODE, code));
						}

						String contentType = response.getContentType();

						if((contentType != null) && (contentType.trim().length() > 0)){

							testResults.append(LINE_SEPARATOR);
							testResults.append(NLS.bind(ToolsUiPluginResourceBundle.HTTP_TEST_RESULTS_CONTENT_TYPE, contentType));								
						}

						int contentLength = response.getContentLength();

						if(contentLength != -1){

							testResults.append(LINE_SEPARATOR);
							testResults.append(NLS.bind(ToolsUiPluginResourceBundle.HTTP_TEST_RESULTS_CONTENT_LENGTH, contentLength));
						}

						long elapsedTime = response.getElapsedTime();

						if(elapsedTime != -1){

							testResults.append(LINE_SEPARATOR);
							testResults.append(NLS.bind(ToolsUiPluginResourceBundle.HTTP_TEST_RESULTS_ELAPSED_TIME, elapsedTime));
						}

						long pageResponseTime = response.getPageResponseTime();

						if(pageResponseTime != -1){

							testResults.append(LINE_SEPARATOR);
							testResults.append(NLS.bind(ToolsUiPluginResourceBundle.HTTP_TEST_RESULTS_PAGE_RESPONSE_TIME, pageResponseTime));
						}

						String details = response.getDetail(); 

						if((details != null) && (details.trim().length() > 0)){

							testResults.append(LINE_SEPARATOR);
							testResults.append(NLS.bind(ToolsUiPluginResourceBundle.HTTP_TEST_RESULTS_DETAILS, details));
						}

						String body = response.getBody(); 

						if((body != null) && (body.trim().length() > 0)){

							testResults.append(LINE_SEPARATOR);
							testResults.append(NLS.bind(ToolsUiPluginResourceBundle.HTTP_TEST_RESULTS_BODY, body));
						}

						String testResultsString = testResults.toString().trim();
						
						if(testResultsString.length() == 0){
							MessageDialog.openError(Display.getDefault().getActiveShell(), UiPluginResourceBundle.W_ERROR, ToolsUiPluginResourceBundle.HTTP_TEST_INVALID_RESPONSE_ERROR); 
						}
						else{
							MessageDialog.openInformation(Display.getDefault().getActiveShell(), ToolsUiPluginResourceBundle.HTTP_TEST_DIALOG_TITLE, testResultsString);
						}
					}
					catch (Throwable t){

						ToolsUiPlugin.logError(t);

						MessageDialog.openError(Display.getDefault().getActiveShell(), UiPluginResourceBundle.W_ERROR, NLS.bind(ToolsUiPluginResourceBundle.HTTP_TEST_ERROR, t.getLocalizedMessage())); 
					}
				}
			}
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.ui.actions.BaseSelectionListenerAction#updateSelection(org.eclipse.jface.viewers.IStructuredSelection)
	 */
	protected boolean updateSelection(IStructuredSelection structuredSelection) {

		if(requestHelper != null){

			requestHelper.dispose();
			requestHelper = null;
		}

		if((structuredSelection != null) && (!structuredSelection.isEmpty())){

			Object firstElement = structuredSelection.getFirstElement();

			if(firstElement instanceof ITestCase){

				requestHelper = new RequestHelper();
				requestHelper.setTestCase((ITestCase)(firstElement));

				return true;
			}
		}

		return false;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose(){

		if(requestHelper != null){

			requestHelper.dispose();
			requestHelper = null;
		}
	}
}
