/*******************************************************************************
 * Copyright (c) 2006, 2007 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JUnitTestSuiteFactoriesProvider.java,v 1.4 2007/05/03 17:54:51 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.ui.java.internal.junit.controls;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.eclipse.hyades.test.tools.core.internal.java.modelsync.JUnitExtensionsRegistry;
import org.eclipse.hyades.test.tools.core.internal.java.modelsync.JUnitExtensionsRegistry.JUnitTestSuiteFactoryDescriptor;
import org.eclipse.hyades.test.tools.core.java.IJUnitTestSuiteFactory;
import org.eclipse.hyades.test.tools.ui.java.internal.junit.navigator.RegularJUnitTestSuiteProxyNode;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.ui.extension.IAssociationConstants;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.hyades.ui.internal.extension.AssociationMappingRegistry;
import org.eclipse.jdt.core.IJavaModel;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.graphics.Image;

/**
 * Provides a list of JUnitTestSuiteFactoryDescriptor. This provider accepts as input:
 * <ul>
 *   <li>A java model: all factories are provided</li>
 *   <li>A java project: only factories that support the project are provided</li>
 *   <li>A collection of {@link org.eclipse.hyades.test.tools.ui.java.internal.junit.navigator.JUnitTestSuiteProxyNode}: only factories that support all
 *   of the proxies are provided.</li>
 * </ul>
 * @author jcanches
 *
 */
public class JUnitTestSuiteFactoriesProvider extends LabelProvider implements IStructuredContentProvider {

	private JUnitTestSuiteFactoryDescriptor[] factories;
	
	public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		if (newInput != null && newInput instanceof IJavaProject) {
			factories = computeFactories((IJavaProject)newInput);
		} else if (newInput != null && newInput instanceof IJavaModel) {
			factories = computeFactories();
		} else if (newInput != null && newInput instanceof Collection) {
			factories = computeFactories((Collection)newInput);
		} else {
			factories = new JUnitTestSuiteFactoryDescriptor[0];
		}
	}

	private JUnitTestSuiteFactoryDescriptor[] computeFactories() {
		return JUnitExtensionsRegistry.getInstance().getAvailableFactories(/*updateSupport*/false);
	}
	
	private JUnitTestSuiteFactoryDescriptor[] computeFactories(IJavaProject projectRestriction) {
		JUnitTestSuiteFactoryDescriptor[] ret = computeFactories();
		List list = new ArrayList(ret.length);
		for (int i = 0; i < ret.length; i++) {
			JUnitTestSuiteFactoryDescriptor descr = ret[i];
			IJUnitTestSuiteFactory factory = descr.getFactory();
			if (factory.isSupported(projectRestriction)) {
				list.add(descr);
			}
		}
		return (JUnitTestSuiteFactoryDescriptor[]) list.toArray(new JUnitTestSuiteFactoryDescriptor[list.size()]);
	}
	
	private JUnitTestSuiteFactoryDescriptor[] computeFactories(Collection proxiesRestriction) {
		JUnitTestSuiteFactoryDescriptor[] ret = computeFactories();
		List list = new ArrayList(ret.length);
		for (int i = 0; i < ret.length; i++) {
			JUnitTestSuiteFactoryDescriptor descr = ret[i];
			IJUnitTestSuiteFactory factory = descr.getFactory();
			if (factorySupports(factory, proxiesRestriction)) {
				list.add(descr);
			}
		}
		return (JUnitTestSuiteFactoryDescriptor[]) list.toArray(new JUnitTestSuiteFactoryDescriptor[list.size()]);
	}
	
	private boolean factorySupports(IJUnitTestSuiteFactory factory, Collection proxiesRestriction) {
		for (Iterator it = proxiesRestriction.iterator(); it.hasNext();) {
			RegularJUnitTestSuiteProxyNode node = (RegularJUnitTestSuiteProxyNode) it.next();
			if (!factory.isSupported(node.getCompilationUnit())) {
				return false;
			}
		}
		return true;
	}

	public void dispose() {
		// NOP
	}
	
	public Object[] getElements(Object inputElement) {
		return factories;
	}

	public String getText(Object element) {
		String type = ((JUnitExtensionsRegistry.JUnitTestSuiteFactoryDescriptor)element).getType();
		AssociationMappingRegistry registry = (AssociationMappingRegistry)TestUIExtension.getTestSuiteMappingRegistry();
		IAssociationMapping associationMapping = registry.getAssociationMapping(IAssociationConstants.EP_TYPE_DESCRIPTIONS);
		IAssociationDescriptor descriptor = associationMapping.getDefaultAssociationDescriptor(type);
		return descriptor.getName();
	}
	
	public Image getImage(Object element) {
		String type = ((JUnitExtensionsRegistry.JUnitTestSuiteFactoryDescriptor)element).getType();
		AssociationMappingRegistry registry = (AssociationMappingRegistry)TestUIExtension.getTestSuiteMappingRegistry();
		IAssociationMapping associationMapping = registry.getAssociationMapping(IAssociationConstants.EP_TYPE_DESCRIPTIONS);
		IAssociationDescriptor descriptor = associationMapping.getDefaultAssociationDescriptor(type);
		return descriptor.getImage();
	}
	
	public ImageDescriptor getImageDescriptor(Object element) {
		String type = ((JUnitExtensionsRegistry.JUnitTestSuiteFactoryDescriptor)element).getType();
		AssociationMappingRegistry registry = (AssociationMappingRegistry)TestUIExtension.getTestSuiteMappingRegistry();
		IAssociationMapping associationMapping = registry.getAssociationMapping(IAssociationConstants.EP_TYPE_DESCRIPTIONS);
		IAssociationDescriptor descriptor = associationMapping.getDefaultAssociationDescriptor(type);
		return descriptor.getImageDescriptor();
	}
	
	public JUnitTestSuiteFactoryDescriptor getElement(String type) {
		for (int i = 0; i < factories.length; i++) {
			if (type.equals(factories[i].getType())) {
				return factories[i];
			}
		}
		return null;
	}
	
}
