/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JUnitEditorExtension.java,v 1.19 2010/05/20 12:47:31 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.ui.java.internal.junit.editor;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.OperationCanceledException;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.test.tools.core.internal.common.codegen.AutomaticDependencyUpdater;
import org.eclipse.hyades.test.tools.core.internal.common.codegen.Helper;
import org.eclipse.hyades.test.tools.core.internal.java.codegen.JUnitGenerator;
import org.eclipse.hyades.test.tools.ui.ToolsUiPlugin;
import org.eclipse.hyades.test.tools.ui.common.internal.editor.TestSuiteEditorExtension;
import org.eclipse.hyades.test.tools.ui.internal.resources.ToolsUiPluginResourceBundle;
import org.eclipse.hyades.test.tools.ui.java.internal.junit.preferences.JUnitUIPreferences;
import org.eclipse.hyades.test.tools.ui.java.internal.junit.wizard.UpdateCodeWizard;
import org.eclipse.hyades.test.ui.editor.form.util.WidgetFactory;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.ui.editor.IHyadesEditorPart;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.ltk.core.refactoring.Change;
import org.eclipse.ltk.core.refactoring.Refactoring;
import org.eclipse.ltk.core.refactoring.RefactoringStatus;
import org.eclipse.ltk.ui.refactoring.RefactoringWizardOpenOperation;
import org.eclipse.swt.widgets.MessageBox;

/**
 * <p>JUnitEditorExtension.java</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Paul E. Slauenwhite
 * @version May 20, 2010
 * @since   February 1, 2005 
 */
public class JUnitEditorExtension extends TestSuiteEditorExtension{
	
	protected boolean behaviorFormDisplayed = false;	
	
	/**
	 * @see org.eclipse.hyades.ui.editor.IEditorExtension#createPages()
	 */
	public void createPages()
	{
		IHyadesEditorPart hyadesEditorPart = getHyadesEditorPart();
		WidgetFactory widgetFactory = new WidgetFactory();
		
		setTestSuiteForm(new JUnitForm(this, widgetFactory));
		getTestSuiteForm().setTestCasePageIndex(PAGE_TEST_CASES);
		hyadesEditorPart.addPage(getTestSuiteForm().createControl());
		hyadesEditorPart.setPageText(PAGE_OVERVIEW, ToolsUiPluginResourceBundle.W_OVERVIEW); 

		setTestCasesForm(new JUnitTestCasesForm(this, widgetFactory));
		hyadesEditorPart.addPage(getTestCasesForm().createControl());
		hyadesEditorPart.setPageText(PAGE_TEST_CASES, ToolsUiPluginResourceBundle.W_TST_METHS); 
		
		behaviorFormDisplayed = !getTestSuite().getImplementor().isExternalImplementor();
		if (behaviorFormDisplayed) {
			setBehaviorForm(new JUnitBehaviorForm(this, widgetFactory));
			hyadesEditorPart.addPage(getBehaviorForm().createControl());
			hyadesEditorPart.setPageText(PAGE_BEHAVIOR, UiPluginResourceBundle.W_BEHAVIOR); 
		} else {
			setBehaviorForm(null);
		}
		
		getTestSuiteForm().updateTitle();
	}
	
	public void setBehaviorFormState(boolean state) {
		IHyadesEditorPart hyadesEditorPart = getHyadesEditorPart();
		if (state && !behaviorFormDisplayed) {
			WidgetFactory widgetFactory = new WidgetFactory();
			setBehaviorForm(new JUnitBehaviorForm(this, widgetFactory));
			hyadesEditorPart.addPage(getBehaviorForm().createControl());
			hyadesEditorPart.setPageText(PAGE_BEHAVIOR, UiPluginResourceBundle.W_BEHAVIOR); 
			getBehaviorForm().load();
		} else if (!state && behaviorFormDisplayed) {
			hyadesEditorPart.removePage(2); // This call disposes the behavior form
			setBehaviorForm(null);
		}
		behaviorFormDisplayed = state;
	}
	
	protected void displayCodeGenerationProblem(String message) {
		MessageBox box = new MessageBox(
				getHyadesEditorPart().getEditorPart().getSite().getShell());
		box.setText(ToolsUiPluginResourceBundle.CODE_UPDATE); 
		box.setMessage(message);
		box.open();
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.test.ui.editor.extension.BaseEditorExtension#save(org.eclipse.core.runtime.IProgressMonitor)
	 */
	public void save(IProgressMonitor monitor) throws Exception {
		
		monitor.beginTask("", 2); //$NON-NLS-1$
		
		try {
			
			//Assumption: The generator handles verifying the source folder, package name, and class name (see org.eclipse.hyades.test.tools.core.internal.java.codegen.JUnitGenerator.checkInitialConditions(IProgressMonitor)).
			//Instantiate our generator:
			JUnitGenerator generator = new JUnitGenerator(getTestSuite(), new AutomaticDependencyUpdater());

			//Generate the code (this operation may modify the test suite model):
			if (!performChange(generator)) {
				throw new OperationCanceledException();
			}

			((JUnitForm)getTestSuiteForm()).reloadSourceInfo();
			
			//Update the test cases section to enable the Open button on the selected test case:
			getTestCasesForm().updateTestCasesSection();
			
			//Perform the save:
			super.save(new SubProgressMonitor(monitor, 1));
		} 
		finally {
			monitor.done();
		}
	}

	private boolean performBatchChange(Change change) {
		try {
			change.initializeValidationData(new NullProgressMonitor());
			if (change.isValid(new NullProgressMonitor()).isOK()) {
				change.perform(new NullProgressMonitor());
				return true;
			}
			return false;
		} catch (OperationCanceledException e) {
			return false;
		} catch (CoreException e) {
			ToolsUiPlugin.logError(e);
			return false;
		}
	}
	
	protected boolean performChange(Refactoring refactoring) {

		int previewPref = ToolsUiPlugin.getDefault().getPluginPreferences().getInt(JUnitUIPreferences.CODE_UPDATE_PREVIEW_MODE);
		boolean showWizard = previewPref != JUnitUIPreferences.NEVER_PREVIEW;
		Change change = null;
		try {
			RefactoringStatus status = refactoring.checkAllConditions(new NullProgressMonitor());
			if (status.hasEntries()) {
				// If there's something that the user should be aware of, always show the
				// preview.
				showWizard = true;
			} else {
				change = refactoring.createChange(new NullProgressMonitor());
				if (previewPref == JUnitUIPreferences.DESTRUCTIVE_CHAGES_PREVIEW && !Helper.isDestructiveChange(change)) {
					showWizard = false;
				}
			}
		} catch (OperationCanceledException e1) {
			return false;
		} catch (CoreException e1) {
			ToolsUiPlugin.logError(e1);
			return false;
		}
		if (showWizard || (change != null && !performBatchChange(change))) {
			UpdateCodeWizard wizard = new UpdateCodeWizard(refactoring, showWizard);
			wizard.setDefaultPageTitle(ToolsUiPluginResourceBundle.CODE_UPDATE_PREVIEW); 
			RefactoringWizardOpenOperation op = new RefactoringWizardOpenOperation(wizard);
			try {
				return op.run(this.getHyadesEditorPart().getEditorPart().getEditorSite().getShell(), ToolsUiPluginResourceBundle.CODE_UPDATE) == IDialogConstants.OK_ID; 
			} catch (InterruptedException e) {
				ToolsUiPlugin.logError(e);
				return false;
			}
		}
		return true;
	}
}