/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SourceInfoViewer.java,v 1.30 2010/04/02 17:30:36 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.ui.java.internal.junit.editor;

import java.util.Iterator;

import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.Path;
import org.eclipse.hyades.models.common.configuration.CFGClass;
import org.eclipse.hyades.models.common.configuration.impl.CFGInstanceImpl;
import org.eclipse.hyades.models.common.configuration.impl.Common_ConfigurationFactoryImpl;
import org.eclipse.hyades.models.common.facades.behavioral.IBlock;
import org.eclipse.hyades.models.common.facades.behavioral.IImplementor;
import org.eclipse.hyades.models.common.facades.behavioral.ITest;
import org.eclipse.hyades.models.common.facades.behavioral.ITestSuite;
import org.eclipse.hyades.models.common.fragments.impl.BVRInteractionImpl;
import org.eclipse.hyades.models.common.fragments.impl.Common_Behavior_FragmentsFactoryImpl;
import org.eclipse.hyades.models.common.interactions.impl.BVRLifelineImpl;
import org.eclipse.hyades.models.common.interactions.impl.Common_Behavior_InteractionsFactoryImpl;
import org.eclipse.hyades.models.common.testprofile.TPFBehavior;
import org.eclipse.hyades.models.common.testprofile.TPFTest;
import org.eclipse.hyades.test.tools.core.CorePlugin;
import org.eclipse.hyades.test.tools.core.internal.java.modelsync.JUnitModelUpdater;
import org.eclipse.hyades.test.tools.core.java.JUnitTestSuiteFacade;
import org.eclipse.hyades.test.tools.ui.ToolsUiPlugin;
import org.eclipse.hyades.test.tools.ui.internal.resources.ToolsUiPluginResourceBundle;
import org.eclipse.hyades.test.tools.ui.java.internal.util.TestJavaUtil;
import org.eclipse.hyades.test.ui.editor.form.util.EditorForm;
import org.eclipse.hyades.test.ui.editor.form.util.WidgetFactory;
import org.eclipse.hyades.test.ui.internal.resources.UiPluginResourceBundle;
import org.eclipse.hyades.test.ui.internal.util.SpecialFieldsBidiListener;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.ui.JavaUI;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.custom.VerifyKeyListener;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.dialogs.ElementTreeSelectionDialog;
import org.eclipse.ui.dialogs.SelectionDialog;

/**
 * <p>Source info viewer.</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Paul E. Slauenwhite
 * @version April 2, 2010
 * @since   February 1, 2005
 */
public class SourceInfoViewer 
implements ModifyListener, SelectionListener
{	 
	private final static String ADVANCED_MODE_SYSTEM_PROPERTY = "TPTP_TESTTOOLS_JUNIT_ADVANCED_EDITOR"; //$NON-NLS-1$
	
	private StyledText sourceFolderText;
	private StyledText packageText;
	private StyledText classNameText;
	private Button sourceFolderButton;
	private Button packageButton;
	private Button updateFromCodeButton;
	private Button openClassButton;
	
	private IImplementor implementor;
	private VerifyKeyListener verifyKeyListener;
	private Button externalBehavior;
	private EditorForm editorForm;
	// This variable keeps the block that is detached from the implementor
	// when the "external behavior" check box is checked. This preserved
	// block will be reattached to the implementor, if the check box is
	// unchecked in the same editor session, so that the content of the
	// behavior pane is not lost if the user checked the box by
	// mistake.
	private IBlock preservedBlock;
	private boolean showExternalBehavior;
	private boolean advanced = System.getProperty(ADVANCED_MODE_SYSTEM_PROPERTY) != null;

	public SourceInfoViewer(EditorForm editorForm) {
		this.editorForm = editorForm;
		showExternalBehavior = editorForm.getBaseEditorExtension() instanceof JUnitEditorExtension;
	}
	
	public SourceInfoViewer() {
		// NOP
	}
	
	/**
	 * @see org.eclipse.update.ui.forms.internal.FormSection#dispose()
	 */
	public void dispose() {
		implementor = null;
		preservedBlock = null;
	}
	
	protected boolean isReadOnly() {
		return false;
	}

	protected IImplementor getImplementor() {
		return implementor;
	}
	
	public Composite createClient(Composite parent, VerifyKeyListener verifyKeyListener, WidgetFactory widgetFactory) {
		this.verifyKeyListener = verifyKeyListener;
		
		Composite composite = null;
		if(widgetFactory != null)
			composite = widgetFactory.createComposite(parent);
		else
			composite = new Composite(parent, SWT.NONE);
			
		composite.setLayout(new GridLayout(2, false));
		composite.setLayoutData(GridDataUtil.createFill());
			
		if (showExternalBehavior) {
			if (widgetFactory != null)
				externalBehavior = widgetFactory.createButton(composite, ToolsUiPluginResourceBundle.STR_TBC_CHECK, SWT.CHECK); 
			else {
				externalBehavior = new Button(composite, SWT.CHECK);
				externalBehavior.setText(ToolsUiPluginResourceBundle.STR_TBC_CHECK); 
			}
			externalBehavior.setLayoutData(new GridData(SWT.BEGINNING, SWT.BEGINNING, false, false, 2, 1));
			//externalBehavior.setLayoutData(GridDataUtil.createHorizontalFill());
		}
		
		Label label;
		if(widgetFactory != null) {
			label = widgetFactory.createLabel(composite, ToolsUiPluginResourceBundle.STR_WB_SRCFOLD); 
		} else {
			label = new Label(composite, SWT.NONE);
			label.setText(ToolsUiPluginResourceBundle.STR_WB_SRCFOLD); 
		}
		label.setLayoutData(new GridData(SWT.BEGINNING, SWT.BEGINNING, false, false, 2, 1));

		if(widgetFactory != null)
			sourceFolderText = widgetFactory.createStyledText(composite, SWT.FULL_SELECTION | SWT.SINGLE | SWT.LEFT_TO_RIGHT,
					ToolsUiPluginResourceBundle.STR_WB_SRCFOLD); 
		else
			sourceFolderText =  new StyledText(composite, SWT.FULL_SELECTION | SWT.SINGLE | SWT.LEFT_TO_RIGHT | SWT.BORDER);
		sourceFolderText.setLayoutData(GridDataUtil.createHorizontalFill());
		sourceFolderText.setEditable(false);
		sourceFolderText.addBidiSegmentListener(SpecialFieldsBidiListener.createPathFieldBidiListener());

		if(widgetFactory != null)
		{
			sourceFolderButton = widgetFactory.createButton(composite, ToolsUiPluginResourceBundle.STR_WB_SRCFOLD_BTN, SWT.PUSH); 
		}
		else
		{
			sourceFolderButton = new Button(composite, SWT.PUSH);
			sourceFolderButton.setText(ToolsUiPluginResourceBundle.STR_WB_SRCFOLD_BTN); 
		}
		sourceFolderButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL));
		sourceFolderButton.addSelectionListener(this);
		
		if(widgetFactory != null) {
			label = widgetFactory.createLabel(composite, ToolsUiPluginResourceBundle.STR_WB_PACK); 
		} else {
			label = new Label(composite, SWT.NONE);
			label.setText(ToolsUiPluginResourceBundle.STR_WB_PACK); 
		}
		label.setLayoutData(new GridData(SWT.BEGINNING, SWT.BEGINNING, false, false, 2, 1));

		if(widgetFactory != null)
			packageText = widgetFactory.createStyledText(composite, SWT.FULL_SELECTION | SWT.SINGLE | SWT.LEFT_TO_RIGHT,
					ToolsUiPluginResourceBundle.STR_WB_PACK); 
		else
			packageText =  new StyledText(composite, SWT.FULL_SELECTION | SWT.SINGLE | SWT.LEFT_TO_RIGHT | SWT.BORDER);
		packageText.setLayoutData(GridDataUtil.createHorizontalFill());
		packageText.setEditable(false);
		packageText.addBidiSegmentListener(SpecialFieldsBidiListener.createPackageFieldBidiListener());

		if(widgetFactory != null)
		{
			packageButton = widgetFactory.createButton(composite, ToolsUiPluginResourceBundle.STR_WB_PACK_BTN, SWT.PUSH); 
		}
		else
		{
			packageButton = new Button(composite, SWT.PUSH);
			packageButton.setText(ToolsUiPluginResourceBundle.STR_WB_PACK_BTN); 
		}
		
		packageButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL));
		packageButton.addSelectionListener(this);

		if(widgetFactory != null) {
			label = widgetFactory.createLabel(composite, ToolsUiPluginResourceBundle.STR_WB_CLASS); 
		} else {
			label = new Label(composite, SWT.NONE);
			label.setText(ToolsUiPluginResourceBundle.STR_WB_CLASS); 
		}
		label.setLayoutData(new GridData(SWT.BEGINNING, SWT.BEGINNING, false, false, 2, 1));

		if(widgetFactory != null)
			classNameText = widgetFactory.createStyledText(composite, SWT.FULL_SELECTION | SWT.SINGLE,
					ToolsUiPluginResourceBundle.STR_WB_CLASS); 
		else
			classNameText =  new StyledText(composite, SWT.FULL_SELECTION | SWT.SINGLE | SWT.BORDER);
		classNameText.setEditable(false);
		
		if(widgetFactory != null) {
			classNameText.setLayoutData(GridDataUtil.createHorizontalFill());
			openClassButton = widgetFactory.createButton(composite, UiPluginResourceBundle.ACT_OPEN, SWT.PUSH); 
			openClassButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL));
			openClassButton.addSelectionListener(this);
		} else {
			classNameText.setLayoutData(new GridData(SWT.FILL, SWT.BEGINNING, true, false, 2, 1));
		}
		
		if (advanced) {
			if (showExternalBehavior) {
				updateFromCodeButton = widgetFactory.createButton(composite, "Refresh test methods from code", SWT.PUSH); //$NON-NLS-1$
				updateFromCodeButton.setLayoutData(new GridData(SWT.END, SWT.BEGINNING, false, false, 2, 1));
				updateFromCodeButton.addSelectionListener(this);
			}
		}
		
		if(widgetFactory != null)			
			widgetFactory.paintBordersFor(composite);
		return composite;
	}
	
	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.BaseEditorSection#setInput(java.lang.Object)
	 */
	public void setInput(Object object) {
		implementor = null;

		if(object instanceof IStructuredSelection) {
			IStructuredSelection structuredSelection = (IStructuredSelection)object;
			if(structuredSelection.size() == 1)
				object = structuredSelection.getFirstElement();
		}
		
		if(object instanceof ITest) {
			object = ((ITest)object).getImplementor();
		}

		if((object != null) && (object instanceof IImplementor)) {
			implementor = (IImplementor)object;
			adjustTexts();
			
			sourceFolderText.setEditable(true);
			packageText.setEditable(true);
			classNameText.setEditable(true);
		} else {
			setSourceFolder(""); //$NON-NLS-1$
			setPackage(""); //$NON-NLS-1$
			setClassName(""); //$NON-NLS-1$

			sourceFolderText.setEditable(false);
			packageText.setEditable(false);
			classNameText.setEditable(false);
		}
		
		updateButtonsState();
	}
	
	protected void adjustTexts() {
		if(verifyKeyListener != null) {
			sourceFolderText.removeVerifyKeyListener(verifyKeyListener);
			packageText.removeVerifyKeyListener(verifyKeyListener);
			classNameText.removeVerifyKeyListener(verifyKeyListener);
		}
		sourceFolderText.removeModifyListener(this);
		packageText.removeModifyListener(this);
		classNameText.removeModifyListener(this);
		if (externalBehavior != null) {
			externalBehavior.removeSelectionListener(this);
		}

		String sourceFolderName = getImplementor().getLocation();
		if (sourceFolderName == null) {
			sourceFolderName = ""; //$NON-NLS-1$
		}
		String className = ""; //$NON-NLS-1$
		String packageName = ""; //$NON-NLS-1$
		String value = getImplementor().getResource();
		if(value != null) {
			int index = value.lastIndexOf('.');
			if(index >= 0)
				packageName = value.substring(0, index);
			else
				index = -1;
				
			if(index != (value.length()-1))
				className = value.substring(index+1);
		}
		boolean isExternal = getImplementor().isExternalImplementor();
		
		setSourceFolder(sourceFolderName);
		setPackage(packageName);
		setClassName(className);
		if (externalBehavior != null) {
			externalBehavior.setSelection(isExternal);
		}

		if(verifyKeyListener != null) {
			sourceFolderText.addVerifyKeyListener(verifyKeyListener);
			packageText.addVerifyKeyListener(verifyKeyListener);
			classNameText.addVerifyKeyListener(verifyKeyListener);
		}
		sourceFolderText.addModifyListener(this);
		packageText.addModifyListener(this);
		classNameText.addModifyListener(this);
		if (externalBehavior != null) {
			externalBehavior.addSelectionListener(this);
		}
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.BaseEditorSection#getInput()
	 */
	public Object getInput() {
		return implementor;
	}

	/**
	 * @see org.eclipse.hyades.test.ui.internal.editor.form.base.BaseEditorSection#getStructuredSelection()
	 */
	public IStructuredSelection getStructuredSelection() {
		if(getImplementor() == null)
			return StructuredSelection.EMPTY;
		return new StructuredSelection(getImplementor());
	}
	
	/**
	 * @see org.eclipse.ui.part.ISetSelectionTarget#selectReveal(org.eclipse.jface.viewers.ISelection)
	 */
	public void selectReveal(ISelection selection) {

	}

	/**
	 * @see org.eclipse.swt.events.ModifyListener#modifyText(org.eclipse.swt.events.ModifyEvent)
	 */
	public void modifyText(ModifyEvent e) {

		if (e.widget == sourceFolderText) {
			getImplementor().setLocation(getSourceFolder());
		} 
		else if((e.widget == classNameText) || (e.widget == packageText)) {
			getImplementor().setResource(getQualifiedTypeName());
		}

		updateButtonsState();
		
		markDirty();
	}
	
	/**
	 * <p>Resolves the qualified type name of the test, in the 
	 * format:</p>
	 * 
	 * <p><code>[&lt;package name&gt;.]&lt;class name&gt;</code></p>
	 * 
	 * @return The qualified type name of the test.
	 */
	private String getQualifiedTypeName() {
		
		String packageName = getPackage();
		
		if ((packageName != null) && (packageName.trim().length() > 0)){
			return (packageName + '.' + getClassName());
		} 
			
		return (getClassName());
	}
	
	private void updateButtonsState() {
		packageButton.setEnabled(findSourceFolder() != null);
		if (openClassButton != null) {
			openClassButton.setEnabled(findCompilationUnit() != null);
		}
	}
	
	/**
	 * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	}

	/**
	 * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
	 */
	public void widgetSelected(SelectionEvent e) {
		
		if(isReadOnly()) {
			
			e.doit = false;

			if (e.widget == externalBehavior) {
				
				//Work-around for the selection changing when the selection event is canceled:
				externalBehavior.setSelection(!externalBehavior.getSelection());
			}
		} 
		else {
			if (e.widget == sourceFolderButton) {
				ElementTreeSelectionDialog dialog = TestJavaUtil.createSourceFolderDialog(sourceFolderButton.getShell(), null);
				dialog.setTitle(ToolsUiPluginResourceBundle.TITLE_SRC_SELECT); 
				dialog.setMessage(ToolsUiPluginResourceBundle.STR_SRC_SELECT); 
				dialog.setEmptyListMessage(ToolsUiPluginResourceBundle.NO_SRC); 
				dialog.setInitialSelection(findSourceFolder());
				if (dialog.open() == IDialogConstants.OK_ID) {
					Object result = dialog.getFirstResult();
					String text = ""; //$NON-NLS-1$
					if (result != null &&
						(result instanceof IJavaProject || result instanceof IPackageFragmentRoot)) {
						try {
							text = ((IJavaElement)result).getCorrespondingResource().getFullPath().toString();
						} catch (JavaModelException e1) {
							ToolsUiPlugin.logError(e1);
							// text is ""
						}
					}
					setSourceFolder(text);
					markDirty();
				}
			} else if (e.widget == packageButton) {
				try {
					IPackageFragmentRoot srcFolder = findSourceFolder();
					if (srcFolder != null) {
						String initialPackageName = getPackage();
						SelectionDialog dialog = JavaUI.createPackageDialog(packageButton.getShell(), srcFolder);
						IPackageFragment initialPackage = null;
						if (initialPackageName != null) {
							initialPackage = srcFolder.getPackageFragment(initialPackageName);
						}
						if (initialPackage != null && initialPackage.exists()) {
							dialog.setInitialSelections(new Object[] {initialPackage});
						}
						dialog.setTitle(ToolsUiPluginResourceBundle.TITLE_PACK_SELECT); 
						dialog.setMessage(ToolsUiPluginResourceBundle.STR_PACK_SELECT); 
						if (dialog.open() == IDialogConstants.OK_ID) {
							IPackageFragment packageFragment = (IPackageFragment)dialog.getResult()[0];
							setPackage(packageFragment.getElementName());
							markDirty();
						}
					}
				} catch (JavaModelException e1) {
					e.doit = false;
				}
			} else if (e.widget == externalBehavior) {
				if (externalBehavior.getSelection()) {
					// First, remove the page so it can manipulate block while it is still in the model
					if (editorForm != null) {
						((JUnitEditorExtension)editorForm.getBaseEditorExtension()).setBehaviorFormState(false);
					}
					// Then, we can safely detach block from the model
					preservedBlock = getImplementor().getBlock();
					JUnitModelUpdater.setImplementorBlock(implementor, null);
				} else {
					if (preservedBlock != null) {
						JUnitModelUpdater.setImplementorBlock(getImplementor(), preservedBlock);
					} else {
						createBlock(getImplementor());
					}
					// Add the page after block is set
					if (editorForm != null) {
						((JUnitEditorExtension)editorForm.getBaseEditorExtension()).setBehaviorFormState(true);
					}
				}
				
				markDirty();
			} else if (e.widget == updateFromCodeButton) {
				try {
					ITestSuite testSuite = (ITestSuite)getImplementor().getOwner();
					if (JUnitModelUpdater.updateTestSuiteFromCode(testSuite, JUnitTestSuiteFacade.getJUnitSourceFile(testSuite))) {
						markDirty();
					}
				} catch (CoreException e1) {
					CorePlugin.logError(e1);
				}
			} else if (e.widget == openClassButton) {
				ICompilationUnit cu = findCompilationUnit();
				if (cu != null) {
					try {
						JavaUI.openInEditor(cu);
					} catch (PartInitException e1) {
						ToolsUiPlugin.logError(e1);
						e.doit = false;
					} catch (JavaModelException e1) {
						ToolsUiPlugin.logError(e1);
						e.doit = false;
					}
				}
			}
		}
	}
	
	protected IPackageFragmentRoot findSourceFolder() {

		IPackageFragmentRoot sourceFolder = null;

		IResource resource = ResourcesPlugin.getWorkspace().getRoot().findMember(getSourceFolder());

		if (resource != null) {

			IJavaElement javaElement = JavaCore.create(resource);

			if (javaElement != null){

				try {

					if (javaElement instanceof IJavaProject) {
						sourceFolder = ((IJavaProject)javaElement).findPackageFragmentRoot(resource.getFullPath());
					} 
					else if (javaElement instanceof IPackageFragmentRoot) {
						sourceFolder = (IPackageFragmentRoot)javaElement;
					}
				} 
				catch (JavaModelException e) {
					//Ignore and return null.
				}
			}
		}

		return sourceFolder;
	}

	protected IPackageFragment findPackage() {
		IPackageFragmentRoot srcFolder = findSourceFolder();
		if (srcFolder != null) {
			IPackageFragment pack = srcFolder.getPackageFragment(getPackage());
			if (pack.exists()) {
				return pack;
			}
		}
		return null;
	}
	
	protected ICompilationUnit findCompilationUnit() {
		IPackageFragment pack = findPackage();
		if (pack != null) {
			ICompilationUnit cu = pack.getCompilationUnit(getClassName() + ".java"); //$NON-NLS-1$
			if (cu.exists()) {
				return cu;
			}
		}
		return null;
	}
	
	static IBlock createBlock(IImplementor implementor) {
		if (!(implementor instanceof TPFBehavior)) return null;
		TPFBehavior testBehavior = (TPFBehavior)implementor;
		TPFTest test = testBehavior.getTest();
		if (!(test instanceof CFGClass)) return null;
		CFGClass theClass = (CFGClass)test;
		String implementedTestName = test.getName();
		if (implementedTestName == null) implementedTestName = ""; //$NON-NLS-1$
		
		// Find or create an instance
		CFGInstanceImpl instance = null;
		Iterator iter = theClass.getInstances().iterator();
		while(iter.hasNext())
		{
			CFGInstanceImpl tempInstance = (CFGInstanceImpl)iter.next();
			if(theClass.equals(tempInstance.getClassType())) {
				instance = tempInstance;
				break;
			}
		}
		if(instance == null)
		{
			// Create instance
			instance = (CFGInstanceImpl) Common_ConfigurationFactoryImpl.eINSTANCE.createCFGInstance();
			// Associate instance with Deployable (testsuite)		
			instance.setClassType(theClass);
			theClass.getInstances().add(instance);
		}
		
		// Create lifeline
		BVRLifelineImpl selfLifeline = (BVRLifelineImpl)Common_Behavior_InteractionsFactoryImpl.eINSTANCE.createBVRLifeline();
		selfLifeline.setName(implementedTestName + "_selfLifeline"); //$NON-NLS-1$
	
		// Since this behavior is modeled, create the mandatory Interaction.
		BVRInteractionImpl testInteraction = (BVRInteractionImpl) Common_Behavior_FragmentsFactoryImpl.eINSTANCE.createBVRInteraction();
		testInteraction.setName(implementedTestName + "_interaction"); //$NON-NLS-1$
		testBehavior.setInteraction(testInteraction);
						
		// Add lifeline to interaction
		testInteraction.getLifelines().add(selfLifeline);
	
		// Associate the lifeline to the instance
		instance.setLifeline(selfLifeline);
		
		return testInteraction;
	}

	public String getSourceFolder() {
		Path path = new Path(sourceFolderText.getText().trim());
		return path.toString();
	}
	
	public String getPackage() {
		return packageText.getText().trim();
	}
	
	public String getClassName() {
		return classNameText.getText().trim();
	}
	
	public void setSourceFolder(String srcFolder) {
		sourceFolderText.setText(srcFolder);
	}
	
	public void setPackage(String pack) {
		packageText.setText(pack);
	}
	
	public void setClassName(String className) {
		classNameText.setText(className);
	}
	
	public void setInitialFocus() {

		if((sourceFolderText != null) && (getSourceFolder().length() == 0)){
			sourceFolderText.setFocus();
		}
		else if((packageText != null) && (getPackage().length() == 0)){
			packageText.setFocus();
		}
		else if((classNameText != null) && (getClassName().length() == 0)){
			classNameText.setFocus();
		}
		else{

			classNameText.setFocus();
			classNameText.setSelection(getClassName().length());
		}
	}
	
	protected void markDirty() {
		if (editorForm != null) {
			editorForm.getBaseEditorExtension().markDirty();
		}
	}
	
}