/**********************************************************************
 * Copyright (c) 2009, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: OpenSourceCommandHandler.java,v 1.2 2010/10/25 12:31:06 bjerome Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.test.tools.ui.java.internal.junit.handlers;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.core.resources.IFile;
import org.eclipse.hyades.test.core.testservices.resources.Messages;
import org.eclipse.hyades.test.tools.ui.ToolsUiPlugin;
import org.eclipse.hyades.test.tools.ui.http.internal.junit.navigator.HTTPTestCaseProxyNode;
import org.eclipse.hyades.test.tools.ui.http.internal.junit.navigator.HTTPTestSuiteProxyNode;
import org.eclipse.hyades.test.tools.ui.internal.resources.ToolsUiPluginResourceBundle;
import org.eclipse.hyades.test.tools.ui.java.IJavaSourceTestCaseProxyNode;
import org.eclipse.hyades.test.tools.ui.java.IJavaSourceTestSuiteProxyNode;
import org.eclipse.hyades.test.tools.ui.java.internal.junit.navigator.JUnitTestCaseProxyNode;
import org.eclipse.hyades.test.tools.ui.java.internal.junit.navigator.JUnitTestSuiteProxyNode;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IMethod;
import org.eclipse.jdt.ui.JavaUI;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.handlers.HandlerUtil;

/**
 * <p>Handler for the open source command (see the 
 * <code>org.eclipse.ui.handlers</code> and 
 * <code>org.eclipse.ui.commands</code> extension points in this plug-in's
 * <code>plugin.xml</code> file).</p>
 *  
 * <p>The open source command is used by the 'Open Source Code' context menu 
 * (see <code>org.eclipse.ui.menus</code> extension point) and 
 * 'Alt+Shift+T, O' key binding (see <code>org.eclipse.ui.bindings</code> 
 * extension point).</p>

 * <p>Note: This handler only supports one {@link JUnitTestSuiteProxyNode}, 
 * {@link JUnitTestCaseProxyNode}, {@link HTTPTestSuiteProxyNode}, or {@link HTTPTestCaseProxyNode} 
 * selection in the Test Navigator, as declared in the implementation of the 
 * <code>org.eclipse.ui.handlers</code> extension point.</p>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @author  Jerome Bozier
 * @version October 15, 2010
 * @since   May 4, 2009
 */
public class OpenSourceCommandHandler extends AbstractHandler{

	/* (non-Javadoc)
	 * @see org.eclipse.core.commands.AbstractHandler#setEnabled(java.lang.Object)
	 */
	public void setEnabled(Object evaluationContext) {
		setBaseEnabled(true);
	}

	/**
	 * internal method that open an error message dialog
	 * @param head : displayed text in header
	 * @param name : displayed text in the message
	 */
	private void displayErrorMessage(final String head,final String name) {
		Display disp = Display.getCurrent();
		if (disp != null) {
			Shell shell = disp.getActiveShell();
			if (shell != null) {
				MessageDialog.openError(shell, head, name);		
			}
		} else {
			final Display disp2 = Display.getDefault();
			disp2.syncExec(new Runnable() {
				public void run() {
					MessageDialog.openError(disp2.getActiveShell(), head, name);
				}
			});
		}		
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.core.commands.AbstractHandler#execute(org.eclipse.core.commands.ExecutionEvent)
	 */
	public Object execute(ExecutionEvent event) throws ExecutionException {

		ISelection selection = HandlerUtil.getCurrentSelection(event); 

		if(selection != null){

			IStructuredSelection structuredSelection = null;
			
			if(selection instanceof IStructuredSelection){
				structuredSelection = ((IStructuredSelection)(selection));		
			}
			else{
				structuredSelection = new StructuredSelection(selection);
			}
			
			IProxyNode proxyNode = ((IProxyNode)(structuredSelection.getFirstElement()));
	    	
	    	if(proxyNode instanceof IJavaSourceTestSuiteProxyNode) {
	    		IJavaSourceTestSuiteProxyNode testSuite = (IJavaSourceTestSuiteProxyNode)proxyNode;
	    		IFile sourceFile = testSuite.getSourceFile(); 
		        
	    		if(sourceFile != null) {
		            UIUtil.openEditor(sourceFile, null, false);
		        } else {
		        	displayErrorMessage(ToolsUiPluginResourceBundle.word_Error,NLS.bind(Messages.WorkspaceResourceProviderService_SOURCE_FILE_DNE, testSuite.getSourceFileName()));
		        }
	    		testSuite.unload();
	    	} 
	    	else if(proxyNode instanceof IJavaSourceTestCaseProxyNode) {
	    		
	    		IMethod method = ((IJavaSourceTestCaseProxyNode)proxyNode).getSourceMethod();

    			if (method != null) {

    				try {

    					IEditorPart part = JavaUI.openInEditor(method);

	    				JavaUI.revealInEditor(part, ((IJavaElement)(method)));
	    			} 
	    			catch (Exception e) {
	    				ToolsUiPlugin.logError(e);
	    			}
	    		} else {
	    			displayErrorMessage(ToolsUiPluginResourceBundle.word_Error,NLS.bind(Messages.WorkspaceResourceProviderService_SOURCE_FILE_DNE, ((IJavaSourceTestCaseProxyNode)proxyNode).getTestCaseName()));
	    		}
	    	}
		}		

		return null;
	}
}
