/********************************************************************** 
 * Copyright (c) 2005, 2010 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: JUnitTypeProviderProxyNode.java,v 1.12 2010/05/21 15:28:49 paules Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.tools.ui.java.internal.junit.navigator;

import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.hyades.models.common.util.ICommonConstants;
import org.eclipse.hyades.test.tools.ui.ToolsUiPlugin;
import org.eclipse.hyades.test.ui.TestUIExtension;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.TestNavigator;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.IExtendedProxyNode;
import org.eclipse.hyades.test.ui.navigator.IFileProxyManager;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.IResourceChangeListenerProxyNode;
import org.eclipse.hyades.test.ui.navigator.ITestFolderContentValidator;
import org.eclipse.hyades.test.ui.navigator.ITypeProviderProxyNode;
import org.eclipse.hyades.test.ui.navigator.ITypedElementProxyNode;
import org.eclipse.hyades.ui.extension.IAssociationConstants;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.hyades.ui.internal.extension.AssociationMappingRegistry;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.swt.graphics.Image;

/** 
 * <p>Root of the proxy nodes hierarchy for TPTP JUnit tests.</p>
 * 
 * 
 * @author  Jerome Gout
 * @author  Paul Slauenwhite
 * @version May 21, 2010
 * @since   March 25, 2005
 */
public class JUnitTypeProviderProxyNode implements ITypeProviderProxyNode, IResourceChangeListenerProxyNode, IExtendedProxyNode {
    
    static class JUnitTypeValidator implements ITestFolderContentValidator {
        private String type;
        private List extensions = new LinkedList();

        public JUnitTypeValidator(String type) {
            this.type = type;
            extensions.add(ICommonConstants.TEST_SUITE_FILE_EXTENSION);
            extensions.add(ICommonConstants.EXECUTION_FILE_EXTENSION);
            extensions.add(ICommonConstants.TEST_COMPONENT_EXTENSION);
        }
       
        /**
         * Ok if only relevant typed files. (test suite, test component and test execution files)
         * @see org.eclipse.hyades.test.ui.navigator.ITestFolderContentValidator#isFileOk(org.eclipse.core.resources.IFile)
         */
        public boolean isFileOk(IFile file) {
            return extensions.contains(file.getFileExtension());
        }

        /**
         * Ok if only proxy of the same given type. 
         * @see org.eclipse.hyades.test.ui.navigator.ITestFolderContentValidator#isProxyOk(org.eclipse.hyades.test.ui.navigator.IProxyNode)
         */
        public boolean isProxyOk(IProxyNode node) {
            if(node instanceof ITypedElementProxyNode) {
                return ((ITypedElementProxyNode)node).getType().equals(type);
            }
            return false;
        }
    }
    
    /** the concrete proxy node that this proxy node delegates to. */
    private IProxyNode proxy;
    private String name;
    private String imageKey;
    private final JUnitTypeValidator validator;
	
    public static JUnitTypeProviderProxyNode create(IProject project, String type, IFileProxyManager fileProxyManager, Object parent) {
        JUnitTypeProviderProxyNode proxy = new JUnitTypeProviderProxyNode(project, type, fileProxyManager, parent);
        if(proxy.getDelegate() != null && proxy.getChildren().length > 0) {
            return proxy;
        }
        return null;
    }
    
    private IPackageFragmentRoot getProjectAsSourceFolder(IProject project) {
        IJavaProject jProject = JavaCore.create(project);
        IPackageFragmentRoot root;
        try {
            root = jProject.findPackageFragmentRoot(project.getFullPath());
        } catch (JavaModelException e) {
            root = null;
            ToolsUiPlugin.logError(e);
        }
        return root;
    }
    
    protected JUnitTypeProviderProxyNode(IProject project, String type, IFileProxyManager fileProxyManager, Object parent) {
        validator = new JUnitTypeValidator(type);
        IPackageFragmentRoot root = getProjectAsSourceFolder(project);
        if(root == null) {
            //- the current project is not a source folder
            proxy = createNonSourceFolderProxyNode(project, type, fileProxyManager, parent);
        } else {
            proxy = createSourceFolderProxyNode(fileProxyManager, parent, root);
        }
        AssociationMappingRegistry registry = (AssociationMappingRegistry)TestUIExtension.getTestSuiteMappingRegistry();
        IAssociationMapping associationMapping = registry.getAssociationMapping(IAssociationConstants.EP_TYPE_DESCRIPTIONS);
        IAssociationDescriptor descriptor = associationMapping.getDefaultAssociationDescriptor(type);
        
        this.name = TestNavigator.getTypeProviderManager().getNameFor(type);
        this.imageKey = TestNavigator.getTypeProviderManager().getImageKeyFor(type);
        
        if ((this.name == null) || (this.name.trim().length() == 0)){

        	if(descriptor != null) {
        		this.name = descriptor.getName();
        	}
        	else {
        		this.name = "[Invalid Descriptor]"; //$NON-NLS-1$
        	}
        }
    }

	/**
	 * @param fileProxyManager
	 * @param parent
	 * @param root
	 * @return
	 */
	protected IProxyNode createSourceFolderProxyNode(IFileProxyManager fileProxyManager, Object parent, IPackageFragmentRoot root) {
        SourceFolderProxyNode _proxy = new SourceFolderProxyNode(root, validator, fileProxyManager, parent, this);
        if (_proxy.getChildren().length > 0) {
            return _proxy;
        }
        return null;
    }
	
	/**
	 * @param project
	 * @param type
	 * @param fileProxyManager
	 * @param parent
	 * @return
	 */
	protected IProxyNode createNonSourceFolderProxyNode(IProject project, String type, IFileProxyManager fileProxyManager, Object parent) {
        JUnitFolderProxyNode _proxy = new JUnitFolderProxyNode(project, validator, type, fileProxyManager, parent, this);
        if (_proxy.getChildren().length > 0) {
            return _proxy;
        }
        return null;
    }
    
    protected IProxyNode getDelegate() {
        return proxy;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getText()
     */
    public String getText() {
        return name;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getImage()
     */
    public Image getImage() {
        if(this.imageKey == null || this.imageKey.length() == 0) {
            return proxy.getImage();
        } else {
            return UiPlugin.getInstance().getImageRegistry().get(this.imageKey);
        }
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getParent()
     */
    public Object getParent() {
        if (proxy == null) return null;
        return proxy.getParent();
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getChildren()
     */
    public IProxyNode[] getChildren() {
        return proxy.getChildren();
    }

    /* (non-Javadoc)
     * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
     */
    public Object getAdapter(Class adapter) {
        return proxy.getAdapter(adapter);
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.IProxy#getUnderlyingResource()
     */
    public IResource getUnderlyingResource() {
        return proxy.getUnderlyingResource();
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.IProxy#getIdentifier()
     */
    public String getIdentifier() {
        return proxy.getIdentifier();
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.tools.ui.java.internal.junit.navigator.IProxyNodeResourceChangedListener#resourceChanged(org.eclipse.core.resources.IResourceDelta)
     */
    public IProxyNode resourceChanged(IResourceDelta rootDelta) {
        IProxyNode lowestChanged = this;
		IProject project = (IProject)rootDelta.getResource();
        JavaCore.create(project);
        IPackageFragmentRoot root = getProjectAsSourceFolder(project);
        if(root == null) {
            //- the current project is not a source folder
            lowestChanged = ((JUnitFolderProxyNode)proxy).resourceChanged(rootDelta);
        } else {
            lowestChanged = ((SourceFolderProxyNode)proxy).resourceChanged(rootDelta);
        }
        //- need to return the lowest changed found or 'this' if the lowest changed is equal to the delegate proxy.
		return lowestChanged == proxy ? this : lowestChanged;
    }

	public IResource getCorrespondingResource() {
		//- type provider does not match resource
		return null;
	}

}
