/*******************************************************************************
 * Copyright (c) 2006, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JavaParentElementProxyNode.java,v 1.5 2009/07/31 15:12:30 bjerome Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.test.tools.ui.java.internal.junit.navigator;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaElementDelta;
import org.eclipse.jdt.core.IParent;
import org.eclipse.jdt.core.JavaModelException;

/**
 * Proxy node representing a java element that contains java elements.
 * @author jcanches
 * @author jbozier
 * @since 4.3
 * @version July 31, 2009
 */
public abstract class JavaParentElementProxyNode extends JavaElementProxyNode {

	protected JavaElementProxyNode[] children;
	
	protected JavaParentElementProxyNode(IJavaElement element, Object parent, IJavaElementDelta delta) {
		super(element, parent);
		List c;
		if (delta != null) {
			c = computeChildren(delta);
		} else {
			c = computeChildren();
		}
		setChildren(c);
	}
	
	/**
	 * Special constructor to avoid children computation during the construction time. If so,
	 * the sub-class should remember to invoke setChildren(computeChildren()) in its own constructor.
	 * @param element
	 * @param parent
	 * @param delayChildrenComputation Whether to skip children computation.
	 */
	protected JavaParentElementProxyNode(IJavaElement element, Object parent, boolean delayChildrenComputation) {
		super(element, parent);
		if (!delayChildrenComputation) {
			setChildren(computeChildren());
		}
	}

	/**
	 * Creates a child proxy for the given element.
	 * @param childElement A java element that is a child of the java element associated to this
	 * proxy.
	 * @param delta Optionnally, the java event information that triggers the computation of this child
	 * proxy.
	 * @return The child proxy, or <code>null</code> if the child java element does not have a
	 * matching proxy.
	 */
	protected abstract JavaElementProxyNode createChildProxy(IJavaElement childElement, IJavaElementDelta delta);
	
	protected void disposeChildProxy(JavaElementProxyNode child) {
		child.dispose();
	}
    
	public IProxyNode[] getChildren() {
		return children;
	}
	
	protected final void setChildren(List _children) {
		children = (JavaElementProxyNode[])_children.toArray(new JavaElementProxyNode[_children.size()]);
	}

	protected List computeChildren() {
		if (getJavaElement() instanceof IParent) {
			try {
				IJavaElement[] elements = ((IParent)getJavaElement()).getChildren();
				List _children = new ArrayList(elements.length); 
				for (int i = 0; i < elements.length; i++) {
					JavaElementProxyNode proxy = createChildProxy(elements[i], null);
					if (proxy != null) {
						_children.add(proxy);
					}
				}
				return _children;
			} catch (JavaModelException e) {
				return Collections.EMPTY_LIST;
			}
		}
		return Collections.EMPTY_LIST;
	}
	
	protected List computeChildren(IJavaElementDelta delta) {
		if (getJavaElement() instanceof IParent) {
			IJavaElementDelta[] children = delta.getAffectedChildren();
			List _children = new ArrayList(children.length);
			for (int i = 0; i < children.length; i++) {
				int kind = children[i].getKind();
				if (kind == IJavaElementDelta.ADDED || kind == IJavaElementDelta.CHANGED) {
					JavaElementProxyNode proxy = createChildProxy(children[i].getElement(), children[i]);
					if (proxy != null) {
						_children.add(proxy);
					}
				}
			}
			return _children;
		}
		return Collections.EMPTY_LIST;
	}

	public IProxyNode elementChanged(IJavaElementDelta delta) {
		IProxyNode lowestChanged = null;
		IJavaElementDelta[] deltas = delta.getAffectedChildren();
		for (int i = 0; i < deltas.length; i++) {
			IJavaElementDelta childDelta = deltas[i];
			IJavaElement childElement = childDelta.getElement();
			IProxyNode lowestChild = null;
			switch(childDelta.getKind()) {
				case IJavaElementDelta.ADDED: {
					JavaElementProxyNode proxy = createChildProxy(childElement, childDelta);
					if (proxy != null) {
						addChildProxy(proxy);
						lowestChild = this;
					}
					break;
				} case IJavaElementDelta.REMOVED: {
					JavaElementProxyNode proxy = getChildProxy(childElement);
					if (proxy != null) {
						removeChildProxy(proxy);
						lowestChild = this;
					}
					break;
				} case IJavaElementDelta.CHANGED: {
					JavaElementProxyNode proxy = getChildProxy(childElement);
					if (proxy != null) {
						lowestChild = childChanged(childDelta, proxy);
					} else {
						proxy = createChildProxy(childElement, childDelta);
						if (proxy != null) {
							addChildProxy(proxy);
							lowestChild = this;
						}
					}
					break;
				}
			}
			if (lowestChild != null) {
				if (lowestChanged == null) {
					lowestChanged = lowestChild;
				} else {
					lowestChanged = this;
				}
			}
		}
		return lowestChanged;
	}

    protected final JavaElementProxyNode getChildProxy(IJavaElement element) {
        for (int i = 0; i < children.length; i++) {
            if (element.equals(children[i].getJavaElement())) {
                return children[i];
            }
        }
        return null;
    }

    protected final void replaceChildProxy(JavaElementProxyNode oldProxy, JavaElementProxyNode newProxy) {
        for (int i = 0; i < children.length; i++) {
            if(children[i] == oldProxy) {
                children[i] = newProxy;
                break;
            }
        }
    }
 
	protected final boolean addChildProxy(JavaElementProxyNode proxy) {
        if(proxy == null) return false;
        JavaElementProxyNode[] newChildren = new JavaElementProxyNode[children.length + 1];
        for (int i = 0; i < children.length; i++) {
            //- if the proxy to add is already a child do not add it
            if(children[i].getCorrespondingResource().equals(proxy.getCorrespondingResource())) {
            	if (!children[i].equals(proxy)) {
            		// same resource but newer version => replace it
            		children[i] = proxy;
            	}
                return false;
            }
            newChildren[i] = children[i];
        }
        newChildren[children.length] = proxy;
        //- change the children of the current proxy node
        children = newChildren;
        return true;
	}

    protected void removeChildProxy(JavaElementProxyNode proxy) {
    	JavaElementProxyNode[] newChildren = new JavaElementProxyNode[children.length - 1];
        int j = 0;
        for (int i = 0; i < children.length; i++) {
            if(children[i] != proxy) {
                newChildren[j++] = children[i];
            }
        }
        //- change the children of the current proxy node
        children = newChildren;
        disposeChildProxy(proxy);
    }

    protected IProxyNode childChanged(IJavaElementDelta delta, JavaElementProxyNode child) {
    	IProxyNode lowestChild = child.elementChanged(delta);
		if (lowestChild != null && child.getChildren().length == 0) {
			removeChildProxy(child);
			lowestChild = this;
		}
		return lowestChild;
    }
    
    public void dispose() {
    	for (int i = 0; i < children.length; i++) {
			disposeChildProxy(children[i]);
		}
    }
    
}
