/********************************************************************** 
 * Copyright (c) 2005, 2007 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: PackageProxyNode.java,v 1.8 2007/06/01 14:51:54 jgout Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/ 
package org.eclipse.hyades.test.tools.ui.java.internal.junit.navigator;

import java.util.LinkedList;
import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.hyades.test.tools.ui.ToolsUiPlugin;
import org.eclipse.hyades.test.ui.UiPlugin;
import org.eclipse.hyades.test.ui.internal.navigator.proxy.IExtendedProxyNode;
import org.eclipse.hyades.test.ui.navigator.IFileProxyManager;
import org.eclipse.hyades.test.ui.navigator.IProxyNode;
import org.eclipse.hyades.test.ui.navigator.IResourceChangeListenerProxyNode;
import org.eclipse.hyades.test.ui.navigator.ITestFolderContentValidator;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.ui.ISharedImages;
import org.eclipse.jdt.ui.JavaUI;
import org.eclipse.swt.graphics.Image;

/**
 * Proxy node from a package. This proxy contains all JUnit test found in the corresponding java package.
 * @author jgout
 * @since 4.0
 */
public class PackageProxyNode implements IResourceChangeListenerProxyNode, IExtendedProxyNode {
    
    private IPackageFragment fragment;
    private IFileProxyManager fileProxyManager;
    private Object parent;
    private IProxyNode[] children;
    private static final IProxyNode[] NO_CHILDREN = new IProxyNode[0];
    private ITestFolderContentValidator validator;
    
    public static PackageProxyNode create(IFolder folder, IResourceDelta delta, ITestFolderContentValidator validator, IFileProxyManager fileProxyManager, SourceFolderProxyNode node) {
        PackageProxyNode pack = new PackageProxyNode(folder, validator, fileProxyManager, node);
        pack.resourceChanged(delta);
        return pack.getChildren().length > 0 ? pack : null;
    }

    public static PackageProxyNode create(IPackageFragment fragment, ITestFolderContentValidator validator, IFileProxyManager fileProxyManager, Object parent) {
        PackageProxyNode proxy = new PackageProxyNode(fragment, validator, fileProxyManager, parent);
        if(proxy.getChildren().length > 0) {
            return proxy;
        } else return null;
    }
    
    private PackageProxyNode(IFolder folder, ITestFolderContentValidator validator, IFileProxyManager fileProxyManager, Object parent) {
        this.fileProxyManager = fileProxyManager;
        this.parent = parent;
        this.validator = validator;
        this.fragment = (IPackageFragment)JavaCore.create(folder);
        this.children = NO_CHILDREN;       
    }
    
    private PackageProxyNode(IPackageFragment fragment, ITestFolderContentValidator validator, IFileProxyManager fileProxyManager, Object parent) {
        this.fragment = fragment;
        this.fileProxyManager = fileProxyManager;
        this.parent = parent;
        this.validator = validator;
        List _children = new LinkedList();
        Object[] resources;
        try {
            resources = this.fragment.getNonJavaResources();
            for (int i = 0; i < resources.length; i++) {
                IResource res = (IResource) resources[i];
                if(res.getType() == IResource.FILE && validator.isFileOk((IFile)res)) {
                    IProxyNode proxy = fileProxyManager.getProxy((IFile)res, this);
                    if(proxy != null && validator.isProxyOk(proxy)) {
                        _children.add(proxy);
                        fileProxyManager.cacheProxy((IFile)res, proxy);
                    }
                } else if(res instanceof IContainer) {
                    //- this is basic folder under a package (there should be an exclude rule on the project)
                    IProxyNode proxy = JUnitFolderProxyNode.create((IContainer)res, validator, fileProxyManager, false, this);
                    if(proxy != null && validator.isProxyOk(proxy)) {
                        _children.add(proxy);
                    }
                }
            }
            children = (IProxyNode[]) _children.toArray(new IProxyNode[_children.size()]);
        } catch (JavaModelException e) {
            ToolsUiPlugin.logError(e);
            children = NO_CHILDREN;
        }
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getText()
     */
    public String getText() {
        return fragment.getElementName();
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getImage()
     */
    public Image getImage() {
        return JavaUI.getSharedImages().getImage(ISharedImages.IMG_OBJS_PACKAGE);
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getParent()
     */
    public Object getParent() {
        return parent;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.IProxyNode#getChildren()
     */
    public IProxyNode[] getChildren() {
        return children;
    }

    /* (non-Javadoc)
     * @see org.eclipse.core.runtime.IAdaptable#getAdapter(java.lang.Class)
     */
    public Object getAdapter(Class adapter) {
        if(adapter == IContainer.class) {
            try {
                return fragment.getUnderlyingResource();
            } catch (JavaModelException e) {
                ToolsUiPlugin.logError(e);
                return null;
            }
        }
        return null;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.IProxy#getUnderlyingResource()
     */
    public IResource getUnderlyingResource() {
        try {
            return fragment.getUnderlyingResource();
        } catch (JavaModelException e) {
            ToolsUiPlugin.logError(e);
            return null;
        }
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.test.ui.navigator.IProxy#getIdentifier()
     */
    public String getIdentifier() {
        return fragment.getElementName();
    }

    private boolean addChild(IProxyNode proxy) {
        if(proxy == null) return false;
        IProxyNode[] newChildren = new IProxyNode[children.length + 1];
        for (int i = 0; i < children.length; i++) {
            //- if the proxy to add is already a child do not add it
            if(children[i].equals(proxy)) {
                return false;
            }
            newChildren[i] = children[i];
        }
        newChildren[children.length] = proxy;
        //- change the children of the current proxy node
        children = newChildren;
        return true;
    }

    private void replaceChild(IProxyNode oldProxy, IProxyNode newProxy) {
        for (int i = 0; i < children.length; i++) {
            if(children[i] == oldProxy) {
                children[i] = newProxy;
                break;
            }
        }
    }
 
    private void removeChild(IProxyNode proxy) {
        IProxyNode[] newChildren = new IProxyNode[children.length - 1];
        int j = 0;
        for (int i = 0; i < children.length; i++) {
            if(children[i] != proxy) {
                newChildren[j++] = children[i];
            }
        }
        //- change the children of the current proxy node
        children = newChildren;
    }

    private IProxyNode getProxy(IResource resource) {
        for (int i = 0; i < children.length; i++) {
            if (resource.equals(children[i].getUnderlyingResource())) {
                return children[i];
            }
        }
        return null;
    }

    public IProxyNode resourceChanged(IResourceDelta rootDelta) {
        IProxyNode lowestChanged = null;
        IResourceDelta[] res = rootDelta.getAffectedChildren();
        for (int i = 0; i < res.length; i++) {
            int kind = res[i].getKind();
            IProxyNode lowestChild = null;
            if (kind == IResourceDelta.ADDED) {
                lowestChild = resourceChangedADDED(res[i]);
            } else if (kind == IResourceDelta.REMOVED) {
                lowestChild = resourceChangedREMOVED(res[i]);
            } else if (kind == IResourceDelta.CHANGED) {
                lowestChild = resourceChangedCHANGED(res[i]);
            }
            if(lowestChild != null) {
                //- the current child has reported a change in its content
                if(lowestChanged == null) {
                    //- not a child modified yet, so the lowest is this child
                    lowestChanged = lowestChild;
                } else {
                    //- this child is not the first child to change, so the lowest is the parent
                    lowestChanged = this;
                }
            }
        }
        return lowestChanged;
    }

    private IProxyNode resourceChangedCHANGED(IResourceDelta delta) {
        IResource resource = delta.getResource();
        //- find the matching proxy in the children array of the current proxy.
        IProxyNode proxy = getProxy(resource);
        if ((delta.getFlags() & IResourceDelta.CONTENT) != 0) {
            if(proxy != null) {
                //- Since proxy exists, we need to update it
                if(validator.isFileOk((IFile)resource)) {
                    fileProxyManager.uncacheProxy((IFile)resource);
                    IProxyNode newProxy = fileProxyManager.getProxy((IFile)resource, this);
                    if(validator.isProxyOk(newProxy)) {
                        replaceChild(proxy, newProxy);
                        fileProxyManager.cacheProxy((IFile)resource, newProxy);
                    } else {
                        removeChild(proxy);
                    }
                    return this;
                }
            } else {
                //- even if the proxy is not yet built, we need to check if it can be converted.
                if(validator.isFileOk((IFile)resource)) {
                    IProxyNode newProxy = fileProxyManager.getProxy((IFile)resource, this);
                    if(validator.isProxyOk(newProxy)) {
                        boolean wasAdded = addChild(newProxy);
                        if(wasAdded) {
                            fileProxyManager.cacheProxy((IFile)resource, newProxy);
                            return this;
                        }
                    }
                }
                return null;
            }
        }
        return null;
    }

    private IProxyNode resourceChangedADDED(IResourceDelta delta) {
        IResource resource = delta.getResource();
        int resType = resource.getType();
        //- a resource (folder or file) has been added to the file system
        if (resType == IResource.FILE) {
            //- new file check if this file should kept
            if (validator.isFileOk((IFile) resource)) {
                IProxyNode newProxy = fileProxyManager.getProxy((IFile)resource, this);
                //- check if new proxy should be added as a child of current
                if (validator.isProxyOk(newProxy)) {
                    boolean wasAdded = addChild(newProxy);
                    if(wasAdded) {
                        fileProxyManager.cacheProxy((IFile)resource, newProxy);
                        return this;
                    }
                }
            }
        } else if (resType == IResource.FOLDER) {
            IFolder folder = (IFolder)resource;
            IPackageFragment frag = (IPackageFragment) JavaCore.create(folder);
            if(frag != null) {
                IProxyNode proxy = PackageProxyNode.create(frag, validator, fileProxyManager, this);
                if(proxy != null) {
                    addChild(proxy);
                    return this;
                }
            }
        }
        return null;
    }

    private IProxyNode resourceChangedREMOVED(IResourceDelta delta) {
        IResource resource = delta.getResource();
        int resType = resource.getType();
        IProxyNode proxy2Remove = getProxy(resource);
        //- a resource (folder or file) has been removed from the file system
        if (resType == IResource.FILE) {
            if(proxy2Remove != null) {
                //- the proxy to remove is one of the current node children, we need to remove it form the cache
                fileProxyManager.uncacheProxy((IFile)resource);
                //- and remove it from the children list of the current folder
                removeChild(proxy2Remove);
                return this;
            } else {
                //- proxy not found among children so this file removal has no impacts in this current folder
                return null;
            }
        } else if (resType == IResource.FOLDER) {
            if(proxy2Remove != null) {
                //- recursive call to remove children
                if (proxy2Remove instanceof IResourceChangeListenerProxyNode) {
                    ((IResourceChangeListenerProxyNode)proxy2Remove).resourceChanged(delta);
                    removeChild(proxy2Remove);
                    return this;
                } else {
                    //- this should never arrive, because the proxy of a folder
                    //- should be a IResourceChangeListenerProxyNode
                    UiPlugin.logError("The proxy node derived from a folder should be a IResourceChangeListenerProxyNode"); //$NON-NLS-1$
                }
            }
        }
        return null;
    }

	public IResource getCorrespondingResource() {
		return getUnderlyingResource();
	}

}
